package com.atlassian.plugins.custom_apps;

import com.atlassian.plugins.custom_apps.api.CustomApp;
import com.atlassian.sal.api.pluginsettings.PluginSettings;
import com.atlassian.sal.api.pluginsettings.PluginSettingsFactory;
import com.google.common.collect.ImmutableList;
import org.apache.log4j.Logger;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.util.ArrayList;
import java.util.List;

/**
 * Implementation of @{link CustomAppStore} based on {@link com.atlassian.sal.api.pluginsettings.PluginSettings}
 *
 * @since v3.1.15
 */
public class PluginSettingsBasedCustomAppStore implements CustomAppStore
{
    private static final Logger log = Logger.getLogger(PluginSettingsBasedCustomAppStore.class);
    private static final String KEY = "com.atlassian.plugins.custom_apps.";
    private static final String CUSTOM_APPS_AS_JSON = "customAppsAsJSON";
    private static final String HAS_CUSTOM_ORDER = "hasCustomOrder";

    private final PluginSettingsFactory pluginSettingsFactory;

    public PluginSettingsBasedCustomAppStore(PluginSettingsFactory pluginSettingsFactory)
    {
        this.pluginSettingsFactory = pluginSettingsFactory;
    }

    /**
     * As the migration check may call methods which need a list of the custom apps, we need a way of getting that list
     * without recursively calling checkMigration
     */
    @Override
    public List<CustomApp> getAll()
    {
        return jsonStringToList((String) settings().get(CUSTOM_APPS_AS_JSON));
    }

    @Override
    public void storeAll(final List<CustomApp> apps)
    {
        settings().put(CUSTOM_APPS_AS_JSON, listToJsonString(apps));
    }

    @Override
    public boolean isCustomOrder()
    {
        return "true".equals(settings().get(HAS_CUSTOM_ORDER));
    }

    @Override
    public void setCustomOrder()
    {
        settings().put(HAS_CUSTOM_ORDER, "true");
    }

    private PluginSettings settings()
    {
        return new PluginSettings()
        {
            private PluginSettings pluginSettings = pluginSettingsFactory.createGlobalSettings();

            @Override
            public Object get(String key)
            {
                return pluginSettings.get(KEY + key);
            }

            @Override
            public Object put(String key, Object value)
            {
                return pluginSettings.put(KEY + key, value);
            }

            @Override
            public Object remove(String key)
            {
                return pluginSettings.remove(KEY + key);
            }
        };
    }

    private List<CustomApp> jsonStringToList(String jsonString)
    {
        List<CustomApp> customApps = new ArrayList<CustomApp>();
        if (jsonString != null)
        {
            try
            {
                JSONArray json = new JSONArray(jsonString);
                for (int i = 0; i < json.length(); ++i)
                {
                    CustomApp c = jsonObjectToCustomApp(json.getJSONObject(i));
                    if (c != null)
                    {
                        customApps.add(c);
                    }
                }
            }
            catch (JSONException e)
            {
                log.error("Error decoding custom apps JSON representation: '" + jsonString + "'.", e);
            }
        }
        return customApps;
    }

    private CustomApp jsonObjectToCustomApp(JSONObject o) throws JSONException
    {
        String id = o.getString(ID);
        String displayName = o.getString(DISPLAY_NAME);
        String url = o.getString(URL);

        // sourceApplicationUrl will not exist the first time we use this version of the plugin, as this field has been added,
        // which is OK as this field should be null in local custom apps. Same applies to all the fields below.
        String sourceApplicationUrl = getString(o, BASE_URL);
        String sourceApplicationName = getString(o, APPLICATION_NAME);
        String sourceApplicationType = getString(o, APPLICATION_TYPE);
        boolean hide = getBoolean(o, HIDE);
        boolean editable = getBoolean(o, EDITABLE);
        List<String> allowedGroups = getStringArray(o, ALLOWED_GROUPS);
        boolean self = getBoolean(o, SELF);

        if (id != null && displayName != null && url != null)
        {
            return new CustomApp(id, displayName, url, sourceApplicationUrl, sourceApplicationName, sourceApplicationType, hide, allowedGroups, editable, self);
        }
        return null;
    }

    private String getString(JSONObject o, String property) throws JSONException
    {
        if (o.has(property))
        {
            return o.getString(property);
        }
        return null;
    }

    private boolean getBoolean(JSONObject o, String property) throws JSONException
    {
        if (o.has(property))
        {
            return o.getBoolean(property);
        }
        return false;
    }

    private List<String> getStringArray(JSONObject o, String property) throws JSONException
    {
        final ImmutableList.Builder<String> builder = ImmutableList.builder();
        if (o.has(property))
        {
            JSONArray a = o.getJSONArray(property);
            for (int i = 0; i < a.length(); ++i)
            {
                builder.add(a.getString(i));
            }
        }
        return builder.build();
    }

    private String listToJsonString(List<CustomApp> apps)
    {
        if (apps == null)
        {
            return null;
        }

        JSONArray json = new JSONArray();
        for (CustomApp app : apps)
        {
            JSONObject o = new JSONObject();
            try
            {
                o.put(ID, app.getId());
                o.put(DISPLAY_NAME, app.getDisplayName());
                o.put(URL, app.getUrl());
                o.put(BASE_URL, app.getSourceApplicationUrl());
                o.put(APPLICATION_NAME, app.getSourceApplicationName());
                o.put(APPLICATION_TYPE, app.getSourceApplicationType());
                o.put(HIDE, app.getHide());
                o.put(ALLOWED_GROUPS, app.getAllowedGroups());
                o.put(EDITABLE, app.getEditable());
                o.put(SELF, app.isSelf());
                json.put(o);
            }
            catch (JSONException e)
            {
                log.error("Error encoding custom app " + app, e);
            }
        }
        return json.toString();
    }
}
