package com.atlassian.plugins.custom_apps;

import com.atlassian.plugins.custom_apps.api.CustomApp;
import com.atlassian.plugins.custom_apps.api.CustomAppService;
import com.atlassian.plugins.navlink.consumer.menu.services.NavigationLinkComparator;
import com.atlassian.plugins.navlink.producer.capabilities.services.ApplicationTypeService;
import com.atlassian.plugins.navlink.producer.navigation.links.LinkSource;
import com.atlassian.plugins.navlink.producer.navigation.services.RawNavigationLink;
import com.atlassian.plugins.navlink.producer.navigation.services.RawNavigationLinkBuilder;
import com.atlassian.plugins.navlink.producer.navigation.services.NavigationLinkRepository;
import com.google.common.base.Function;
import com.google.common.base.Predicate;
import com.google.common.collect.Iterables;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.util.List;

import static com.google.common.base.Preconditions.checkNotNull;
import static com.google.common.base.Predicates.notNull;


public class NavigationLinkRepositoryAdapter implements NavigationLinkRepository
{
    private static final String NAVIGATION_LINK_KEY = "custom-apps";

    private final CustomAppService customAppService;
    private final ApplicationTypeService applicationTypeService;

    public NavigationLinkRepositoryAdapter(@Nonnull CustomAppService customAppService,
                                           @Nonnull ApplicationTypeService applicationTypeService)
    {
        this.customAppService = checkNotNull(customAppService);
        this.applicationTypeService = checkNotNull(applicationTypeService);
    }

    @Nonnull
    @Override
    public Iterable<RawNavigationLink> all()
    {
        return matching(null);
    }

    @Nonnull
    @Override
    public Iterable<RawNavigationLink> matching(@Nullable final Predicate<RawNavigationLink> predicate)
    {
        final List<CustomApp> customAppList = customAppService.getCustomApps();
        final Iterable<RawNavigationLink> navigationLinks = transformCustomAppsToNavigationLinks(customAppList);
        final Iterable<RawNavigationLink> filteredByPredicate = filterByPredicate(predicate, navigationLinks);
        final Iterable<RawNavigationLink> result = filterNulls(filteredByPredicate);
        return Iterables.unmodifiableIterable(result);
    }

    @Nonnull
    private Iterable<RawNavigationLink> transformCustomAppsToNavigationLinks(@Nonnull final Iterable<CustomApp> customAppList)
    {
        return Iterables.transform(customAppList, new Function<CustomApp, RawNavigationLink>()
        {
            @Override
            public RawNavigationLink apply(@Nullable final CustomApp customApp)
            {
                return customApp != null ? createLocalNavigationLink(customApp) : null;
            }
        });
    }

    @Nonnull
    private Iterable<RawNavigationLink> filterByPredicate(@Nullable final Predicate<RawNavigationLink> predicate, @Nonnull final Iterable<RawNavigationLink> navigationLinks)
    {
        return predicate != null ? Iterables.filter(navigationLinks, predicate) : navigationLinks;
    }

    @Nonnull
    private Iterable<RawNavigationLink> filterNulls(@Nonnull final Iterable<RawNavigationLink> filtered)
    {
        return Iterables.filter(filtered, notNull());
    }

    @Nonnull
    private RawNavigationLink createLocalNavigationLink(@Nonnull final CustomApp customApp)
    {
        return new RawNavigationLinkBuilder()
                .key(NAVIGATION_LINK_KEY)
                .labelKey(customApp.getDisplayName())
                .href(customApp.getUrl())
                .source(LinkSource.localDefault())
                .applicationType(applicationTypeService.get())
                .weight(NavigationLinkComparator.Weights.MAX.value())
                .build();
    }


}
