package com.atlassian.paralyzer.core.reporting;

import com.atlassian.paralyzer.api.Extension;
import com.atlassian.paralyzer.api.TestResult;
import com.atlassian.paralyzer.api.engine.AfterEach;
import com.atlassian.paralyzer.api.engine.BeforeEach;
import com.atlassian.paralyzer.api.engine.TestDetails;
import lombok.var;

import java.time.Instant;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.ConcurrentHashMap;
import java.util.function.Predicate;

/**
 * @since 1.0
 */
public class TestExecutionTimeListener implements BeforeEach, AfterEach {
    private final String extensionType = "DetailedInfo";
    private final String propertyName = "executionTime";
    private final Map<String, Instant> startTimes = new ConcurrentHashMap<>();

    @Override
    public Predicate<String> getSupportedEnginePredicate() {
        return b -> true;
    }

    @Override
    public void afterEach(TestDetails details) {
        Instant now = Instant.now();
        Instant startTime = startTimes.get(getTestId(details.getTestResult()));
        TestResult result = details.getTestResult();
        if (startTime != null) {
            startTimes.remove(getTestId(result));
            Double executionTime = (now.toEpochMilli() - startTime.toEpochMilli()) / 1000.0;
            Optional<Extension> infoExtension = getExtension(result);
            infoExtension.get().setProperty(propertyName, executionTime);
        }
    }

    private Optional<Extension> getExtension(TestResult result) {
        var infoExtension = result.getExtensions().stream()
                .filter(extension -> extension.getExtensionType().equals(extensionType))
                .findAny();

        if (!infoExtension.isPresent()) {
            infoExtension = Optional.of(new Extension(extensionType));
            result.addExtension(infoExtension.get());
        }
        return infoExtension;
    }

    @Override
    public void beforeEach(TestDetails details) {
        startTimes.put(getTestId(details.getTestResult()), Instant.now());
    }

    private String getTestId(TestResult testResult) {
        return testResult.getTestEngineId() + "_" + testResult.getUniqueId();
    }
}
