package com.atlassian.paralyzer.core.reporting;

import com.atlassian.paralyzer.api.TestResult;
import com.atlassian.paralyzer.api.TestResultCollector;
import com.atlassian.paralyzer.api.TestSuite;
import com.google.inject.Inject;
import com.google.inject.name.Named;

import java.io.File;
import java.net.URLEncoder;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

/**
 * Result collector saving test result to xml files using Surefire format
 *
 * @since 1.0
 */
public class SurefireReportCollector implements TestResultCollector {
    private final File reportsDirectory;

    @Inject
    public SurefireReportCollector(@Named("reportsDirectory") File reportsDirectory) {
        if (!reportsDirectory.isDirectory() && reportsDirectory.exists()) {
            throw new IllegalArgumentException("reportsDirectory must be directory");
        }
        this.reportsDirectory = reportsDirectory;
    }

    @Override
    public void collectResults(List<TestResult> results) {
        reportsDirectory.mkdirs();
        results.stream()
                .collect(Collectors.groupingBy(TestResult::getTestSuite))
                .forEach((key, value) -> {
                    File reportFile = new File(reportsDirectory, covertSuiteNameToFilename(getPrettyName(key)));
                    if (reportFile.exists()) {
                        reportFile = new File(reportsDirectory, covertSuiteNameToFilename(key.getTestEngineId()));
                    }
                    new SurefireReport(reportFile, value, key).writeReport();
                });
    }

    private String covertSuiteNameToFilename(String key) {
        return "TEST-" + URLEncoder.encode(key) + ".xml";
    }

    private String getPrettyName(TestSuite key) {
        return key.getExtensions().stream()
                .filter(extension -> extension.getExtensionType().equals("PrettyName"))
                .map(extension -> (String) extension.getProperty("name"))
                .filter(Objects::nonNull)
                .findAny()
                .orElse(key.getUniqueId());
    }
}
