package com.atlassian.paralyzer.core;

import com.atlassian.paralyzer.api.NoRunnerAvailableException;
import com.atlassian.paralyzer.api.Paralyzer;
import com.atlassian.paralyzer.api.PluginModule;
import com.atlassian.paralyzer.api.Runner;
import com.atlassian.paralyzer.api.RunnerConnector;
import com.atlassian.paralyzer.api.RunnerExecutionStrategy;
import com.atlassian.paralyzer.api.Settings;
import com.atlassian.paralyzer.api.TestDiscoveryRequest;
import com.atlassian.paralyzer.api.TestEngine;
import com.atlassian.paralyzer.api.TestResult;
import com.atlassian.paralyzer.api.TestResultCollector;
import com.atlassian.paralyzer.api.TestResultListener;
import com.atlassian.paralyzer.api.TestSuiteProcessorChain;
import com.atlassian.paralyzer.api.dependency.management.DependencyContainer;
import com.atlassian.paralyzer.api.engine.AfterAll;
import com.atlassian.paralyzer.api.engine.AfterEach;
import com.atlassian.paralyzer.api.engine.AfterSuite;
import com.atlassian.paralyzer.api.engine.BeforeAll;
import com.atlassian.paralyzer.api.engine.BeforeEach;
import com.atlassian.paralyzer.api.engine.BeforeSuite;
import com.atlassian.paralyzer.api.engine.TestEngineListener;
import com.google.inject.ConfigurationException;
import lombok.extern.slf4j.Slf4j;
import lombok.var;

import java.util.ArrayList;
import java.util.List;
import java.util.Set;
import java.util.concurrent.Semaphore;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;


@Slf4j
public class ParalyzerCore implements Paralyzer {
    protected final DependencyContainer dependencyContainer = new DependencyContainer();
    protected final List<PluginModule> plugins = new ArrayList<>();
    protected Settings settings = new Settings();
    protected Set<TestEngine> testEngines;
    protected TestSuiteProcessorChain processorChain;
    protected RunnerExecutionStrategy executionStrategy;
    protected RunnerConnector runnerConnector;

    public ParalyzerCore() {
        // Guice is not throwing exception when nothing is registered for these classes
        dependencyContainer.addMultiInstanceType(TestEngine.class);
        dependencyContainer.addMultiInstanceType(TestResultCollector.class);
        dependencyContainer.addMultiInstanceType(TestResultListener.class);
        dependencyContainer.addMultiInstanceType(TestEngineListener.class);
        dependencyContainer.addMultiInstanceType(BeforeAll.class);
        dependencyContainer.addMultiInstanceType(AfterAll.class);
        dependencyContainer.addMultiInstanceType(AfterSuite.class);
        dependencyContainer.addMultiInstanceType(BeforeSuite.class);
        dependencyContainer.addMultiInstanceType(AfterEach.class);
        dependencyContainer.addMultiInstanceType(BeforeEach.class);

        dependencyContainer.addSingleInstanceType(TestSuiteProcessorChain.class);
        dependencyContainer.addSingleInstanceType(RunnerExecutionStrategy.class);
        dependencyContainer.addSingleInstanceType(RunnerConnector.class);

        dependencyContainer.addObject(new DefaultTestSuiteProcessorChain());
    }

    @Override
    public void addPluginObject(Object object) {
        dependencyContainer.addObject(object);
    }

    @Override
    public void addPluginModule(PluginModule module) {
        log.info("Registered new plugin into Paralyzer: {}", module);
        dependencyContainer.addModule(module);
        plugins.add(module);
    }

    @Override
    public Settings getSettings() {
        return settings;
    }

    @Override
    public void setSettings(Settings settings) {
        this.settings = settings;
    }

    @Override
    public void execute(TestDiscoveryRequest request) throws NoRunnerAvailableException {
        setUpPlugins();
        log.info("Init dependency container");
        dependencyContainer.init();
        log.info("Execution started");
        validateSettings(request);
        log.info("Settings validation successful");
        prepare();
        log.info("Prepare successful");

        try {
            Thread thread = new Thread(() -> {
                log.info("Test execution started");
                executeTests(request);
                log.info("Test execution successful");
                tearDown();
                log.info("Tear down successful");
                log.info("Still active threads: " +
                        Thread.activeCount());
                Thread.currentThread().getThreadGroup().interrupt();
            });
            thread.start();
            thread.join();
        } catch (InterruptedException e) {
            log.info("Still active threads: " +
                    Thread.activeCount());
        }
    }

    private void setUpPlugins() {
        plugins.forEach(pluginModule -> {
            log.debug("Set up plugin: {}", pluginModule);
            pluginModule.setUp(settings);
        });
    }

    protected void tearDown() {
        plugins.forEach(pluginModule -> {
            log.debug("Tear down plugin: {}", pluginModule);
            pluginModule.tearDown();
        });
    }

    protected void executeTests(TestDiscoveryRequest request) {
        List<TestResult> testResults = runTests(request);
        runnerConnector.tearDown();
        collectResults(testResults);
    }

    private void collectResults(List<TestResult> testResults) {
        log.info("Collecting results");
        var testResultCollector = dependencyContainer.getSetOfInstances(TestResultCollector.class);
        if (testResultCollector != null) {
            testResultCollector
                    .forEach(collector -> {
                        log.debug("Collector: {}", collector);
                        collector.collectResults(testResults);
                    });
        }
    }

    private List<TestResult> runTests(TestDiscoveryRequest request) {
        testEngines.forEach(testEngine -> {
            log.debug("Executing TestEngine: {}", testEngine);
            var testSuites = testEngine.discoverTests(request);
            var testSuitesStream = processorChain.process(testSuites.stream(), testEngine.getUniqueId());
            executionStrategy.addTests(testSuitesStream.collect(Collectors.toList()));
        });
        return executionStrategy.executeTests();
    }

    protected void validateSettings(TestDiscoveryRequest request) {
        if (request == null) {
            throw new NullPointerException();
        }
        try {
            testEngines = dependencyContainer.getSetOfInstances(TestEngine.class);
            processorChain = dependencyContainer.getInstance(TestSuiteProcessorChain.class);
            executionStrategy = dependencyContainer.getInstance(RunnerExecutionStrategy.class);
            runnerConnector = dependencyContainer.getInstance(RunnerConnector.class);
        } catch (ConfigurationException exception) {
            throw new IllegalStateException(exception);
        }
        if (testEngines.isEmpty()
                || processorChain == null
                || executionStrategy == null
                || runnerConnector == null) {
            throw new IllegalStateException();
        }
    }

    protected void prepare() throws NoRunnerAvailableException {
        prepareRunners();
        prepareTestEngines();
        preparePlugins();
    }

    protected void preparePlugins() {
        plugins.forEach(pluginModule -> {
            log.debug("Set up processor chain by plugin: {}", pluginModule);
            pluginModule.setUpProcessorsChain(processorChain);
        });
    }

    protected void prepareTestEngines() {
        var testEngineListeners = dependencyContainer.getSetOfInstances(TestEngineListener.class);
        testEngines.forEach(testEngine -> {
            testEngineListeners.stream()
                    .filter(listener -> listener.getSupportedEnginePredicate().test(testEngine.getUniqueId()))
                    .forEach(testEngine::addListener);
        });
    }

    protected void prepareRunners() throws NoRunnerAvailableException {
        var testResultListeners = dependencyContainer.getSetOfInstances(TestResultListener.class);
        List<Runner> availableRunners = new ArrayList<>(runnerConnector.getCurrentRunners());
        runnerConnector.addNewRunnerCallback(runner -> prepareTestRunner(testResultListeners, runner));
        waitForRunnersIfNeeded(availableRunners);
        availableRunners.forEach(runner -> prepareTestRunner(testResultListeners, runner));
    }

    protected void waitForRunnersIfNeeded(List<Runner> availableRunners) throws NoRunnerAvailableException {
        if (availableRunners.isEmpty()) {
            log.info("Waiting for runners");
            if (runnerConnector.isAcceptingNewRunners()) {
                waitForRunner();
            } else {
                log.error("No runners detected");
                throw new NoRunnerAvailableException();
            }
        }
    }

    protected void prepareTestRunner(Set<TestResultListener> testResultListeners, Runner runner) {
        testResultListeners.forEach(runner::addTestResultListener);
        executionStrategy.addTestRunner(runner);
    }

    protected void waitForRunner() throws NoRunnerAvailableException {
        Semaphore lock = new Semaphore(1);
        try {
            lock.acquire();
            runnerConnector.addNewRunnerCallback(runner -> {
                lock.release();
            });

            if (!lock.tryAcquire(settings.getWaitingForRunnersTimeoutMs(), TimeUnit.MILLISECONDS)) {
                log.error("Waiting for runners timeout");
                throw new NoRunnerAvailableException();
            }

        } catch (InterruptedException e) {
            log.error("Exception during waiting for runners", e);
            throw new NoRunnerAvailableException();
        }
    }
}
