package com.atlassian.oauth2.provider.api.authorization;

import com.atlassian.oauth2.provider.api.pkce.CodeChallengeMethod;
import com.atlassian.oauth2.scopes.api.Scope;

import javax.annotation.Nonnull;
import java.time.Duration;
import java.util.Optional;

/**
 * Service for handling 3LO authentication
 */
public interface AuthorizationService {

    /**
     * Creates an authorization request
     *
     * @param clientId            used for this authorization request
     * @param redirectUri         the redirect uri verified when getting a token
     * @param scope               scope used in authorization flow
     * @param codeChallengeMethod determines how the code verifier should be validated during PKCE. Passing `null` will disable PKCE for the flow.
     * @param codeChallenge       the code challenge produced by transforming the legitimate code verifier (based on the code challenge method). Can be null if PKCE disabled.
     * @return the code for the session
     */
    String startAuthorizationFlow(@Nonnull String clientId,
                                  @Nonnull String redirectUri,
                                  @Nonnull Scope scope,
                                  CodeChallengeMethod codeChallengeMethod,
                                  String codeChallenge);

    /**
     * Completes authorization flow and returns {@link Authorization} if successful
     *
     * @param clientId    verify same client id used
     * @param redirectUri verify the same redirect uri as at the start of the flow
     * @param code        verify the same code as generated at the start of the flow
     * @return {@link AuthorizationFlowResult} verification result
     */
    AuthorizationFlowResult completeAuthorizationFlow(@Nonnull String clientId,
                                                      @Nonnull String redirectUri,
                                                      @Nonnull String code);

    /**
     * fetches the Authorization object for the given code
     *
     * @param authorizationCode authorization code to lookup
     * @return Authorization object matching the provided code
     */
    Optional<Authorization> getAuthorization(@Nonnull String authorizationCode);

    /**
     * Checks if PKCE is enabled for the given authorization request.
     * Mandatory security checks will be performed when enabled.
     *
     * @param authorizationCode the code provided by the authorization server
     * @return true if PKCE enabled at the time of the initial authorization request
     */
    boolean isPkceEnabledForAuthorization(@Nonnull String authorizationCode);

    /**
     * Checks if the provided code verifier produces the expected code challenge for the authorization associated
     * with the provided authorization code.
     * Used in the context of PKCE.
     *
     * @param codeVerifier the code verifier to check
     * @param authorizationCode the authorization code
     * @return true if the expected code challenge is produced
     */
    boolean isPkceCodeVerifierValidAgainstAuthorization(@Nonnull String codeVerifier, @Nonnull String authorizationCode);

    /**
     * Remove expired authorizations after a expiration period
     * @param expirationPeriod the period of time after which we remove authorisations
     */
    void removeExpiredAuthorizations(@Nonnull Duration expirationPeriod);
}
