package com.atlassian.oauth2.client.api.storage.config;

import com.atlassian.oauth2.client.api.storage.token.exception.ConfigurationNotFoundException;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.util.List;
import java.util.Optional;

public interface ClientConfigStorageService {

    /**
     * Creates or updates the {@link ClientConfigurationEntity}
     * @param clientConfigurationEntity the entity to create
     * @return The saved entity, with the ID if created
     * @throws ConfigurationNotFoundException if the configuration for the given id wasn't found
     */
    @Nonnull
    ClientConfigurationEntity save(@Nonnull ClientConfigurationEntity clientConfigurationEntity) throws ConfigurationNotFoundException;

    /**
     * @param id of {@link ClientConfigurationEntity} to delete
     * @throws ConfigurationNotFoundException if the configuration for the given id wasn't found
     */
    void delete(@Nonnull String id) throws ConfigurationNotFoundException;

    /**
     * @param id of the {@link ClientConfigurationEntity} from the store
     * @return The {@link Optional} {@link ClientConfigurationEntity} from the store for the id, or {@link Optional#empty}
     */
    @Nonnull
    Optional<ClientConfigurationEntity> getById(@Nonnull String id);

    /**
     * @param id of the {@link ClientConfigurationEntity} from the store
     * @return The {@link Optional} {@link ClientConfigurationEntity} from the store for the id
     * @throws ConfigurationNotFoundException if the configuration for the given id wasn't found
     */
    @Nonnull
    ClientConfigurationEntity getByIdOrFail(@Nonnull final String id) throws ConfigurationNotFoundException;

    /**
     * @param name of the {@link ClientConfigurationEntity} from the store
     * @return The {@link Optional} {@link ClientConfigurationEntity} from the store for the name, or {@link Optional#empty}
     */
    @Nonnull
    Optional<ClientConfigurationEntity> getByName(@Nonnull String name);

    /**
     * @return All the saved {@link ClientConfigurationEntity} ordered by name ascending, or an empty list
     */
    @Nonnull
    List<ClientConfigurationEntity> list();

    /**
     * Returns whether given name is unique.
     * @param id id of the persisted configuration or null, if it's a new one
     * @param name name to be checked for uniqueness
     * @return whether given name is unique
     */
    boolean isNameUnique(@Nullable String id, @Nonnull String name);
}
