package com.atlassian.oauth2.client.api.lib.token;

import com.atlassian.oauth2.client.api.ClientConfiguration;
import com.atlassian.oauth2.client.api.ClientToken;

import java.time.Duration;

/**
 * Service for obtaining and refreshing {@link ClientToken}.
 */
public interface TokenService {
    /**
     * Refreshes provided token.
     * @param config OAuth client config for the given token
     * @param token original token
     * @return refreshed token
     * @throws TokenServiceException if there was an error while calling token endpoint
     * @throws IllegalArgumentException if {@code token} has no refresh token
     */
    ClientToken forceRefresh(ClientConfiguration config, ClientToken token) throws TokenServiceException, IllegalArgumentException;

    /**
     * Refreshes provided token if needed. For tokens without refresh token original token will be returned.
     * @param config OAuth client config for the given token
     * @param token original token
     * @param margin max lifetime of the token that should be refreshed
     * @return refreshed token or original token (if refresh wasn't needed)
     */
    ClientToken refreshIfNeeded(ClientConfiguration config, ClientToken token, Duration margin) throws TokenServiceException;

    /**
     * Provides check to see if refresh is needed. For {@code token} with no refresh token this method will always return
     * {@code false}.
     * @param token original token
     * @param margin max lifetime of the token that should be refreshed
     * @return whether a refresh is required
     */
    boolean isRefreshNeeded(ClientToken token, Duration margin);
}
