package com.atlassian.oauth2.client.api.lib.flow;

import com.atlassian.annotations.PublicApi;
import com.atlassian.oauth2.client.api.ClientConfiguration;
import com.atlassian.oauth2.client.api.ClientToken;

import javax.annotation.Nonnull;
import javax.servlet.http.HttpSession;
import java.util.function.Function;

/**
 * Enables clients to start flow requests and to get their results.
 */
@PublicApi
public interface FlowRequestService {
    /**
     * Creates the flow request
     *
     * @param session             session, in which the data will be stored
     * @param clientConfiguration endpoints to hit and required credentials
     * @param clientRedirectUrl   function that, based on id, will produce url that the flow will be redirected to
     *                            after obtaining the token;
     * @return The {@link FlowRequest} contains the redirect url and id
     * @throws IllegalArgumentException thrown if the provided arguments are invalid, e.g. {@code clientConfiguration}
     *                                  endpoints are not https
     * @throws IllegalStateException    thrown if the session is invalidated, or base URL is not https
     *                                  (TODO: update this doc depending on the decision from go/j/KRAK-3359)
     */
    @Nonnull
    FlowRequest createFlowRequest(@Nonnull final HttpSession session,
                                  @Nonnull final ClientConfiguration clientConfiguration,
                                  @Nonnull final Function<String, String> clientRedirectUrl)
            throws IllegalArgumentException, IllegalStateException;

    /**
     * Returns stored flow result. This method should be called only once for the given {@code flowRequestId} -
     * subsequent calls will fail.
     *
     * @param session session to fetch the result from; has to be the same session
     *                as provided to {{@link #createFlowRequest(HttpSession, ClientConfiguration, Function)}}
     * @param flowRequestId the id of the {@link FlowRequest} from {{@link #createFlowRequest(HttpSession, ClientConfiguration, Function)}}
     * @return an {@link FlowResult} containing either {@link ClientToken} or {@link FlowRequestError} if error occurred
     * @throws IllegalArgumentException if there is no result for corresponding {@code flowRequestId}
     */
    @Nonnull
    FlowResult getFlowResult(@Nonnull final HttpSession session, @Nonnull final String flowRequestId) throws IllegalArgumentException;
}
