package com.atlassian.oauth2.client.api.storage.token;

import com.atlassian.oauth2.client.api.ClientToken;
import com.atlassian.oauth2.client.api.ClientTokenMetadata;
import com.google.common.base.MoreObjects;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.time.Instant;
import java.util.Objects;

public class ClientTokenEntity implements ClientToken, ClientTokenMetadata {
    public static final Instant MAX_TIMESTAMP = Instant.ofEpochMilli(Long.MAX_VALUE);

    private final String id;
    private final String configId;
    private final String accessToken;
    private final Instant accessTokenExpiration;
    private final String refreshToken;
    private final Instant refreshTokenExpiration;
    private final ClientTokenStatus status;
    private final Instant lastRefreshed;
    private final int refreshCount;
    private final Instant lastStatusUpdated;

    private ClientTokenEntity(@Nullable final String id,
                              @Nonnull final String configId,
                              @Nonnull final String accessToken,
                              @Nonnull final Instant accessTokenExpiration,
                              @Nullable final String refreshToken,
                              @Nullable final Instant refreshTokenExpiration,
                              @Nonnull final ClientTokenStatus status,
                              @Nullable final Instant lastRefreshed,
                              final int refreshCount,
                              @Nonnull Instant lastStatusUpdated) {
        this.id = id;
        this.configId = Objects.requireNonNull(configId, "Config ID cannot be null");
        this.accessToken = Objects.requireNonNull(accessToken, "Access token cannot be null");
        this.accessTokenExpiration = Objects.requireNonNull(accessTokenExpiration, "Expiration time of the access token cannot be null");
        this.refreshToken = maybeRequireNonNull(refreshToken, refreshTokenExpiration != null,
                "Refresh token cannot be null if it's expiration time is not null");
        this.refreshTokenExpiration = maybeRequireNonNull(refreshTokenExpiration, refreshToken != null,
                "Expiration time of the non-null refresh token cannot be null");
        this.status = Objects.requireNonNull(status, "Token status cannot be null");
        this.lastRefreshed = lastRefreshed;
        this.refreshCount = refreshCount;
        this.lastStatusUpdated = Objects.requireNonNull(lastStatusUpdated, "Last status updated cannot be null");
    }

    @Nullable
    public String getId() {
        return id;
    }

    @Nonnull
    public String getConfigId() {
        return configId;
    }

    @Override
    @Nonnull
    public String getAccessToken() {
        return accessToken;
    }

    @Override
    @Nonnull
    public Instant getAccessTokenExpiration() {
        return accessTokenExpiration;
    }

    @Nullable
    @Override
    public String getRefreshToken() {
        return refreshToken;
    }

    @Nullable
    @Override
    public Instant getRefreshTokenExpiration() {
        return refreshTokenExpiration;
    }

    @Override
    @Nonnull
    public ClientTokenStatus getStatus() {
        return status;
    }

    @Override
    @Nullable
    public Instant getLastRefreshed() {
        return lastRefreshed;
    }

    @Override
    public int getRefreshCount() {
        return refreshCount;
    }

    @Override
    @Nonnull
    public Instant getLastStatusUpdated() {
        return lastStatusUpdated;
    }

    @Nonnull
    public Builder toBuilder() {
        return builder(this);
    }

    @Nonnull
    public static Builder builder() {
        return new Builder();
    }

    @Nonnull
    public static Builder builder(final ClientToken data) {
        return new Builder(data);
    }

    @Nonnull
    public static Builder builder(final ClientTokenMetadata data) {
        return new Builder(data);
    }

    @Nonnull
    public static Builder builder(final ClientTokenEntity data) {
        return new Builder(data);
    }

    @Override
    public boolean equals(final Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }

        final ClientTokenEntity that = (ClientTokenEntity) o;

        return Objects.equals(this.getId(), that.getId()) &&
                Objects.equals(this.getConfigId(), that.getConfigId()) &&
                Objects.equals(this.getAccessToken(), that.getAccessToken()) &&
                Objects.equals(this.getAccessTokenExpiration(), that.getAccessTokenExpiration()) &&
                Objects.equals(this.getRefreshToken(), that.getRefreshToken()) &&
                Objects.equals(this.getRefreshTokenExpiration(), that.getRefreshTokenExpiration()) &&
                Objects.equals(this.getStatus(), that.getStatus()) &&
                Objects.equals(this.getLastRefreshed(), that.getLastRefreshed()) &&
                Objects.equals(this.getRefreshCount(), that.getRefreshCount()) &&
                Objects.equals(this.getLastStatusUpdated(), that.getLastStatusUpdated());
    }

    @Override
    public int hashCode() {
        return Objects.hash(
                getId(),
                getConfigId(),
                getAccessToken(),
                getAccessTokenExpiration(),
                getRefreshToken(),
                getRefreshTokenExpiration(),
                getStatus(),
                getLastRefreshed(),
                getRefreshCount(),
                getLastStatusUpdated());
    }

    @Override
    public String toString() {
        return MoreObjects.toStringHelper(this)
                .add("id", getId())
                .add("configId", getConfigId())
                .add("accessToken", "*****")
                .add("accessTokenExpiration", getAccessTokenExpiration())
                .add("refreshToken", "*****")
                .add("refreshTokenExpiration", getRefreshTokenExpiration())
                .add("status", getStatus())
                .add("lastRefreshed", getLastRefreshed())
                .add("refreshCount", getRefreshCount())
                .add("lastStatusUpdated", getLastStatusUpdated())
                .toString();
    }

    public static final class Builder {

        private String id;
        private String configId;
        private String accessToken;
        private Instant accessTokenExpiration;
        private String refreshToken;
        private Instant refreshTokenExpiration;
        private ClientTokenStatus status = ClientTokenStatus.UNKNOWN;
        private Instant lastRefreshed;
        private int refreshCount;
        private Instant lastStatusUpdated;

        private Builder() {
        }

        private Builder(@Nonnull final ClientToken initialData) {
            updateFrom(initialData);
        }

        private Builder(@Nonnull final ClientTokenMetadata initialData) {
            updateFrom(initialData);
        }

        private Builder(@Nonnull final ClientTokenEntity initialData) {
            updateFrom(initialData);
        }

        public Builder id(@Nullable final String id) {
            this.id = id;
            return this;
        }

        public Builder configId(@Nonnull final String configId) {
            this.configId = configId;
            return this;
        }

        public Builder accessToken(@Nonnull final String accessToken) {
            this.accessToken = accessToken;
            return this;
        }

        public Builder accessTokenExpiration(@Nonnull final Instant accessTokenExpiration) {
            this.accessTokenExpiration = accessTokenExpiration;
            return this;
        }

        public Builder refreshToken(@Nullable final String refreshToken) {
            this.refreshToken = refreshToken;
            return this;
        }

        public Builder refreshTokenExpiration(@Nullable final Instant refreshTokenExpiration) {
            this.refreshTokenExpiration = refreshTokenExpiration;
            return this;
        }

        public Builder status(@Nonnull final ClientTokenStatus status) {
            this.status = status;
            return this;
        }

        public Builder lastStatusUpdated(@Nonnull Instant lastStatusUpdated) {
            this.lastStatusUpdated = lastStatusUpdated;
            return this;
        }

        public Builder lastRefreshed(@Nullable final Instant lastRefreshed) {
            this.lastRefreshed = lastRefreshed;
            return this;
        }

        public Builder refreshCount(final int refreshCount) {
            this.refreshCount = refreshCount;
            return this;
        }

        public Builder incrementRefreshCount() {
            this.refreshCount++;
            return this;
        }

        public Builder updateFrom(final ClientToken clientToken) {
            return accessToken(clientToken.getAccessToken())
                    .accessTokenExpiration(clientToken.getAccessTokenExpiration())
                    .refreshToken(clientToken.getRefreshToken())
                    .refreshTokenExpiration(clientToken.getRefreshTokenExpiration());
        }

        public Builder updateFrom(final ClientTokenMetadata metadata) {
            return status(metadata.getStatus())
                    .lastStatusUpdated(metadata.getLastStatusUpdated())
                    .lastRefreshed(metadata.getLastRefreshed())
                    .refreshCount(metadata.getRefreshCount());
        }

        public Builder updateFrom(final ClientTokenEntity entity) {
            return updateFrom((ClientToken) entity)
                    .updateFrom((ClientTokenMetadata) entity)
                    .id(entity.getId())
                    .configId(entity.getConfigId());
        }

        public ClientTokenStatus getStatus() {
            return status;
        }

        public Instant getLastStatusUpdated() {
            return lastStatusUpdated;
        }

        public ClientTokenEntity build() {
            return new ClientTokenEntity(
                    id,
                    configId,
                    accessToken,
                    accessTokenExpiration,
                    refreshToken,
                    refreshTokenExpiration,
                    status,
                    lastRefreshed,
                    refreshCount,
                    lastStatusUpdated);
        }
    }

    private static <T> T maybeRequireNonNull(final T obj, final boolean requireNonNull, final String message) {
        return requireNonNull ? Objects.requireNonNull(obj, message) : obj;
    }
}
