package com.atlassian.multitenant.plugins;

import com.atlassian.event.api.EventListener;
import com.atlassian.event.api.EventPublisher;
import com.atlassian.multitenant.MultiTenantComponentFactory;
import com.atlassian.multitenant.MultiTenantCreator;
import com.atlassian.multitenant.MultiTenantDestroyer;
import com.atlassian.multitenant.MultiTenantComponentMap;
import com.atlassian.multitenant.MultiTenantManager;
import com.atlassian.multitenant.Tenant;
import com.atlassian.plugin.ModuleDescriptor;
import com.atlassian.plugin.ModuleDescriptorFactory;
import com.atlassian.plugin.Plugin;
import com.atlassian.plugin.PluginParseException;
import com.atlassian.plugin.StateAware;
import com.atlassian.plugin.descriptors.AbstractModuleDescriptor;
import com.atlassian.plugin.event.events.PluginFrameworkShutdownEvent;
import com.atlassian.plugin.hostcontainer.HostContainer;

import net.sf.cglib.proxy.Callback;
import net.sf.cglib.proxy.CallbackFilter;
import net.sf.cglib.proxy.Enhancer;
import net.sf.cglib.proxy.Factory;
import net.sf.cglib.proxy.MethodInterceptor;
import net.sf.cglib.proxy.MethodProxy;

import net.sf.hibernate.collection.Set;
import org.dom4j.Element;

import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.Arrays;
import java.util.concurrent.atomic.AtomicMarkableReference;
import java.util.concurrent.atomic.AtomicReference;

/**
 * Module descriptor factory that maintains a module descriptor instance for tenant for each module descriptor it
 * reutrns
 */
public class MultiTenantModuleDescriptorFactory implements ModuleDescriptorFactory
{
    // This class can be a cause of performance issues.  Possible future optimisations:
    //
    // Determine what methods on the initedModuleDescriptor are safe to always delegate to, and supply a separate
    // Dispatcher callback that always invokes that for those methods from the CallbackFilter (currently only delegates
    // to that after a module descriptor has been destroyed)
    //
    // Provide a separate callback for getModuleClass() that caches the result after the first invocation (currently
    // each invocation requires a lookup on the current tenant)

    /**
     * Indexes for the callbacks for the module descriptors.
     */
    private static final int LIFECYCLE_INTERCEPTOR = 0;
    private static final int TENANT_INTERCEPTOR = 1;
    private static final int GET_MODULE_CLASS_INTERCEPTOR = 2;

    private final ModuleDescriptorFactory target;
    private final HostContainer hostContainer;
    private final EventPublisher eventPublisher;
    private final MultiTenantComponentFactory factory;
    private final MultiTenantManager manager;

    /**
     * Construct the factory
     *
     * @param hostContainer The hosts host container, must *not* be a multi tenant host container
     * @param moduleDescriptorFactory The delegate moduleDescriptorFactory
     * @param eventPublisher The event publisher
     * @param factory
     * @param manager
     */
    public MultiTenantModuleDescriptorFactory(final HostContainer hostContainer, final ModuleDescriptorFactory moduleDescriptorFactory,
            final EventPublisher eventPublisher, MultiTenantComponentFactory factory, MultiTenantManager manager)
    {
        this.target = moduleDescriptorFactory;
        this.hostContainer = hostContainer;
        this.eventPublisher = eventPublisher;
        this.factory = factory;
        this.manager = manager;
    }

    public ModuleDescriptor<?> getModuleDescriptor(final String type)
            throws PluginParseException, IllegalAccessException, InstantiationException, ClassNotFoundException
    {
        final Class<? extends ModuleDescriptor> moduleDescriptorClazz = getModuleDescriptorClass(type);

        Enhancer enhancer = new Enhancer();
        enhancer.setSuperclass(moduleDescriptorClazz);
        enhancer.setCallbackTypes(new Class[] { LifecycleInterceptor.class, LifecycleInterceptor.TenantInterceptor.class,
                LifecycleInterceptor.GetModuleClassInterceptor.class });
        enhancer.setCallbackFilter(ModuleDescriptorCallbackFilter.INSTANCE);
        @SuppressWarnings ("unchecked")
        Class<? extends ModuleDescriptor> enhancedClass = enhancer.createClass();

        ModuleDescriptor<?> module = hostContainer.create(enhancedClass);
        if (module instanceof Factory)
        {
            // the object should be an instance of a cglib Factory
            LifecycleInterceptor interceptor = new LifecycleInterceptor(moduleDescriptorClazz);
            ((Factory) module).setCallbacks(new Callback[] { interceptor, interceptor.getTenantInterceptor(), interceptor.getGetModuleClassInterceptor() });
        }
        return module;
    }

    public Class<? extends ModuleDescriptor> getModuleDescriptorClass(final String type)
    {
        return target.getModuleDescriptorClass(type);
    }

    public boolean hasModuleDescriptor(final String type)
    {
        return target.hasModuleDescriptor(type);
    }

    private static class ModuleDescriptorCallbackFilter implements CallbackFilter
    {
        // Use the same instance for every enhancer, ensures correct caching of class generation is done
        private static ModuleDescriptorCallbackFilter INSTANCE = new ModuleDescriptorCallbackFilter();

        public int accept(Method method)
        {
            if (isInitMethod(method) || isDestoryMethod(method) || isEnabledMethod(method) || isDisabledMethod(method))
            {
                return LIFECYCLE_INTERCEPTOR;
            }
            else if (methodMatches(method, "getModuleClass"))
            {
                return GET_MODULE_CLASS_INTERCEPTOR;
            }
            else
            {
                return TENANT_INTERCEPTOR;
            }
        }
    }

    public class LifecycleInterceptor<M extends ModuleDescriptor> implements MethodInterceptor
    {
        private final Class<M> moduleDescriptorClass;
        private final MultiTenantComponentMap<M> map;
        private final AbstractModuleDescriptor initedModuleDescriptor;
        private final TenantInterceptor tenantInterceptor = new TenantInterceptor();
        private final GetModuleClassInterceptor getModuleClassInterceptor = new GetModuleClassInterceptor();
        private volatile Element initElement;
        private volatile Plugin plugin;
        private volatile boolean enabled;
        private volatile boolean destroyed;

        private LifecycleInterceptor(final Class<M> moduleDescriptorClass)
        {
            this.moduleDescriptorClass = moduleDescriptorClass;
            // This module descriptor instance exists so that we can parse the basic module descriptor elements out from
            // the module descriptor, eg module key etc, by calling the init method, so that we can then delegate to it
            // after the underlying module descriptors have been destroyed.  This is necessary in a few places like UPM
            // and the plugins system where generic module descriptor methods are called after being uninstalled.
            this.initedModuleDescriptor = new AbstractModuleDescriptor()
            {
                @Override
                public Object getModule()
                {
                    return null;
                }
            };
            map = factory.createComponentMapBuilder(new MultiTenantModuleDescriptorCreator()).
                    setLazyLoad(MultiTenantComponentMap.LazyLoadStrategy.EAGER_LOAD).construct();
            eventPublisher.register(this);
        }

        public TenantInterceptor getTenantInterceptor()
        {
            return tenantInterceptor;
        }

        public GetModuleClassInterceptor getGetModuleClassInterceptor()
        {
            return getModuleClassInterceptor;
        }

        public Object intercept(final Object obj, final Method method, final Object[] args, final MethodProxy proxy)
                throws Throwable
        {
            if (isInitMethod(method))
            {
                handleInit((Plugin) args[0], (Element) args[1]);
                return null;
            }
            if (isEnabledMethod(method))
            {
                handleEnabled();
                return null;
            }
            if (isDisabledMethod(method))
            {
                handleDisabled();
                return null;
            }
            if (isDestoryMethod(method))
            {
                handleDestroy((Plugin) args[0]);
                return null;
            }
            // We should never reach this point
            throw new IllegalStateException(method.getName() + " illegally invoked on MultiTenantModuleDescriptorInterceptor");
        }

        /**
         * This works around the fact that destroy() is not called on module descriptors when the plugin system is shut
         * down.
         *
         * @param event The event
         */
        @EventListener
        public void destroy(PluginFrameworkShutdownEvent event)
        {
            if (!destroyed)
            {
                destroy();
            }
        }

        private void destroy()
        {
            if (plugin != null)
            {
                map.destroy();
                this.plugin = null;
                enabled = false;
            }
            destroyed = true;
            eventPublisher.unregister(this);
        }

        private void handleInit(final Plugin plugin, final Element element) throws Throwable
        {
            // First init the initedModuleDescriptor
            initedModuleDescriptor.init(plugin, element);
            manager.runForEachTenant(new Runnable()
            {
                public void run()
                {
                    map.get().init(plugin, element);
                }
            }, true);
            this.plugin = plugin;
            this.initElement = element;
        }

        private void handleEnabled()
        {
            manager.runForEachTenant(new Runnable()
            {
                public void run()
                {
                    ModuleDescriptor md = map.get();
                    if (md instanceof StateAware)
                    {
                        ((StateAware) md).enabled();
                    }
                }
            }, true);
            enabled = true;
        }

        private void handleDisabled()
        {
            if (enabled)
            {
                enabled = false;
                getModuleClassInterceptor.resetCache();
                manager.runForEachTenant(new Runnable()
                {
                    public void run()
                    {
                        ModuleDescriptor md = map.get();
                        if (md instanceof StateAware)
                        {
                            ((StateAware) md).disabled();
                        }
                    }
                }, true);
            }
        }

        private void handleDestroy(Plugin plugin)
        {
            destroy();
        }

        private class MultiTenantModuleDescriptorCreator implements MultiTenantCreator<M>, MultiTenantDestroyer<M>
        {
            public M create(final Tenant tenant)
            {
                M md = hostContainer.create(moduleDescriptorClass);
                if (plugin != null && initElement != null)
                {
                    md.init(plugin, initElement);
                }
                if (enabled)
                {
                    if (md instanceof StateAware)
                    {
                        ((StateAware) md).enabled();
                    }
                }
                return md;
            }

            public void destroy(final Tenant tenant, final M instance)
            {
                if (plugin != null)
                {
                    if (enabled)
                    {
                        if (instance instanceof StateAware)
                        {
                            ((StateAware) instance).disabled();
                        }
                    }
                    instance.destroy(plugin);
                }
            }
        }

        public class TenantInterceptor implements MethodInterceptor
        {
            public Object intercept(Object obj, Method method, Object[] args, MethodProxy proxy) throws Throwable
            {
                ModuleDescriptor moduleDescriptor;
                Method methodToInvoke;
                if (destroyed)
                {
                    moduleDescriptor = initedModuleDescriptor;
                    // If a descriptor has overridden this method, we can't call it on the initedModuleDescriptor,
                    // otherwise we get a ClassCastException
                    try
                    {
                        methodToInvoke = AbstractModuleDescriptor.class.getMethod(method.getName(), method.getParameterTypes());
                    }
                    catch (NoSuchMethodException e)
                    {
                        throw new IllegalStateException("Method invoked on module descriptor after it was destroyed. This is only legal for some methods on AbstractModuleDescriptor.");
                    }
                }
                else
                {
                    moduleDescriptor = map.get();
                    methodToInvoke = method;
                }
                try
                {
                    return methodToInvoke.invoke(moduleDescriptor, args);
                }
                catch (InvocationTargetException e)
                {
                    throw e.getCause();
                }
            }
        }

        /**
         * Interceptor that caches the result of getModuleClass(), because this is a very commonly invoked method
         */
        public class GetModuleClassInterceptor implements MethodInterceptor
        {
            /**
             * We use a AtomicMarkableReference to allow us to cache null.
             * A lazy reference to either the cached value or a cached null
             * The boolean part of the AtomicMarkableReference indicates if we've cached it yet, or not.
             */
            private AtomicMarkableReference<Class> moduleClass = new AtomicMarkableReference<Class>(null, false);

            public Object intercept(Object obj, Method method, Object[] args, MethodProxy proxy) throws Throwable
            {
                // We only cache the value if the descriptor is enabled, when it's disabled it could change
                if (enabled)
                {
                    boolean[] isCached = new boolean[1];
                    Class clazz = moduleClass.get(isCached);
                    if (!isCached[0])
                    {
                        try
                        {
                            clazz = (Class) method.invoke(map.get());
                            moduleClass.compareAndSet(null, clazz, false, true);
                        }
                        catch (InvocationTargetException e)
                        {
                            throw e.getCause();
                        }
                    }
                    return clazz;
                }
                else
                {
                    try
                    {
                        return method.invoke(map.get());
                    }
                    catch (InvocationTargetException e)
                    {
                        throw e.getCause();
                    }
                }
            }

            public void resetCache()
            {
                Class oldValue = moduleClass.getReference();
                moduleClass.compareAndSet(oldValue, null, true, false);
            }
        }
    }

    private static boolean isInitMethod(Method method)
    {
        return methodMatches(method, "init", Plugin.class, Element.class);
    }

    private static boolean isEnabledMethod(Method method)
    {
        return methodMatches(method, "enabled");
    }

    private static boolean isDisabledMethod(Method method)
    {
        return methodMatches(method, "disabled");
    }

    private static boolean isDestoryMethod(Method method)
    {
        return methodMatches(method, "destroy", Plugin.class);
    }

    private static boolean methodMatches(Method method1, String methodName, Class<?>... parameterTypes)
    {
        return method1.getName().equals(methodName) && (Arrays.equals(method1.getParameterTypes(), parameterTypes));
    }

}