package com.atlassian.multitenant.impl;

import com.atlassian.multitenant.MultiTenantContext;
import com.atlassian.multitenant.MultiTenantLifecycleController;
import com.atlassian.multitenant.MultiTenantLifecycleAware;
import com.atlassian.multitenant.Tenant;
import com.atlassian.multitenant.MultiTenantManager;
import com.atlassian.multitenant.TenantReference;
import com.atlassian.multitenant.servlet.MultiTenantServletFilter;
import com.atlassian.util.concurrent.Sink;
import org.apache.log4j.Logger;

import java.lang.ref.WeakReference;
import java.util.Collection;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.ListIterator;
import java.util.concurrent.Callable;
import java.util.concurrent.CopyOnWriteArrayList;
import javax.servlet.http.HttpSession;

/**
 * Implementation of the MultiTenantManager and MultiTenantLifecycleController
 */
public class DefaultMultiTenantManager implements MultiTenantManager, MultiTenantLifecycleController
{
    private static final Logger log = Logger.getLogger(DefaultMultiTenantManager.class);

    private final MultiTenantDatastore datastore;
    private final TenantReference tenantReference;
    private final boolean singleTenantMode;

    // We weakly reference the listeners because some listeners may be specific to a tenant, and hence may hold reference
    // to the tenants expensive resources.
    private final List<WeakReference<MultiTenantLifecycleAware>> listeners = new CopyOnWriteArrayList<WeakReference<MultiTenantLifecycleAware>>();

    private final List<Tenant> startedTenants = new CopyOnWriteArrayList<Tenant>();

    public DefaultMultiTenantManager(final MultiTenantDatastore datastore, final TenantReference tenantReference,
            final boolean singleTenantMode)
    {
        this.datastore = datastore;
        this.tenantReference = tenantReference;
        this.singleTenantMode = singleTenantMode;
    }

    public void runForEachTenant(final Runnable runnable, boolean override)
    {
        for (Tenant tenant : startedTenants)
        {
            runForTenant(tenant, runnable, override);
        }
    }

    public void runForTenant(final Tenant tenant, final Runnable runnable, boolean override)
    {
        tenantReference.set(tenant, override);
        try
        {
            runnable.run();
        }
        finally
        {
            tenantReference.remove();
        }
    }

    public <T> T callForTenant(final Tenant tenant, final Callable<T> callable, boolean override)
            throws Exception
    {
        tenantReference.set(tenant, override);
        try
        {
            return callable.call();
        }
        finally
        {
            tenantReference.remove();
        }
    }

    /**
     * Registers a listener for start and stop events.  Listeners are executed in the order that they are registered on
     * start, and in the reverse order that they are registered on stop.
     *
     * @param lifecycleListener The listener to register
     */
    public void registerListener(final MultiTenantLifecycleAware lifecycleListener)
    {
        listeners.add(new WeakReference<MultiTenantLifecycleAware>(lifecycleListener));
    }

    /**
     * Deregisters a listener for start and stop events.
     *
     * @param lifecycleListener the listener to deregister
     */
    public void deregisterListener(final MultiTenantLifecycleAware lifecycleListener)
    {
        for (WeakReference<MultiTenantLifecycleAware> listener : listeners)
        {
            MultiTenantLifecycleAware hardListener = listener.get();
            if (hardListener != null)
            {
                if (hardListener.equals(lifecycleListener))
                {
                    listeners.remove(listener);
                }
            }
            else
            {
                listeners.remove(listener);
            }
        }
    }

    public synchronized void addTenant(final Tenant tenant)
    {
        datastore.addTenant(tenant);
        startTenant(tenant);
    }

    public synchronized void startTenant(final Tenant tenant)
    {
        startedTenants.add(tenant);
        try
        {
            tenantReference.set(tenant, true);
            runForEachListener(new Sink<MultiTenantLifecycleAware>()
            {
                public void consume(final MultiTenantLifecycleAware listener)
                {
                    listener.onTenantStart(tenant);
                }
            }, true);
        }
        finally
        {
            try
            {
                tenantReference.remove();
            }
            catch (IllegalStateException ise)
            {
                log.error(ise);
            }
        }
    }

    public synchronized void removeTenant(String tenantName)
    {
        Tenant tenant = datastore.get(tenantName);
        datastore.removeTenant(tenant);
        stopTenant(tenant);
    }

    public synchronized void stopTenant(final Tenant tenant)
    {
        try
        {
            tenantReference.set(tenant, true);
            runForEachListener(new Sink<MultiTenantLifecycleAware>()
            {
                public void consume(final MultiTenantLifecycleAware listener)
                {
                    try
                    {
                        listener.onTenantStop(tenant);
                    }
                    catch (Exception e)
                    {
                        // Just because one fails, doesn't mean we don't want the rest to run
                        log.error("Error stopping listener", e);
                    }
                }
            }, false);
        }
        finally
        {
            try
            {
                tenantReference.remove();
            }
            catch (IllegalStateException ise)
            {
                log.error(ise);
            }
        }
        startedTenants.remove(tenant);
    }

    public synchronized void refreshTenant(final Tenant tenant)
    {
        stopTenant(tenant);
        startTenant(tenant);
    }

    public void startAll()
    {
        for (Tenant tenant : datastore.getAll())
        {
            startTenant(tenant);
        }
    }

    public void stopAll()
    {
        for (Tenant tenant : startedTenants)
        {
            stopTenant(tenant);
        }
    }

    /**
     * Handles disposal of garbage collected listeners
     *
     * @param consumer The consumer to consume the listener
     * @param forward True if the listeners should be iterated through forwards, otherwise false
     */
    private void runForEachListener(Sink<MultiTenantLifecycleAware> consumer, boolean forward)
    {
        Iterator<WeakReference<MultiTenantLifecycleAware>> iterator;
        if (forward)
        {
            iterator = listeners.iterator();
        }
        else
        {
            // Create a reverse iterator
            final ListIterator<WeakReference<MultiTenantLifecycleAware>> listIterator = listeners.listIterator(listeners.size());
            iterator = new Iterator<WeakReference<MultiTenantLifecycleAware>>()
            {
                public boolean hasNext()
                {
                    return listIterator.hasPrevious();
                }

                public WeakReference<MultiTenantLifecycleAware> next()
                {
                    return listIterator.previous();
                }

                public void remove()
                {
                    listIterator.remove();
                }
            };
        }
        while (iterator.hasNext())
        {
            WeakReference<MultiTenantLifecycleAware> listener = iterator.next();
            MultiTenantLifecycleAware hardListener = listener.get();
            if (hardListener != null)
            {
                consumer.consume(hardListener);
            }
            else
            {
                // Don't call iterator.remove(), CopyOnWriteArrayList doesn't support it.
                listeners.remove(listener);
            }
        }
    }

    public boolean isSingleTenantMode()
    {
        return singleTenantMode;
    }

    public Collection<Tenant> getAllTenants()
    {
        return Collections.unmodifiableCollection(startedTenants);
    }

    public Tenant getTenantFromSession(final HttpSession session)
    {
        final String name = MultiTenantServletFilter.getTenantName(session);
        return datastore.get(name);
    }

    public Tenant getTenantByName(final String name)
    {
        return datastore.get(name);
    }

    public boolean isSystemTenant()
    {
        return tenantReference.get() == MultiTenantContext.getSystemTenant();
    }
}
