package com.atlassian.multitenant;

import java.util.Collection;
import java.util.concurrent.Callable;
import javax.servlet.http.HttpSession;

/**
 * Manager for Multi-Tenant stuff
 * <p/>
 * Contains an event system and methods for running something for a particular tenant.
 */
public interface MultiTenantManager
{
    /**
     * Register the given listener
     *
     * @param lifecycleListener The listener to register
     */
    void registerListener(MultiTenantLifecycleAware lifecycleListener);

    /**
     * Deregister the given listener
     *
     * @param lifecycleListener The listener to deregister
     */
    void deregisterListener(MultiTenantLifecycleAware lifecycleListener);

    /**
     * Run the given runnable for each tenant
     *
     * @param runnable The runnable to run
     * @param override True if overriding an existing context is allowed
     */
    void runForEachTenant(Runnable runnable, boolean override);

    /**
     * Run the given runnable for the given tenant
     *
     * @param tenant The tenant to run the runnable for
     * @param runnable The runnable to run
     * @param override True if overriding an existing context is allowed
     */
    void runForTenant(Tenant tenant, Runnable runnable, boolean override);

    /**
     * Call the given callable for the given tenant
     *
     * @param tenant The tenant to call the callable for
     * @param callable The callable to call
     * @param override True if overriding an existing context is allowed
     * @return The result of the callable
     * @throws Exception If an exception was encountered by the callable
     */
    <T> T callForTenant(Tenant tenant, Callable<T> callable, boolean override) throws Exception;

    /**
     * True if we're in single tenant mode.  What this means is that there is a single tenant, the system tenant, and no
     * other tenants are supported.  This is for behind the firewall operation, and the main implication of this is that
     * the filter will always set the tenant to be the system tenant, regardless of the hostname.
     *
     * @return True if we're in single tenant mode
     */
    boolean isSingleTenantMode();

    /**
     * Get all the started tenants
     *
     * @return All the started tenants in the system
     */
    Collection<Tenant> getAllTenants();

    /**
     * Given a session, figure out what tenant it belongs to.
     *
     * @param session the session you want the tenant for
     * @return the tenant associated with the sessions
     * @throws IllegalStateException if no tenant is associated with the session
     */
    Tenant getTenantFromSession(HttpSession session);

    /**
     * Get the tenant with the given name
     *
     * @param name The name of the tenant to get
     * @return The tenant, or null if none found
     */
    Tenant getTenantByName(String name);

    /**
     * Is the current tenant the system tenant?
     *
     * @return true if the current tenant is the system tenant
     */
    boolean isSystemTenant();
}
