package com.atlassian.multitenant.servlet;

import com.atlassian.multitenant.MultiTenantContext;
import com.atlassian.multitenant.Tenant;
import org.dom4j.Document;
import org.dom4j.DocumentHelper;
import org.dom4j.Element;
import org.dom4j.io.OutputFormat;
import org.dom4j.io.XMLWriter;

import java.io.IOException;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

/**
 * Servlet for controlling multi tenant tenants
 */
public class MultiTenantServlet extends HttpServlet
{
    /**
     * This is overridden to provide security, no methods are allowed to be called on this servlet if the request
     * doesn't have permission.
     *
     * @param req The request
     * @param resp The response
     */
    @Override
    protected void service(final HttpServletRequest req, final HttpServletResponse resp)
            throws ServletException, IOException
    {
        if (hasPermission(req, resp))
        {
            super.service(req, resp);
        }
    }

    @Override
    protected void doGet(final HttpServletRequest req, final HttpServletResponse resp)
            throws ServletException, IOException
    {
        // Crude list of all tenants. with the hostnames they are configured to listen on, nothing more
        Document document = DocumentHelper.createDocument();
        if (req.getPathInfo() != null && req.getPathInfo().length() > 0)
        {
            String tenantName = req.getPathInfo().substring(1);
            Tenant tenant = MultiTenantContext.getManager().getTenantByName(tenantName);
            if (tenant == null)
            {
                resp.sendError(HttpServletResponse.SC_NOT_FOUND, "Tenant not found");
                return;
            }
            Element tenantElement = document.addElement("tenant");
            populateTenant(tenantElement, tenant);
        }
        else
        {
            Element rootElement = document.addElement("multitenant");
            for (Tenant tenant : MultiTenantContext.getManager().getAllTenants())
            {
                Element tenantElement = rootElement.addElement("tenant");
                populateTenant(tenantElement, tenant);
            }
        }
        // Make sure content type is application/xml
        resp.setContentType("application/xml");

        XMLWriter xmlWriter = new XMLWriter(resp.getWriter(), OutputFormat.createPrettyPrint());
        xmlWriter.write(document);
    }

    private void populateTenant(Element tenantElement, Tenant tenant)
    {
        tenantElement.addAttribute("name", tenant.getName());
        Element hostnamesElement = tenantElement.addElement("hostnames");
        for (String hostname : tenant.getHostnames())
        {
            Element hostnameElement = hostnamesElement.addElement("hostname");
            hostnameElement.setText(hostname);
        }
    }

    @Override
    protected void doPost(final HttpServletRequest req, final HttpServletResponse resp)
            throws ServletException, IOException
    {
        // Prevent XSRF
        if (!req.getContentType().startsWith("application/xml"))
        {
            resp.sendError(HttpServletResponse.SC_FORBIDDEN, "Only application/xml allowed");
            return;
        }
        Tenant tenant = MultiTenantContext.getParser().parseTenant(req.getReader());
        MultiTenantContext.getController().addTenant(tenant);
        resp.setStatus(HttpServletResponse.SC_NO_CONTENT);
    }

    @Override
    protected void doDelete(final HttpServletRequest req, final HttpServletResponse resp)
            throws ServletException, IOException
    {
        String name = req.getPathInfo();
        if (name != null && name.length() > 1)
        {
            // Strip leading slash
            name = name.substring(1);
            MultiTenantContext.getController().removeTenant(name);
            resp.setStatus(HttpServletResponse.SC_NO_CONTENT);
        }
        else
        {
            resp.sendError(HttpServletResponse.SC_NOT_FOUND, "A resource to delete was not specified");
        }
    }

    private boolean hasPermission(HttpServletRequest request, HttpServletResponse response) throws IOException
    {
        // First ensure that this is the system tenant. != is fine, they must be the same reference
        if (MultiTenantContext.getSystemTenant() != MultiTenantContext.getTenantReference().get())
        {
            response.sendError(HttpServletResponse.SC_FORBIDDEN, "Not system tenant");
            return false;
        }
        // Check if the request is authorised
        if (!MultiTenantContext.getAuthorisationProvider().canManageTenants(request))
        {
            response.sendError(HttpServletResponse.SC_FORBIDDEN, "Not authorised");
            return false;
        }
        return true;
    }
}
