package com.atlassian.multitenant.impl.datastore;

import com.atlassian.multitenant.CustomConfigHandler;
import com.atlassian.multitenant.Tenant;
import com.atlassian.multitenant.impl.DefaultTenant;
import com.atlassian.multitenant.impl.MultiTenantDatastore;
import com.atlassian.multitenant.impl.MultiTenantParser;
import com.atlassian.util.concurrent.CopyOnWriteMap;
import org.apache.commons.io.IOUtils;
import org.apache.log4j.Logger;
import org.dom4j.Document;
import org.dom4j.DocumentException;
import org.dom4j.DocumentHelper;
import org.dom4j.Element;
import org.dom4j.io.OutputFormat;
import org.dom4j.io.SAXReader;
import org.dom4j.io.XMLWriter;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.io.Reader;
import java.io.Writer;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;

/**
 * Datastore that stores everything in an XML file.
 * <p/>
 * This datastore is a write through cache.
 */
@SuppressWarnings("unchecked")
public class XmlMultiTenantDatastore implements MultiTenantDatastore, MultiTenantParser
{
    private final Map<String, CustomConfigHandler<?>> handlers;
    private final File configFile;

    private final Map<String, Tenant> tenants = CopyOnWriteMap.newHashMap();
    private final Map<String, Tenant> tenantsByHostname = CopyOnWriteMap.newHashMap();

    private static final Logger log = Logger.getLogger(XmlMultiTenantDatastore.class);

    public XmlMultiTenantDatastore(final Map<String, CustomConfigHandler<?>> handlers, final File configFile)
    {
        this.handlers = handlers;
        this.configFile = configFile;
        if (configFile != null)
        {
            loadFrom(configFile);
        }
    }

    public Tenant get(final String name)
    {
        return tenants.get(name);
    }

    public Collection<Tenant> getAll()
    {
        return Collections.unmodifiableCollection(tenants.values());
    }

    public Tenant getByHostname(final String hostname)
    {
        return tenantsByHostname.get(hostname);
    }

    void loadFrom(File file)
    {
        if (!file.exists())
        {
            // This not only initialises the file, it ensures we have write access to it
            Writer writer = null;
            try
            {
                // First try and make the parent directory of the file, because at this point, it's often the case
                // that the directory doesn't exist, such as in the JIRA func tests
                file.getParentFile().mkdirs();
                writer = new FileWriter(file);
                writer.write("<multitenant/>");
            }
            catch (IOException ioe)
            {
                throw new IllegalArgumentException(ioe);
            }
            finally
            {
                IOUtils.closeQuietly(writer);
            }
        }

        Document document = loadDocument(file);
        Element root = document.getRootElement();
        for (Element tenantElement : (Iterable<Element>) root.elements("tenant"))
        {
            Tenant tenant = parseTenant(tenantElement);
            addTenantInternal(tenant);
        }
    }

    private void addTenantInternal(final Tenant tenant)
    {
        tenants.put(tenant.getName(), tenant);
        for (String hostname : tenant.getHostnames())
        {
            if (tenantsByHostname.containsKey(hostname))
            {
                log.warn("Duplicate hostname detected in config: " + hostname);
            }
            tenantsByHostname.put(hostname, tenant);
        }
    }

    public synchronized void addTenant(final Tenant tenant)
    {
        addTenantInternal(tenant);
        save();
    }

    public synchronized void removeTenant(final Tenant tenant)
    {
        for (String hostname : tenant.getHostnames())
        {
            tenantsByHostname.remove(hostname);
        }
        tenants.remove(tenant.getName());

        save();
    }

    public Tenant parseTenant(Reader reader) throws IOException
    {
        SAXReader xmlReader = new SAXReader();
        try
        {
            Document document = xmlReader.read(reader);
            return parseTenant(document.getRootElement());
        }
        catch (DocumentException e)
        {
            throw new RuntimeException("Error parsing XML", e);
        }
    }

    private Tenant parseTenant(Element tenant)
    {
        String name = tenant.attributeValue("name");
        Map<Class<?>, Object> customConfig = new HashMap<Class<?>, Object>();
        for (Element config : (Iterable<Element>) tenant.elements())
        {
            String configName = config.getName();
            if (!configName.equals("hostnames") && !configName.equals("homeDir"))
            {
                CustomConfigHandler<?> handler = handlers.get(configName);
                if (handler != null)
                {
                    customConfig.put(handler.getBeanClass(), handler.parse(config));
                }
                else
                {
                    log.error("Unknown config item in multi tenant config file: " + configName);
                }
            }
        }

        Collection<String> hostnames = new HashSet<String>();
        Element hostnamesElement = tenant.element("hostnames");
        if (hostnamesElement != null)
        {
            for (Element hostnameElement : (Iterable<Element>) hostnamesElement.elements("hostname"))
            {
                hostnames.add(hostnameElement.getTextTrim());
            }
        }

        String homeDir = tenant.elementTextTrim("homeDir");

        return new DefaultTenant(name, hostnames, homeDir, customConfig);
    }

    private void save()
    {
        if (configFile != null)
        {
            saveTo(configFile);
        }
    }

    void saveTo(File file)
    {
        Document document = DocumentHelper.createDocument();

        Element rootElement = document.addElement("multitenant");
        for (Tenant tenant : getAll())
        {
            Element tenantElement = rootElement.addElement("tenant");
            tenantElement.addAttribute("name", tenant.getName());
            Element hostnamesElement = tenantElement.addElement("hostnames");
            for (String hostname : tenant.getHostnames())
            {
                Element hostnameElement = hostnamesElement.addElement("hostname");
                hostnameElement.setText(hostname);
            }
            Element homeDirElement = tenantElement.addElement("homeDir");
            homeDirElement.setText(tenant.getHomeDir());
            for (Map.Entry<String, CustomConfigHandler<?>> handler : handlers.entrySet())
            {
                Object customConfig = tenant.getConfig(handler.getValue().getBeanClass());
                if (customConfig != null)
                {
                    Element customConfigElement = tenantElement.addElement(handler.getKey());
                    // Stupid generics
                    CustomConfigHandler<Object> objectHandler = (CustomConfigHandler<Object>) handler.getValue();
                    objectHandler.writeTo(customConfigElement, customConfig);
                }
            }

        }
        FileWriter writer = null;
        try
        {
            writer = new FileWriter(file);
            XMLWriter xmlWriter = new XMLWriter(writer, OutputFormat.createPrettyPrint());
            xmlWriter.write(document);
        }
        catch (IOException ioe)
        {
            throw new IllegalArgumentException("Error writing config file", ioe);
        }
        finally
        {
            IOUtils.closeQuietly(writer);
        }
    }

    private Document loadDocument(File file)
    {
        FileReader fileReader = null;
        SAXReader xmlReader = new SAXReader();
        try
        {
            fileReader = new FileReader(file);
            return xmlReader.read(fileReader);
        }
        catch (DocumentException de)
        {
            throw new IllegalArgumentException("Error parsing config file", de);
        }
        catch (FileNotFoundException fnfe)
        {
            throw new IllegalArgumentException("Error opening config file", fnfe);
        }
        finally
        {
            IOUtils.closeQuietly(fileReader);
        }
    }
}

