package com.atlassian.multitenant.impl;

import com.atlassian.multitenant.MultiTenantContext;
import com.atlassian.multitenant.MultiTenantCreator;
import com.atlassian.multitenant.MultiTenantDestroyer;
import com.atlassian.multitenant.MultiTenantComponentMap;
import com.atlassian.multitenant.MultiTenantLifecycleAware;
import com.atlassian.multitenant.MultiTenantManager;
import com.atlassian.multitenant.TenantReference;
import com.atlassian.multitenant.Tenant;

import java.lang.reflect.ParameterizedType;
import java.lang.reflect.Type;
import java.util.ArrayList;
import java.util.Collection;

/**
 * Implementation of the MultiTenantComponentMap.
 */
public class MultiTenantComponentMapImpl<C> implements MultiTenantComponentMap<C>, MultiTenantLifecycleAware
{
    private final MultiTenantCreator<C> creator;
    private final TenantReference tenantReference;
    private final MultiTenantManager manager;
    private MultiTenantDestroyer<C> destroyer;
    private LazyLoadStrategy lazyLoadStrategy = LazyLoadStrategy.LAZY_LOAD;
    private NoTenantStrategy strategy = NoTenantStrategy.FAIL;

    @SuppressWarnings("unchecked")
    MultiTenantComponentMapImpl(final MultiTenantCreator<C> creator,
            final TenantReference tenantReference, final MultiTenantManager manager)
    {
        this.creator = creator;
        this.tenantReference = tenantReference;
        this.manager = manager;
        if (creator instanceof MultiTenantDestroyer)
        {
            this.destroyer = (MultiTenantDestroyer<C>) creator;
        }
    }

    /**
     * This should be called once the map is fully constructed (all mutable properties set).  It should only ever be
     * called by the map builder.
     */
    void init()
    {
        if (lazyLoadStrategy == LazyLoadStrategy.EAGER_LOAD)
        {
            initialiseAll();
        }
    }

    private TenantComponentMap getTenant()
    {
        return castTenant(tenantReference.get());
    }

    private TenantComponentMap castTenant(Tenant tenant)
    {
        if (tenant instanceof TenantComponentMap)
        {
            return (TenantComponentMap) tenant;
        }
        else
        {
            throw new IllegalStateException("Tenant is not a component map tenant!");
        }
    }

    public C get()
    {
        TenantComponentMap tenant;
        if (strategy == NoTenantStrategy.FAIL)
        {
            tenant = getTenant();
        }
        else
        {
            if (tenantReference.isSet())
            {
                tenant = getTenant();
            }
            else
            {
                tenant = (TenantComponentMap) MultiTenantContext.getSystemTenant();
            }
        }
        return get(tenant);
    }

    public Collection<C> getAll()
    {
        Collection<C> components = new ArrayList<C>();
        for (Tenant tenant : manager.getAllTenants())
        {
            C component = castTenant(tenant).getObject(this);
            if (component != null)
            {
                components.add(component);
            }
        }
        return components;
    }

    private C get(TenantComponentMap tenant)
    {
        // Must use containsKey, in case the creator returns null, and so the component is validly a null value
        if (!tenant.hasObject(this))
        {
            synchronized (this)
            {
                // Double check
                if (!tenant.hasObject(this))
                {
                    if (creator == null)
                    {
                        throw new IllegalStateException("No implementation has been registered for this tenant, "
                                + "and there's no creator to create it");
                    }
                    C value = creator.create(tenant);
                    tenant.putObject(this, value);
                }
            }
        }
        return tenant.getObject(this);
    }

    public void initialiseAll()
    {
        for (Tenant tenant : manager.getAllTenants())
        {
            get(castTenant(tenant));
        }
    }

    public void destroy()
    {
        for (Tenant tenant : manager.getAllTenants())
        {
            // This will clean the component up and remove it
            onTenantStop(tenant);
        }
        // Deregister from the manager
        manager.deregisterListener(this);
    }

    public boolean isInitialised()
    {
        TenantComponentMap tenant = getTenant();
        return tenant.hasObject(this);
    }

    public void addInstance(final C object)
    {
        TenantComponentMap tenant = getTenant();
        tenant.putObject(this, object);
    }

    public void onTenantStart(final Tenant tenant)
    {
        if (lazyLoadStrategy != LazyLoadStrategy.LAZY_LOAD)
        {
            get(castTenant(tenant));
        }
    }

    public void onTenantStop(final Tenant tenant)
    {
        TenantComponentMap tenantMap = castTenant(tenant);
        C component = tenantMap.getObject(this);
        if (component != null)
        {
            if (destroyer != null)
            {
                destroyer.destroy(tenant, component);
            }
            tenantMap.removeObject(this);
        }
    }

    void setDestroyer(final MultiTenantDestroyer<C> destroyer)
    {
        this.destroyer = destroyer;
    }

    void setLazyLoad(final LazyLoadStrategy lazyLoadStrategy)
    {
        this.lazyLoadStrategy = lazyLoadStrategy;
    }

    void setNoInstanceStrategy(NoTenantStrategy strategy)
    {
        this.strategy = strategy;
    }

    public String toString()
    {
        // Try and work out what our type is... usually creators have an explicit generic type declared
        for (Type type : creator.getClass().getGenericInterfaces())
        {
            if (type instanceof ParameterizedType)
            {
                ParameterizedType paramType = (ParameterizedType) type;
                if (paramType.getRawType().equals(MultiTenantCreator.class))
                {
                    // Excellent
                    Type[] params = paramType.getActualTypeArguments();
                    if (params.length > 0 && params[0] instanceof Class)
                    {
                        // We have a type
                        return "Multi-Tenant component map of " + ((Class) params[0]).getName();
                    }
                }
            }
        }
        return "Multi-Tenant component map with creator: " + creator.toString();
    }
}
