package com.atlassian.multitenant.impl;

import com.atlassian.multitenant.MultiTenantComponentFactory;
import com.atlassian.multitenant.MultiTenantComponentMap;
import com.atlassian.multitenant.MultiTenantComponentMapBuilder;
import com.atlassian.multitenant.MultiTenantCreator;
import com.atlassian.multitenant.MultiTenantManager;
import com.atlassian.multitenant.Tenant;
import com.atlassian.multitenant.TenantReference;
import net.sf.cglib.proxy.Enhancer;
import net.sf.cglib.proxy.MethodInterceptor;
import net.sf.cglib.proxy.MethodProxy;
import org.apache.log4j.Logger;

import java.lang.reflect.InvocationHandler;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.lang.reflect.Proxy;
import java.util.Collections;
import java.util.Set;

/**
 * This is a factory for creating components that need to be aware of which tenant they are working for.
 */
public class MultiTenantComponentFactoryImpl implements MultiTenantComponentFactory
{
    private static final Logger log = Logger.getLogger(MultiTenantComponentFactoryImpl.class);
    private final TenantReference tenantReference;
    private final MultiTenantManager manager;
    private final MultiTenantDatastore datastore;

    public MultiTenantComponentFactoryImpl(final TenantReference tenantReference, final MultiTenantManager manager,
            final MultiTenantDatastore datastore)
    {
        this.tenantReference = tenantReference;
        this.manager = manager;
        this.datastore = datastore;
    }

    public <C> MultiTenantComponentMapBuilder<C> createComponentMapBuilder(MultiTenantCreator<C> creator)
    {
        return new MultiTenantComponentMapBuilderImpl<C>(creator, tenantReference, manager);
    }

    public <C> MultiTenantComponentMap<C> createComponentMap(MultiTenantCreator<C> creator)
    {
        return createComponentMapBuilder(creator).construct();
    }

    public <C> Object createComponent(MultiTenantComponentMap<C> map, ClassLoader classLoader, Class<? super C>... interfaces)
    {
        return createComponent(map, classLoader, Collections.<Method>emptySet(), interfaces);
    }

    @SuppressWarnings ("unchecked")
    public <C> C createComponent(MultiTenantComponentMap<C> map, Class<C> inter)
    {
        return (C) createComponent(map, inter.getClassLoader(), new Class[] { inter });
    }

    public <C> C createComponent(MultiTenantCreator<C> creator, Class<C> inter)
    {
        return createComponent(createComponentMap(creator), inter);
    }

    public <C> C createComponent(final Class<? extends C> clazz, Class<C> inter)
    {
        return createComponent(new SimpleInstantiationCreator<C>(clazz), inter);
    }

    @SuppressWarnings ("unchecked")
    public <C> C createComponent(MultiTenantComponentMap<C> map, Set<Method> invokeForAllMethods,
            Class<C> inter)
    {
        return (C) createComponent(map, inter.getClassLoader(), invokeForAllMethods, new Class[] { inter });
    }

    public <C> Object createComponent(MultiTenantComponentMap<C> map, ClassLoader classLoader, Set<Method> invokeForAllMethods,
            Class<? super C>... interfaces)
    {
        MultiTenantAwareInvocationHandler handler = new MultiTenantAwareInvocationHandler<C>(map, invokeForAllMethods,
                tenantReference, manager);
        return Proxy.newProxyInstance(classLoader, interfaces, handler);
    }

    private <C> Enhancer createEnhancer(MultiTenantComponentMap<C> map, Class<C> superClass)
    {
        Enhancer enhancer = new Enhancer();
        MultiTenantAwareMethodInterceptor interceptor = new MultiTenantAwareMethodInterceptor<C>(map, tenantReference,
                manager);
        enhancer.setSuperclass(superClass);
        enhancer.setCallback(interceptor);
        return enhancer;
    }

    public <C> C createEnhancedComponent(MultiTenantCreator<C> creator, Class<C> superClass)
    {
        return createEnhancedComponent(createComponentMap(creator), superClass);
    }

    public <C> C createEnhancedComponent(Class<C> superClass)
    {
        return createEnhancedComponent(new SimpleInstantiationCreator<C>(superClass), superClass);
    }

    @SuppressWarnings ("unchecked")
    public <C> C createEnhancedComponent(MultiTenantComponentMap<C> map, Class superClass)
    {
        return (C) createEnhancer(map, superClass).create();
    }

    /**
     * Abstract invocation handler, uses the MultiTenantComponentMap to do most of its hard work
     */
    private static abstract class AbstractMultiTenantAwareInvocationHandler<C>
    {
        private final MultiTenantComponentMap<C> map;
        private final Set<Method> invokeForAllMethods;
        private final TenantReference tenantReference;
        private final MultiTenantManager manager;

        private AbstractMultiTenantAwareInvocationHandler(final MultiTenantComponentMap<C> map,
                final Set<Method> invokeForAllMethods, final TenantReference tenantReference,
                final MultiTenantManager manager)
        {
            this.map = map;
            this.invokeForAllMethods = invokeForAllMethods;
            this.tenantReference = tenantReference;
            this.manager = manager;
        }

        private AbstractMultiTenantAwareInvocationHandler(final MultiTenantComponentMap<C> map,
                final TenantReference tenantReference, final MultiTenantManager manager)
        {
            this(map, Collections.<Method>emptySet(), tenantReference, manager);
        }

        protected Object invokeInternal(final Method method, final Object[] args, final Object proxyObject) throws Throwable
        {
            // Don't proxy Object calls
            if (method.getDeclaringClass().equals(Object.class))
            {
                if (method.getName().equals("equals"))
                {
                    return proxyObject == args[0];
                }
                else if (method.getName().equals("toString"))
                {
                    return "Multi-Tenant proxy for " + map.toString();
                }
                else if (method.getName().equals("hashCode"))
                {
                    // Use our own hash code
                    return hashCode();
                }
            }
            if (invokeForAllMethods.contains(method) && !tenantReference.isSet())
            {
                manager.runForEachTenant(new Runnable()
                {
                    public void run()
                    {
                        C component = map.get();
                        try
                        {
                            method.invoke(component, args);
                        }
                        catch (Exception e)
                        {
                            log.error("Error invoking invoke for all method on tenant", e);
                        }
                    }
                });
                return null;
            }
            else
            {
                C component = map.get();
                try
                {
                    return method.invoke(component, args);
                }
                catch (InvocationTargetException ite)
                {
                    throw ite.getTargetException();
                }
            }
        }
    }

    /**
     * Java Dynamic Proxy invocation handler
     */
    public static class MultiTenantAwareInvocationHandler<C> extends AbstractMultiTenantAwareInvocationHandler<C>
            implements InvocationHandler
    {
        public MultiTenantAwareInvocationHandler(MultiTenantComponentMap<C> map, final Set<Method> invokeForAllMethods,
                final TenantReference tenantReference, final MultiTenantManager manager)
        {
            super(map, invokeForAllMethods, tenantReference, manager);
        }

        public MultiTenantAwareInvocationHandler(MultiTenantComponentMap<C> map, final TenantReference tenantReference,
                final MultiTenantManager manager)
        {
            this(map, Collections.<Method>emptySet(), tenantReference, manager);
        }

        public Object invoke(final Object proxy, final Method method, final Object[] args) throws Throwable
        {
            return invokeInternal(method, args, proxy);
        }
    }

    /**
     * CGLib MethodInterceptor
     */
    public static class MultiTenantAwareMethodInterceptor<C> extends AbstractMultiTenantAwareInvocationHandler<C>
            implements MethodInterceptor
    {
        public MultiTenantAwareMethodInterceptor(MultiTenantComponentMap<C> map, final TenantReference tenantReference,
                final MultiTenantManager manager)
        {
            super(map, tenantReference, manager);
        }

        public Object intercept(final Object obj, final Method method, final Object[] args, final MethodProxy proxy)
                throws Throwable
        {
            return invokeInternal(method, args, obj);
        }
    }

    private static class SimpleInstantiationCreator<C> implements MultiTenantCreator<C>
    {
        private final Class<? extends C> clazz;

        private SimpleInstantiationCreator(final Class<? extends C> clazz)
        {
            this.clazz = clazz;
        }

        public C create(final Tenant tenant)
        {
            try
            {
                return clazz.newInstance();
            }
            catch (IllegalAccessException iae)
            {
                throw new IllegalArgumentException("Unable to instantiate class", iae);
            }
            catch (InstantiationException ie)
            {
                throw new IllegalArgumentException("Unable to instantiate class", ie);
            }
        }
    }

}
