package com.atlassian.multitenant;

import com.atlassian.multitenant.impl.DefaultMultiTenantMatcher;
import com.atlassian.multitenant.impl.DefaultTenantReference;
import com.atlassian.multitenant.impl.DefaultMultiTenantManager;
import com.atlassian.multitenant.impl.MultiTenantComponentFactoryImpl;
import com.atlassian.multitenant.impl.MultiTenantDatastore;
import com.atlassian.multitenant.impl.MultiTenantParser;
import com.atlassian.multitenant.impl.NoPermissionAuthorisationProvider;
import com.atlassian.multitenant.impl.SystemTenantReference;
import com.atlassian.multitenant.impl.datastore.DelegatingSystemTenantDatastore;
import com.atlassian.multitenant.impl.datastore.EmptyDatastore;
import com.atlassian.multitenant.impl.datastore.XmlMultiTenantDatastore;
import com.atlassian.multitenant.impl.matchers.HostnameMatcher;
import com.atlassian.multitenant.impl.matchers.RequestHeaderMatcher;
import org.apache.commons.io.IOUtils;
import org.apache.log4j.Logger;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.Arrays;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.Map;
import java.util.Properties;

/**
 * The base class from which the multi tenant system is accessed.  Before any application starts accessing this API,
 * doDefaultInit() should be called.  Test code need not call this though, and is encouraged to inject mock objects into
 * the static setters.
 */
public class MultiTenantContext
{
    private static final String MULTI_TENANT_PROPERTIES_FILE = "multitenant.properties";
    private static final String MULTI_TENANT_XML_FILE_NAME = "multitenant.xml";
    private static final String MULTI_TENANT_AUTHORISATION_PROVIDER_KEY = "multitenant.authorisation.provider.class";
    public static final String MULTI_TENANT_ENABLED_KEY = "multitenant.enabled";
    public static final String SYSTEM_TENANT_PROVIDER_KEY = "multitenant.system.tenant.provider";
    /**
     * Indicates that although we're running with multi-tenanting enabled, there is only a single tenant, the system
     * tenant.
     */
    public static final String SINGLE_TENANT_KEY = "multitenant.single.tenant.mode";
    public static final String MULTI_TENANT_SYSTEM_HOME_PROPERTY = "multitenant.system.home";
    private static final String DEFAULT_MULTI_TENANT_SYSTEM_HOME = ".";
    private static final String HANDLER_PREFIX = "multitenant.handler.";
    private static final Logger log = Logger.getLogger(MultiTenantContext.class);

    private static TenantReference tenantReference;
    private static MultiTenantManager manager;
    private static MultiTenantLifecycleController controller;
    private static MultiTenantComponentFactory factory;
    private static MultiTenantMatcher matcher;
    private static MultiTenantParser parser;
    private static File multiTenantSystemHome;
    private static Tenant systemTenant;
    private static AuthorisationProvider authorisationProvider;


    /**
     * Do default initialisation.  This initialises the multitenant library, and must be called before anything else,
     * unless you are setting up things up manually for testing etc.
     */
    public static void defaultInit()
    {
        // Look for a multitenant properties file, if none, multi tenant mode is not enabled
        Properties props = loadProperties();
        if (props != null)
        {
            defaultInit(props);
        }
    }

    public static void defaultInit(Properties props)
    {
        if (Boolean.parseBoolean(props.getProperty(MULTI_TENANT_ENABLED_KEY, "false")))
        {
            Map<String, CustomConfigHandler<?>> handlers = initHandlers(props);

            // See if there's a system tenant
            systemTenant = getSystemTenant(props);
            if (systemTenant != null)
            {
                multiTenantSystemHome = new File(systemTenant.getHomeDir());
            }
            else
            {
                multiTenantSystemHome = locateSystemHome(props);
            }

            // Create the backing datastore
            MultiTenantDatastore backingDatastore;
            boolean singleTenantMode = Boolean.parseBoolean(props.getProperty(SINGLE_TENANT_KEY, "true"));
            if (singleTenantMode)
            {
                EmptyDatastore emptyDatastore = new EmptyDatastore();
                parser = emptyDatastore;
                backingDatastore = emptyDatastore;
                tenantReference = new SystemTenantReference();
            }
            else
            {
                File multiTenantXmlFile = new File(multiTenantSystemHome, MULTI_TENANT_XML_FILE_NAME);
                XmlMultiTenantDatastore xmlDatastore = new XmlMultiTenantDatastore(handlers, multiTenantXmlFile);
                parser = xmlDatastore;
                backingDatastore = xmlDatastore;
                tenantReference = new DefaultTenantReference();
            }

            // If there's a system tenant, then the actual datastore needs to include it
            MultiTenantDatastore datastore;
            if (systemTenant != null)
            {
                datastore = new DelegatingSystemTenantDatastore(backingDatastore, systemTenant);
            }
            else
            {
                datastore = backingDatastore;
            }

            DefaultMultiTenantManager manager = new DefaultMultiTenantManager(datastore, tenantReference, singleTenantMode);
            MultiTenantContext.manager = manager;
            controller = manager;
            factory = new MultiTenantComponentFactoryImpl(tenantReference, manager, datastore);

            // Initialise the matcher, maybe this will be configurable in future
            matcher = new DefaultMultiTenantMatcher(Arrays.asList(new RequestHeaderMatcher(datastore),
                    new HostnameMatcher(datastore)));

            authorisationProvider = initialiseAuthorisationProvider(props, singleTenantMode);
        }
    }

    /**
     * True if multi-tenant mode is enabled
     *
     * @return True if multi-tenant mode is enabled
     */
    public static boolean isEnabled()
    {
        return manager != null;
    }

    /**
     * Get the system home directory
     *
     * @return The system home directory
     */
    public static File getSystemHome()
    {
        return multiTenantSystemHome;
    }

    /**
     * Get the tenant reference holder
     *
     * @return The tenant reference holder
     */
    public static TenantReference getTenantReference()
    {
        return tenantReference;
    }

    /**
     * Get the MultiTenantManager
     *
     * @return The manager
     */
    public static MultiTenantManager getManager()
    {
        return manager;
    }

    /**
     * Get the MultiTenantLifecycleController
     *
     * @return The controller
     */
    public static MultiTenantLifecycleController getController()
    {
        return controller;
    }

    /**
     * Get the MultiTenantComponentFactory
     *
     * @return The factory
     */
    public static MultiTenantComponentFactory getFactory()
    {
        return factory;
    }

    /**
     * Get the matcher
     *
     * @return The matcher
     */
    public static MultiTenantMatcher getMatcher()
    {
        return matcher;
    }

    /**
     * Get the MultiTenantParser
     *
     * @return The parser
     */
    public static MultiTenantParser getParser()
    {
        return parser;
    }

    /**
     * Set the tenant reference.  Useful only for testing.
     *
     * @param tenantReference The tenant reference to set.
     */
    public static void setTenantReference(final TenantReference tenantReference)
    {
        MultiTenantContext.tenantReference = tenantReference;
    }

    /**
     * Set the MultiTenantManager.  Useful only for testing.
     *
     * @param manager The manager to set.
     */
    public static void setManager(final MultiTenantManager manager)
    {
        MultiTenantContext.manager = manager;
    }

    /**
     * Set the MultiTenantLifecycleController.  Useful only for testing.
     *
     * @param controller The controller to set.
     */
    public static void setController(final MultiTenantLifecycleController controller)
    {
        MultiTenantContext.controller = controller;
    }

    /**
     * Set the component factory.  Useful only for testing.
     *
     * @param factory The factory to set.
     */
    public static void setFactory(final MultiTenantComponentFactory factory)
    {
        MultiTenantContext.factory = factory;
    }

    /**
     * Set the matcher.  Useful only for testing.
     *
     * @param matcher The matcher to set.
     */
    public static void setMatcher(final MultiTenantMatcher matcher)
    {
        MultiTenantContext.matcher = matcher;
    }

    /**
     * Set the parser.  Useful only for testing.
     *
     * @param parser The parser to set
     */
    public static void setParser(final MultiTenantParser parser)
    {
        MultiTenantContext.parser = parser;
    }

    /**
     * Get the system tenant
     *
     * @return The system tenant, or null if there is no system tenant
     */
    public static Tenant getSystemTenant()
    {
        return systemTenant;
    }

    /**
     * Set the system tenant.  Useful only for testing.
     *
     * @param systemTenant The system tenant to set
     */
    public static void setSystemTenant(final Tenant systemTenant)
    {
        MultiTenantContext.systemTenant = systemTenant;
    }

    /**
     * Gets the authorisation provider
     *
     * @return The authorisation provider
     */
    public static AuthorisationProvider getAuthorisationProvider()
    {
        return authorisationProvider;
    }

    /**
     * Sets the authorisation provider.  Useful only for testing.
     *
     * @param authorisationProvider The authorisation provider to set
     */
    public static void setAuthorisationProvider(final AuthorisationProvider authorisationProvider)
    {
        MultiTenantContext.authorisationProvider = authorisationProvider;
    }

    private static Properties loadProperties()
    {
        // Context classloader
        InputStream is = Thread.currentThread().getContextClassLoader().getResourceAsStream(MULTI_TENANT_PROPERTIES_FILE);
        if (is != null)
        {
            log.info("Found " + MULTI_TENANT_PROPERTIES_FILE + " in context classloader");
        }
        else
        {
            // This classloader
            is = MultiTenantContext.class.getClassLoader().getResourceAsStream(MULTI_TENANT_PROPERTIES_FILE);
        }
        if (is != null)
        {
            log.info("Found " + MULTI_TENANT_PROPERTIES_FILE + " in this classloader");
        }
        else
        {
            // Try a file in the working directory
            File file = new File(MULTI_TENANT_PROPERTIES_FILE);
            if (file.exists() && file.isFile() && file.canRead())
            {
                log.info("Found " + MULTI_TENANT_PROPERTIES_FILE + " on filesystem at " + file.getAbsolutePath());
                try
                {
                    is = new FileInputStream(file);
                }
                catch (IOException ioe)
                {
                    log.warn("Error loading multitenant properties file", ioe);
                }
            }
        }
        if (is == null)
        {
            return null;
        }
        Properties properties = new Properties();
        try
        {
            properties.load(is);
        }
        catch (IOException ioe)
        {
            log.warn("Error loading multitenant properties file", ioe);
            return null;
        }
        finally
        {
            IOUtils.closeQuietly(is);
        }
        properties.putAll(System.getProperties());
        return properties;
    }

    @SuppressWarnings ({ "unchecked" })
    private static Map<String, CustomConfigHandler<?>> initHandlers(Properties props)
    {
        Map<String, CustomConfigHandler<?>> handlers = new HashMap<String, CustomConfigHandler<?>>();
        Enumeration<String> names = (Enumeration<String>) props.propertyNames();
        while (names.hasMoreElements())
        {
            String name = names.nextElement();
            if (name.startsWith(HANDLER_PREFIX))
            {
                String handlerName = name.substring(HANDLER_PREFIX.length());
                String className = props.getProperty(name);
                handlers.put(handlerName, loadExtension(className, CustomConfigHandler.class));
            }
        }
        return handlers;
    }

    private static File locateSystemHome(Properties props)
    {
        String homePath = props.getProperty(MULTI_TENANT_SYSTEM_HOME_PROPERTY);
        if (homePath == null)
        {
            log.warn("Multi tenant system home path not set, using current working directory: " + new File(".").getAbsolutePath());
            homePath = DEFAULT_MULTI_TENANT_SYSTEM_HOME;
        }
        File homeDir = new File(homePath);
        if (!homeDir.exists())
        {
            throw new IllegalArgumentException("Multitenant home directory does not exist: " + homeDir.getAbsolutePath());
        }
        if (!homeDir.isDirectory())
        {
            throw new IllegalArgumentException("Multitenant home directory is not a directory: " + homeDir.getAbsolutePath());
        }
        if (!homeDir.canRead())
        {
            throw new IllegalArgumentException("Multitenant home directory is not readable: " + homeDir.getAbsolutePath());
        }
        if (!homeDir.canWrite())
        {
            throw new IllegalArgumentException("Multitenant home directory is not writable: " + homeDir.getAbsolutePath());
        }
        return homeDir;
    }

    private static Tenant getSystemTenant(Properties props)
    {
        String systemTenantProviderClassName = props.getProperty(SYSTEM_TENANT_PROVIDER_KEY);
        if (systemTenantProviderClassName != null)
        {
            SystemTenantProvider provider = loadExtension(systemTenantProviderClassName, SystemTenantProvider.class);
            return provider.getSystemTenant();
        }
        return null;
    }

    private static AuthorisationProvider initialiseAuthorisationProvider(Properties props, boolean singleTenantMode)
    {
        String authorisationProviderClassName = props.getProperty(MULTI_TENANT_AUTHORISATION_PROVIDER_KEY);
        if (authorisationProviderClassName == null || singleTenantMode)
        {
            return new NoPermissionAuthorisationProvider();
        }
        else
        {
            return loadExtension(authorisationProviderClassName, AuthorisationProvider.class);
        }
    }

    private static <T> T loadExtension(String className, Class<T> interfaceClass)
    {
        // First try and load the class from the context classloader
        Class loadedClass = null;
        try
        {
            loadedClass = Class.forName(className, true, Thread.currentThread().getContextClassLoader());
        }
        catch (ClassNotFoundException cfne)
        {
            // Ignore
        }
        if (loadedClass == null)
        {
            // Try this classes class loader
            try
            {
                loadedClass = Class.forName(className);
            }
            catch (ClassNotFoundException cnfe)
            {
                throw new RuntimeException("Couldn't find class for extension", cnfe);
            }
        }
        if (!interfaceClass.isAssignableFrom(loadedClass))
        {
            throw new ClassCastException("Loaded extension (" + className + ") is not of type (" + interfaceClass + ")");
        }
        try
        {
            Class<? extends T> castClass = loadedClass;
            return castClass.newInstance();
        }
        catch (IllegalAccessException iae)
        {
            throw new RuntimeException("Error instantiating extension " + className, iae);
        }
        catch (InstantiationException ie)
        {
            throw new RuntimeException("Error instantiating extension " + className, ie);
        }
    }

}
