package com.atlassian.multitenant;

import java.util.Collection;

/**
 * A map of tenants to components.  This map uses the current tenant returned from TenantReference as the key to
 * retrieve components on.  It supports lazy and eager initialisation, with custom creation and destruction callbacks.
 * <p/>
 * The map supports null components.
 */
public interface MultiTenantComponentMap<T>
{
    /**
     * Get the component for the current tenant
     *
     * @return The component for the current tenant
     * @throws IllegalStateException if there is no tenant currently set, or if there is no creation callback and a
     * component for the current tenant hasn't been added
     */
    T get() throws IllegalStateException;

    /**
     * Get all of the components that this map holds.  This may trigger lazily initialised components to be initialised
     * if they haven't been iniatialised for all tenants.
     *
     * @return A collection of all the components for all tenants
     */
    Collection<T> getAll();

    /**
     * Initialise all of the components for all tenants.  This is intended for use with the {@link
     * LazyLoadStrategy.EAGER_AFTER_STARTUP} strategy.
     */
    void initialiseAll();

    /**
     * Whether or not the component has been initialised for this tenant.
     *
     * @return True if the component has been initialised for this tenant, otherwise false.
     * @throws IllegalStateException if there is no tenant currently set
     */
    boolean isInitialised() throws IllegalStateException;

    /**
     * Add an instance for the current tenant.  This is intended for use when a null creator is being used.
     *
     * @param object The instance to set for the current tenant.
     * @throws IllegalStateException if there is no tenant currently set
     */
    void addInstance(T object) throws IllegalStateException;

    /**
     * Destroy all instances and deregister from the multi tenant manager
     */
    void destroy();

    /**
     * The strategy to follow when there is no instance
     */
    public enum NoTenantStrategy
    {
        /**
         * Fail, throwing an IllegalStateException if no tenant is currently set
         */
        FAIL,

        /**
         * Return the component for the system tenant, if there is a system tenant
         */
        SYSTEM
    }

    /**
     * The strategy to use for loading components
     */
    public enum LazyLoadStrategy
    {
        /**
         * The components are lazy loaded, on first call to get()
         */
        LAZY_LOAD,

        /**
         * Eagerly loaded when a new tenant is started, but not on system start up, the map expects initialiseAll() to be
         * called to load components for tenants already installed.
         */
        EAGER_AFTER_STARTUP,

        /**
         * Eagerly loaded, when the map is first created, and when new tenants are started.
         */
        EAGER_LOAD
    }
}
