package com.atlassian.maven.plugins;

import java.io.File;
import java.util.Iterator;
import java.util.Set;

import org.apache.maven.artifact.manager.WagonConfigurationException;
import org.apache.maven.artifact.manager.WagonManager;
import org.apache.maven.plugin.AbstractMojo;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.settings.Proxy;
import org.apache.maven.settings.Settings;
import org.apache.maven.wagon.ConnectionException;
import org.apache.maven.wagon.UnsupportedProtocolException;
import org.apache.maven.wagon.Wagon;
import org.apache.maven.wagon.WagonException;
import org.apache.maven.wagon.observers.Debug;
import org.apache.maven.wagon.proxy.ProxyInfo;
import org.apache.maven.wagon.repository.Repository;

/**
 * Deploys the given resource (file or directory) using a suitable wagon
 * provider.
 * 
 * @goal upload
 */
public class UploadMojo extends AbstractMojo
{
	/**
	 * Resource to be uploaded. Can be a file or directory. Also supports
	 * wildcards.
	 * 
	 * @see PathParserUtil#toFiles(String)
	 * @parameter expression="${upload.resourceSrc}"
	 * @required
	 */
	private String resourceSrc;

	/**
	 * Path on the server to upload the resource to. If not specified - assumed
	 * to be ".".
	 * 
	 * For instance: * src=dir1/dir2 dest=xyz will create xyz/dir2 on the server *
	 * src=dir1/dir2/* dest=xyz will create xyz/ and put all the content of dir2
	 * there * src=dir1/dir2 will create dir2 on the server with all the dir2
	 * content
	 * 
	 * @parameter expression="${upload.resourceDest}" default-value="."
	 */
	private String resourceDest;

	/**
	 * URL to upload to. Upload will be performed to uploadURL/uploadResource
	 * 
	 * @parameter expression="${upload.url}"
	 * @required
	 */
	private String url;

	/**
	 * ID of the server to upload to. This is used when wagon needs extra
	 * authentication information for instance.
	 * 
	 * @parameter expression="${upload.serverId}"
	 * @required
	 */
	private String serverId;

    /**
     * If true, ignores invalid source resources during execution
     * @parameter expression="${upload.ignoreInvalidResource}" default-value="false"
     */
    private boolean ignoreInvalidResource;

    /**
	 * @component
	 */
	private WagonManager wagonManager;

	/**
	 * The current user system settings for use in Maven.
	 * 
	 * @parameter expression="${settings}"
	 * @required
	 * @readonly
	 */
	private Settings settings;

	public void execute() throws MojoExecutionException
	{
        final Set uploadResources = PathParserUtil.toFiles(resourceSrc);
        if (uploadResources.isEmpty())
        {
            final String message = "Resource " + resourceSrc + " does not match an existing file or directory.";
            if (ignoreInvalidResource)
            {
                getLog().info(message);
                return;
            }
            else
            {
                throw new InvalidResourceException(message);
            }
        }

        if (uploadResources == null || uploadResources.isEmpty())
		{
			throw new MojoExecutionException("The resources to upload are not specified.");
		}

		if (url == null)
		{
			throw new MojoExecutionException("The URL to upload to is missing.");
		}

        upload(uploadResources);
	}

    private void upload(Set uploadResources) throws MojoExecutionException
    {
        final Repository repository = new Repository(serverId, url);
        Debug debug = new Debug();
        getLog().debug("Upload URL: " + url);
        getLog().debug("Upload Server ID: " + serverId);

        try
		{
			final Wagon wagon = wagonManager.getWagon(repository);

            for (Iterator iterator = uploadResources.iterator(); iterator.hasNext();)
            {
                File resource = (File) iterator.next();
                if (resource.isDirectory() && !wagon.supportsDirectoryCopy())
                {
                    throw new MojoExecutionException("Wagon protocol '" + repository.getProtocol()
                            + "' doesn't support directory copying. " + resource + " will fail to upload.");
                }
            }

            try
            {
                wagon.addSessionListener(debug);
                wagon.addTransferListener(debug);

                ProxyInfo proxyInfo = getProxyInfo(settings);
                if (proxyInfo != null)
                {
                    wagon.connect(repository, wagonManager.getAuthenticationInfo(repository.getId()), proxyInfo);
                }
                else
                {
                    wagon.connect(repository, wagonManager.getAuthenticationInfo(repository.getId()));
                }

                for (Iterator iterator = uploadResources.iterator(); iterator.hasNext();)
                {
                    File resource = (File) iterator.next();
                    if (resource.isDirectory())
                        wagon.putDirectory(resource, resourceDest + '/' + resource.getName());
                    else
                        wagon.put(resource, resourceDest + '/' + resource.getName());
                }
            }
            catch (WagonException e)
            {
                throw new MojoExecutionException("Error uploading resource", e);
            }
            finally
            {
                try
                {
                    wagon.disconnect();
                }
                catch (ConnectionException e)
                {
                    getLog().debug("Error disconnecting wagon - ignored", e);
                }
            }
        }
        catch (UnsupportedProtocolException e)
        {
            throw new MojoExecutionException("Unsupported protocol: '" + repository.getProtocol() + "'", e);
        }
        catch (WagonConfigurationException e)
        {
            throw new MojoExecutionException("Unable to configure Wagon: '" + repository.getProtocol() + "'", e);
        }
    }

    /**
	 * Convenience method to map a <code>Proxy</code> object from the user
	 * system settings to a <code>ProxyInfo</code> object.
	 * 
	 * @return a proxyInfo object or null if no active proxy is define in the
	 *         settings.xml
	 */
	private static ProxyInfo getProxyInfo(Settings settings)
	{
		ProxyInfo proxyInfo = null;
		if (settings != null && settings.getActiveProxy() != null)
		{
			Proxy settingsProxy = settings.getActiveProxy();

			proxyInfo = new ProxyInfo();
			proxyInfo.setHost(settingsProxy.getHost());
			proxyInfo.setType(settingsProxy.getProtocol());
			proxyInfo.setPort(settingsProxy.getPort());
			proxyInfo.setNonProxyHosts(settingsProxy.getNonProxyHosts());
			proxyInfo.setUserName(settingsProxy.getUsername());
			proxyInfo.setPassword(settingsProxy.getPassword());
		}

		return proxyInfo;
	}
}