package com.atlassian.distribution.mojo;

import com.atlassian.distribution.DependencyBundle;
import com.atlassian.distribution.ExtraResources;
import com.atlassian.distribution.MavenVersion;
import com.atlassian.distribution.scriptwriter.AbstractScriptWriter;
import com.atlassian.distribution.scriptwriter.FinalMavenBuildCommand;
import com.atlassian.distribution.scriptwriter.PosixScriptWriter;
import com.atlassian.distribution.scriptwriter.WindowsScriptWriter;
import org.apache.commons.io.FileUtils;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.maven.artifact.Artifact;
import org.apache.maven.artifact.factory.ArtifactFactory;
import org.apache.maven.artifact.metadata.ArtifactMetadataSource;
import org.apache.maven.artifact.repository.ArtifactRepository;
import org.apache.maven.artifact.resolver.ArtifactCollector;
import org.apache.maven.artifact.resolver.ArtifactNotFoundException;
import org.apache.maven.artifact.resolver.ArtifactResolutionException;
import org.apache.maven.artifact.resolver.ArtifactResolver;
import org.apache.maven.execution.MavenSession;
import org.apache.maven.plugin.AbstractMojo;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugins.annotations.Component;
import org.apache.maven.plugins.annotations.Mojo;
import org.apache.maven.plugins.annotations.Parameter;
import org.apache.maven.plugins.annotations.ResolutionScope;
import org.apache.maven.project.MavenProject;
import org.apache.maven.project.MavenProjectBuilder;
import org.apache.maven.project.ProjectBuildingException;
import org.apache.maven.project.artifact.InvalidDependencyVersionException;
import org.apache.maven.shared.dependency.tree.DependencyNode;
import org.apache.maven.shared.dependency.tree.DependencyTreeBuilder;
import org.apache.maven.shared.dependency.tree.DependencyTreeBuilderException;
import org.apache.maven.shared.invoker.DefaultInvocationRequest;
import org.apache.maven.shared.invoker.DefaultInvoker;
import org.apache.maven.shared.invoker.InvocationRequest;
import org.apache.maven.shared.invoker.InvocationResult;
import org.apache.maven.shared.invoker.Invoker;
import org.apache.maven.shared.invoker.MavenInvocationException;

import java.io.File;
import java.io.FileFilter;
import java.io.FilenameFilter;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Properties;
import java.util.Queue;
import java.util.Set;

/**
 * Checkout source code of the project, download dependency source jar from maven.atlassian.com, download maven, generate build script
 */
@Mojo(name="source", aggregator = true, requiresProject = true, threadSafe = false,
        requiresDependencyResolution = ResolutionScope.TEST)
public class SourceDistributionMojo extends AbstractMojo{

    /**
     * target directory of maven project
     */
    @Parameter(defaultValue = "${project.build.directory}", readonly = true)
    private File outputDirectory;

    /**
     * Root directory of the local scm repository
     */
    @Parameter(defaultValue = "${user.dir}", property = "source.baseProjectDirectory")
    private File baseProjectDirectory;

    /**
     * By default copy ${user.dir} as project source. Sometimes need to checkout from scm. E.g. confluence has different repo for source and distribution projects
     */
    @Parameter(defaultValue = "false")
    private String checkoutFromScm;

    /**
     * The directory name where the source code will be checkout
     */
    @Parameter(defaultValue = "checkouts")
    private String checkoutDirectoryName;

    /**
     * Works with <code>scmVersionType</code> together to tell the version information of the project
     * If <code>scmVersionType</code> is tag, then it should be a tag name to be check out
     * If <code>scmVersionType</code> is branch, then it should be the branch name to be checkout
     *
     * Usually the plugin can figure out the version information. Use these params when the plugin fails to do that
     */
    @Parameter( property = "scmVersionName")
    private String scmVersionName;

    /**
     * Works with <code>scmVersionName</code> to tell the version information of the project.
     * Possible values are branch or tag
     *
     * Usually the plugin can figure out the version information. Use these params when the plugin fails to do that
     */
    @Parameter( property = "scmVersionType")
    private String scmVersionType;

    @Parameter (defaultValue = "${project}", readonly = true)
    private MavenProject project;

    @Parameter (defaultValue = "${reactorProjects}", readonly = true)
    private List<MavenProject> reactorProjects;

    @Parameter(defaultValue = "${localRepository}", readonly = true)
    protected ArtifactRepository localRepository;

    @Parameter(defaultValue = "${project.remoteArtifactRepositories}", readonly = true)
    protected List remoteRepositories;

    @Deprecated
    @Parameter
    private String[] excludedSubtrees;

    /**
     * The settings file shipped with source code. This plugin provides a default one
     */
    @Parameter
    private String includeSettingsFile;

    /**
     * Specify local maven repo name in generated build script
     */
    @Parameter
    private String localRepoDirectory;

    /**
     * Maven command to build the project. e.g. mvn clean install -DskipTests
     *
     */
    @Parameter
    private FinalMavenBuildCommand[] finalMavenBuildCommands;

    /**
     * The artifacts whose sourcecode will not be shipped to customer
     */
    @Parameter
    private List excludedArtifacts;

    /**
     * List of maven artifacts used for including additional sets of dependencies.
     * "Bundles" are resolved as Maven projects, that is different from normal dependencies.
     * All of "bundle" direct non-test dependencies will be added to dependencySources directory.
     *
     * This can be used for adding sources of bundled within product artifacts to the distribution,
     * as it will add provided scope dependencies of "bundle" as well.
     */
    @Parameter
    private List<DependencyBundle> bundles;

    /**
     * Skip sourcecode checkout, only download sourcecode of dependencies
     */
    @Parameter
    private boolean skipCheckout;

    /**
     * Skip downloading dependency source jars
     */
    @Parameter( defaultValue = "false")
    private boolean skipDownloadingDependencyJars;

    /**
     * Continue if there are dependencies whose source artifact does not exist in maven repo
     * default value: true
     */
    @Parameter( defaultValue = "false")
    private boolean skipUnResolvedSourceJar;

    /**
     * Use Reactor to build the dependency tree
     */
    @Parameter (defaultValue = "true")
    private boolean useReactor;

    /**
     * The project whose code will be checked out. Needed when <code>useReactor</code> is false.
     * e.g. com.atlassian.jira:jira-project
     */
    @Parameter
    private String baseProject;


    private MavenProject baseMavenProject;

    @Component
    protected ArtifactMetadataSource metadataSource;

    @Component
    protected ArtifactResolver resolver;

    @Component
    protected ArtifactFactory artifactFactory;

    @Component
    protected ArtifactResolver artifactResolver;

    @Component
    private MavenSession session;

    @Component
    private DependencyTreeBuilder treeBuilder;

    @Component
    private ArtifactCollector artifactCollector;

    @Component
    private MavenProjectBuilder projectBuilder;

    private Set<String> totalExcludedArtifacts;

    public void execute() throws MojoExecutionException{
        if ( !outputDirectory.exists() ){
            outputDirectory.mkdirs();
        }

        File checkoutDirectory = createCheckoutDirectory();
        initialize();

        if(Boolean.valueOf(checkoutFromScm)) {
            callScmCheckoutPlugin(checkoutDirectory);
        }else {
            copyCurrentRepoAsSource(checkoutDirectory);
        }

        downloadAtlassianDependencySource(checkoutDirectory);

        Set<MavenVersion> mavenVersions = getMavenVersions();

        ExtraResources.copyResources(checkoutDirectory, artifactFactory, resolver, remoteRepositories, localRepository, mavenVersions, getLog());

        generateBuildCommand();

    }

    private Set<MavenVersion> getMavenVersions() throws MojoExecutionException
    {
        final Set<MavenVersion> result = new HashSet<MavenVersion>();
        for(FinalMavenBuildCommand command : finalMavenBuildCommands)
        {
            result.add(command.getMavenVersionEnum());
        }
        return result;
    }

    private void initialize() throws MojoExecutionException {
        if(useReactor)
            this.baseMavenProject = this.reactorProjects.get(0);
        else {
            try{
                final Set<Artifact> artifacts = project.createArtifacts(artifactFactory, null, null);
                for(Artifact artifact : artifacts) {
                    if((artifact.getGroupId() + ":" + artifact.getArtifactId()).equals(baseProject)) {
                        getLog().info("init result: " + formatArtifact(artifact));
                        baseMavenProject = projectBuilder.buildFromRepository(artifact, remoteRepositories, localRepository);
                    }
                }
            }catch(InvalidDependencyVersionException e) {
                throw new MojoExecutionException("Failed to get dependencies for project: " + project, e);
            }catch(ProjectBuildingException e) {
                throw new MojoExecutionException("Failed to build project:" + baseProject, e);
            }
        }
    }

    private File createCheckoutDirectory() throws MojoExecutionException{
        File checkoutDirectory = new File(outputDirectory, this.checkoutDirectoryName);
        if (!checkoutDirectory.exists() && !checkoutDirectory.mkdirs())
        {
            throw new MojoExecutionException("Could not create directory " + checkoutDirectory.getAbsolutePath());
        }
        return checkoutDirectory;
    }

    /**
     * Instead of checking out source from VCS, just copy the current repo
     * Does not work with confluence because it has a separated SVN repo for all distribution projects
     *
     * @param checkoutDirectory target directory of source code, e.g. target/checkouts/${artifactId}
     * @throws MojoExecutionException
     */
    private void copyCurrentRepoAsSource(File checkoutDirectory) throws MojoExecutionException
    {
        try {
            getLog().info("Copying from " + baseProjectDirectory + " to " + checkoutDirectory);
            File projectDir = new File(checkoutDirectory, baseMavenProject.getArtifactId());
            projectDir.mkdir();
            FileUtils.copyDirectory(baseProjectDirectory, projectDir, new FileFilter(){
                @Override
                public boolean accept(File file) {
                    return !isSCMFile(file.getName()) && !(file.isDirectory() && file.getName().equals("target"));
                }
            }, true);
        } catch(IOException e) {
            throw new MojoExecutionException("Failed to copy source dir", e);
        }
    }

    private void callScmCheckoutPlugin(File checkoutDirectory) throws MojoExecutionException {

        final String projectOutputDirectory = checkoutDirectory.getAbsolutePath() + File.separator + this.baseMavenProject.getArtifactId();
        if(!this.baseMavenProject.getScm().getConnection().startsWith("scm:svn") && (StringUtils.isEmpty(scmVersionName) || StringUtils.isEmpty(scmVersionType))) {
            // Detect the scm version info if it is not in the plugin configuration
            getLog().info("Project is SVN: " + this.baseMavenProject.getScm().getConnection().startsWith("scm:svn"));
            getLog().info("scmVersionName is: " + scmVersionName);
            getLog().info("scmVersionType is: " + scmVersionType);

            initSCMVersion();
        }
        getLog().info("Trying to checkout " + this.baseMavenProject.getScm().getConnection() + ", " + scmVersionType + " "
                + scmVersionName + " to directory " + projectOutputDirectory);

        if(skipCheckout) {
            getLog().info("Skipping Source code checkout");
            return;
        }
        final Invoker invoker = new DefaultInvoker(){};
        invoker.setLocalRepositoryDirectory(new File(session.getLocalRepository().getBasedir()));

        final InvocationRequest request = new DefaultInvocationRequest();
        request.setGoals(Arrays.asList("org.apache.maven.plugins:maven-scm-plugin:1.9:checkout") );
        request.setPomFile(baseMavenProject.getFile());

        Properties arguments = new Properties(session.getUserProperties());
        request.setProperties(arguments);

        arguments.put("checkoutDirectory", projectOutputDirectory);
        if(!this.baseMavenProject.getScm().getConnection().startsWith("scm:svn")){
            arguments.put("scmVersion", scmVersionName);
            arguments.put("scmVersionType", scmVersionType);
        }
        arguments.put("connectionUrl", this.baseMavenProject.getScm().getConnection());

        final InvocationResult result;
        try {
            result = invoker.execute( request );
        } catch (MavenInvocationException e) {
            throw new MojoExecutionException(e.getMessage(), e.getCause());
        }

        if ( result.getExitCode() != 0 ) {
            throw new IllegalStateException( "Failed to checkout. See messages above. " );
        }

        try {
            deleteSCMFiles(new File(projectOutputDirectory));
        }catch(Exception e) {
            throw new MojoExecutionException("Failed to delete scm files", e);
        }
        getLog().info("Checkout complete to " + projectOutputDirectory);
    }

    private void deleteSCMFiles(File projectOutputDirectory) throws IOException {
        File[] scmFiles = projectOutputDirectory.listFiles(new FilenameFilter() {
            @Override
            public boolean accept(File dir, String name) {
                return isSCMFile(name);
            }
        });
        for(File scmFile : scmFiles) {
            if(scmFile.exists()) {
                if(scmFile.isDirectory()) {
                    FileUtils.deleteDirectory(scmFile);
                } else {
                    scmFile.delete();
                }
            }
        }
    }

    private boolean isSCMFile(String fileName) {
        return ".git".equals(fileName) || ".gitignore".equals(fileName) || ".hg".equals(fileName) || ".svn".equals(fileName);
    }

    private void downloadAtlassianDependencySource(File checkoutDirectory) throws MojoExecutionException
    {
        if(skipDownloadingDependencyJars) {
            getLog().info("Skip downloading dependency jars");
            return;
        }
        getLog().info("Begin to download source jars of dependencies from maven.atlassian.com");
        checkoutDirectory = new File(checkoutDirectory, "dependencySources");
        checkoutDirectory.mkdir();

        final Set<Artifact> atlassianDependencies = new HashSet<Artifact>();
        final List<MavenProject> projectList = new ArrayList<MavenProject>();
        if (useReactor) {
            projectList.addAll(reactorProjects);
        } else {
            projectList.add(project);
        }
        if (bundles != null) {
            projectList.addAll(buildBundledDependenciesProjectList());
        }
        for (final MavenProject project : projectList){
            final Set<Artifact> artifacts = retrieveDependenciesFromTree(project, localRepository, this.excludedSubtrees == null? new HashSet<String>():new HashSet(Arrays.asList(this.excludedSubtrees)));
            for (final Artifact artifact : artifacts){
                getLog().debug("Adding artifact to list:" + formatArtifact(artifact));

                Artifact artifactWithClassifier = artifactFactory.createArtifactWithClassifier(artifact.getGroupId(),
                        artifact.getArtifactId(), artifact.getVersion(), artifact.getType(), "sources");
                artifactWithClassifier.setScope(artifact.getScope());
                atlassianDependencies.add(artifactWithClassifier);
            }
        }
        resolveAndCopyArtifacts(atlassianDependencies, checkoutDirectory);

    }

    private List<MavenProject> buildBundledDependenciesProjectList() {
        List<MavenProject> list = new ArrayList<MavenProject>();

        for (DependencyBundle bundle : bundles) {
            try {
                Artifact artifact = artifactFactory.createArtifactWithClassifier(
                        bundle.getGroupId(), bundle.getArtifactId(), bundle.getVersion(),
                        bundle.getType(), bundle.getClassifier());
                artifactResolver.resolve(artifact, remoteRepositories, localRepository);
                list.add(projectBuilder.buildFromRepository(artifact, remoteRepositories, localRepository));
            } catch (ProjectBuildingException e) {
                throw new RuntimeException(e);
            } catch (ArtifactNotFoundException e) {
                throw new RuntimeException(e);
            } catch (ArtifactResolutionException e) {
                throw new RuntimeException(e);
            }
        }

        return list;
    }

    private boolean shouldDownloadSource(Artifact artifact) throws MojoExecutionException {
        return artifact.getGroupId().startsWith("com.atlassian") && (!artifact.getScope().equals("test"))
                && !getTotalExcludedSingleArtifacts().contains(artifact.getGroupId() + ":" + artifact.getArtifactId()) && !artifact.getType().equals("pom") && !isIncludedInReactor(artifact);
    }

    private boolean isIncludedInReactor(Artifact artifact) throws MojoExecutionException
    {
        for(MavenProject project : this.reactorProjects) {
            if(project.getGroupId().equals(artifact.getGroupId()) && project.getArtifactId().equals(artifact.getArtifactId()))
            {
                getLog().debug("Ignoring " + formatArtifact(artifact) + " because it is sub module of base project");
                return true;
            }
        }
        if(!useReactor) {
            try {
                MavenProject parent = projectBuilder.buildFromRepository(artifact, remoteRepositories, localRepository);
                while(parent != null) {
                    if((parent.getGroupId() + ":" + parent.getArtifactId()).equals(baseProject))
                    {
                        getLog().debug("Ignoring " + formatArtifact(artifact) + " because it is sub module of base project");
                        return true;
                    }
                    parent = parent.getParent();
                }
            } catch (ProjectBuildingException e) {
                throw new MojoExecutionException("Failed to build project:", e);
            }
        }

        return false;
    }

    private Set<String> getTotalExcludedSingleArtifacts() throws MojoExecutionException {
        if(this.totalExcludedArtifacts == null) {
            totalExcludedArtifacts = new HashSet<String>();
            try {
                InputStream is = this.getClass().getResourceAsStream("/excludedArtifacts.txt");
                totalExcludedArtifacts.addAll(IOUtils.readLines(is));
                is.close();
                if(this.excludedArtifacts != null)
                    totalExcludedArtifacts.addAll(this.excludedArtifacts);
            }catch(IOException ie) {
                throw new MojoExecutionException("Cannot read excluded artifacts:", ie);
            }
        }
        return totalExcludedArtifacts;
    }

    private Set<Artifact> retrieveDependenciesFromTree(final MavenProject project, final ArtifactRepository localRepository, final Set<String> ignoredTrees) throws MojoExecutionException {

        DependencyNode rootNode = null;
        try
        {
            rootNode = treeBuilder.buildDependencyTree(project, localRepository,
                    artifactFactory, metadataSource, null, artifactCollector);
        }
        catch (DependencyTreeBuilderException e)
        {
            throw new IllegalArgumentException("The project couldn't generate a dependency tree", e);
        }

        final Set<Artifact> depArtifacts = new HashSet<Artifact>();
        final Queue<DependencyNode> nodesToVisit = new LinkedList<DependencyNode>(rootNode.getChildren());
        DependencyNode actualNode = null;
        while ( (actualNode = nodesToVisit.poll()) != null)
        {
            if (!isNodeIdPresentOnSet(ignoredTrees, actualNode))
            {
                // need to look at the scope now. Because some 'test' artifacts may be added into the artifacts and prevented the same artifact of others scope being added
                // The equals method of artifact only compare GAV and classifier. Not scope.
                if (actualNode.getState() == DependencyNode.INCLUDED && !actualNode.getArtifact().getScope().equals("test"))
                {
                    depArtifacts.add(actualNode.getArtifact());
                }else {
                    getLog().debug("Ignoring artifact: " + formatArtifact(actualNode.getArtifact()));
                }
                nodesToVisit.addAll(actualNode.getChildren());
            }else {
                getLog().debug("Ignore sub tree: " + formatArtifact(actualNode.getArtifact()));
            }
        }
        return depArtifacts;
    }

    private boolean isNodeIdPresentOnSet(Set<String> artifacts, DependencyNode node)
    {
        Artifact artifact = node.getArtifact();
        String prefix = artifact.getGroupId() + ":" + artifact.getArtifactId();
        return artifacts.contains(prefix);
    }

    private String formatArtifact(Artifact artifact) {
        return String.format("%s:%s:%s", artifact.getGroupId(), artifact.getArtifactId(), artifact.getVersion());
    }

    private Set<Artifact> resolveAndCopyArtifacts(Set<Artifact> artifacts, File checkoutDirectory) throws MojoExecutionException {
        List<Artifact> artifactsCannotBeFound = new ArrayList<Artifact>();
        for (Artifact artifact : artifacts){
            if(!shouldDownloadSource(artifact))
                continue;

            try{
                getLog().debug("Resolving artifact:" + formatArtifact(artifact));
                resolver.resolve(artifact, remoteRepositories, localRepository);

                getLog().debug("Copying " + artifact.getFile().getPath() + " to " + checkoutDirectory.getPath());
                FileUtils.copyFileToDirectory(artifact.getFile(), checkoutDirectory);
            } catch (ArtifactResolutionException e){
                artifactsCannotBeFound.add(artifact);
            }catch (ArtifactNotFoundException e){
                artifactsCannotBeFound.add(artifact);
            } catch (IOException e) {
                artifactsCannotBeFound.add(artifact);
            }
       }
        if(artifactsCannotBeFound.size() > 0) {
            final StringBuilder message =  new StringBuilder("\n\nArtifacts whose source jar cannot be downloaded from maven.atlassian.com( ");
            message.append(artifactsCannotBeFound.size()).append(" artifacts )\n");

            for(Artifact atf : artifactsCannotBeFound) {
                message.append("\t - ").append(formatArtifact(atf));
                message.append(":").append(atf.getType()).append("\n");
            }

            message.append("\n");
            getLog().error(message.toString());
            if(!skipUnResolvedSourceJar)
                throw new MojoExecutionException("There are artifacts that don't have the sources available for download. See message above for more information. ");
        }
        return artifacts;
    }

    private void generateBuildCommand() throws MojoExecutionException
    {
        getLog().info("Generating build command");
        File directory = new File(outputDirectory, this.checkoutDirectoryName);
        AbstractScriptWriter posixWriter = new PosixScriptWriter(directory, "build.sh", false);
        AbstractScriptWriter winWriter = new WindowsScriptWriter(directory, "build.bat", false);

        posixWriter.addSettingsLine("settings.xml");
        winWriter.addSettingsLine("settings.xml");

        if(StringUtils.isNotBlank(localRepoDirectory))
        {
            posixWriter.addLocalRepoLine(localRepoDirectory);
            winWriter.addLocalRepoLine(localRepoDirectory);
        }

        if(finalMavenBuildCommands != null && finalMavenBuildCommands.length > 0) {
            for(FinalMavenBuildCommand command : finalMavenBuildCommands){
                if(StringUtils.isEmpty(command.getMavenVersion()))
                {
                    getLog().warn("mavenVersion is not specified in finalMavenBuildCommand, will default to use maven 3.0.5");
                }
                posixWriter.addCommand(command.getMavenVersionEnum().getMavenRunnerCmd() + "." + MavenVersion.MAVEN_CMD_POSIX, command.getCmdArgs(), true, true);
                winWriter.addCommand(command.getMavenVersionEnum().getMavenRunnerCmd() + "." + MavenVersion.MAVEN_CMD_WIN, command.getCmdArgs(), true, true);
            }
        } else {
            throw new MojoExecutionException("You need to configure the command to build your project");
        }

        try {
            if(this.includeSettingsFile != null && this.includeSettingsFile.length() > 0) {
                getLog().info("re write settings file: " + this.includeSettingsFile);
                posixWriter.includeSettingsFile(this.includeSettingsFile, "settings.xml");
            }

            posixWriter.writeScript();
            winWriter.writeScript();
        }catch(IOException e) {
            throw new MojoExecutionException("Failed to generate build script: ", e);
        }
    }

    /**
     * Try to detect the scm version/branch info automatically if it's not in the plugin configuration
     */
    private void initSCMVersion() {
        getLog().info("Trying to detect the SCM information automatically");
        if(this.baseMavenProject.getVersion().contains("SNAPSHOT")) {
            getLog().info("Base project is snapshot, default to master branch");
            this.scmVersionName = "master";
            this.scmVersionType = "branch";
        }else {
            String pomTag = baseMavenProject.getOriginalModel().getScm().getTag();
            String pomPropertiesTag = baseMavenProject.getProperties().getProperty("atlassian.release.scm.tag.prefix");
            scmVersionType = "tag";
            if(pomTag != null && !pomTag.equals("HEAD")){
                getLog().info("Using <tag> element inside <scm> element as tag");
                scmVersionName = pomTag;
            }else if(pomPropertiesTag != null){
                getLog().info("Using \"${atlassian.release.scm.tag.prefix}-version\" as the tag");
                scmVersionName = pomPropertiesTag + "-" + project.getVersion();
            }else{
                getLog().info("Defaulting to ${artifactId}-${version} as the tag");
                scmVersionName = baseMavenProject.getArtifactId() + "-" + baseMavenProject.getVersion();
            }
        }
    }
}
