package com.atlassian.distribution;

import org.apache.commons.io.FileUtils;
import org.apache.commons.io.IOUtils;
import org.apache.maven.artifact.Artifact;
import org.apache.maven.artifact.factory.ArtifactFactory;
import org.apache.maven.artifact.repository.ArtifactRepository;
import org.apache.maven.artifact.resolver.ArtifactNotFoundException;
import org.apache.maven.artifact.resolver.ArtifactResolutionException;
import org.apache.maven.artifact.resolver.ArtifactResolver;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugin.logging.Log;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.FileWriter;
import java.io.IOException;
import java.io.InputStream;
import java.util.Enumeration;
import java.util.List;
import java.util.Set;
import java.util.zip.ZipEntry;
import java.util.zip.ZipFile;

/**
 *  Download maven and generate related files, including settings.xml, mvn3xx.sh/bat
 */
public class ExtraResources
{

    private static final String[] MAVEN_ALIAS_SCRIPTS = new String[]{"mvn3.sh", "mvn3.bat"};

    public static void copyResources(File outputDirectory, ArtifactFactory artifactFactory, ArtifactResolver resolver,
                                     List remoteRepositories, ArtifactRepository localRepository, Set<MavenVersion> mavenVersions, Log log) throws MojoExecutionException
    {
        copySettingsFiles(outputDirectory);
        for(MavenVersion mavenVersion : mavenVersions)
        {
            downloadMaven(artifactFactory, resolver, remoteRepositories, localRepository,
                    outputDirectory, mavenVersion.getVersionString(), "maven3", log);
        }
        copyMavenAliasScripts(outputDirectory, mavenVersions);

    }
    private static void copySettingsFiles(File outputDirectory) throws MojoExecutionException
    {
        try
        {
            InputStream settingsFileStream = ExtraResources.class.getResourceAsStream("/settings.xml");

            File settingsFile = new File(outputDirectory, "settings.xml");
            FileWriter settingsWriter = new FileWriter(settingsFile);

            IOUtils.copy(settingsFileStream, settingsWriter);

            settingsFileStream.close();
            settingsWriter.flush();
            settingsWriter.close();
        }
        catch(IOException e)
        {
            throw new MojoExecutionException("Failed to write settings.xml ", e);
        }
    }

    private static void downloadMaven(ArtifactFactory artifactFactory, ArtifactResolver resolver, List remoteRepositories, ArtifactRepository localRepository,
                                      File checkoutDirectory, String mavenVersion, String mavenDirectoryName, Log log)
            throws MojoExecutionException
    {
        Artifact mavenArtifact = artifactFactory.createArtifactWithClassifier("org.apache.maven", "apache-maven", mavenVersion, "zip", "bin");
        try
        {
            resolver.resolve(mavenArtifact, remoteRepositories, localRepository);
            File mavenBinary = mavenArtifact.getFile();
            File mavenDir = new File(checkoutDirectory, mavenDirectoryName);
            if(mavenDir.exists() && mavenDir.isDirectory())
            {
                try
                {
                    FileUtils.deleteDirectory(new File(mavenDir, "apache-maven-" + mavenVersion));
                }
                catch(IOException e)
                {
                    throw new MojoExecutionException("Failed to delete existing maven dir: " + mavenDir.getAbsolutePath(), e);
                }
            }
            mavenDir.mkdirs();

            ZipHelper zipHelper = new ZipHelper(log);
            zipHelper.unzipArchive(mavenBinary, mavenDir, true);
        }
        catch(ArtifactResolutionException e)
        {
            throw new MojoExecutionException("Failed to resolve maven binary artifact: ", e);
        }
        catch(ArtifactNotFoundException e)
        {
            throw new MojoExecutionException("Failed to find maven binary artifact: ", e);
        }

    }

    /**
     * Writes out alias scripts mvn3xx that invoke maven from the bundled versions that come with the source
     * distribution
     */
    private static void copyMavenAliasScripts(File checkoutDirectory, Set<MavenVersion> mavenVersions)
            throws MojoExecutionException
    {
        try
        {
            for(MavenVersion mavenVersion : mavenVersions)
            {
                for (String scriptName : MAVEN_ALIAS_SCRIPTS)
                {
                    InputStream scriptInputStream = ExtraResources.class.getResourceAsStream("/" + scriptName);

                    File scriptFile = new File(checkoutDirectory, mavenVersion.getMavenRunnerCmd() + scriptName.substring(scriptName.lastIndexOf(".")));
                    FileWriter scriptWriter = new FileWriter(scriptFile);
                    final String scriptContent = IOUtils.toString(scriptInputStream);
                    final String updatedScript = scriptContent.replaceAll("\\{mavenVersion\\}", mavenVersion.getVersionString());
                    IOUtils.write(updatedScript, scriptWriter);

                    scriptFile.setExecutable(true, false);

                    scriptInputStream.close();
                    scriptWriter.flush();
                    scriptWriter.close();
                }
            }
        }
        catch(IOException e)
        {
            throw new MojoExecutionException("Couldn't write mvn alias scripts: ", e);
        }
    }

    public static class ZipHelper
    {
        Log log;

        public ZipHelper(Log log)
        {
            this.log = log;
        }

        public void unzipArchive(File archive, File outputDir, boolean makeFilesExecutable)
        {
            try
            {
                ZipFile zipfile = new ZipFile(archive);
                for (Enumeration e = zipfile.entries(); e.hasMoreElements(); )
                {
                    ZipEntry entry = (ZipEntry) e.nextElement();
                    unzipEntry(zipfile, entry, outputDir, makeFilesExecutable);
                }
            }
            catch (Exception e)
            {
                log.error("Error while extracting file " + archive, e);
            }
        }

        private void unzipEntry(ZipFile zipfile, ZipEntry entry, File outputDir, boolean makeFilesExecutable) throws IOException
        {

            if (entry.isDirectory())
            {
                createDir(new File(outputDir, entry.getName()));
                return;
            }

            File outputFile = new File(outputDir, entry.getName());
            if (!outputFile.getParentFile().exists())
            {
                createDir(outputFile.getParentFile());
            }

            log.debug("Extracting: " + entry);
            BufferedInputStream inputStream = new BufferedInputStream(zipfile.getInputStream(entry));
            BufferedOutputStream outputStream = new BufferedOutputStream(new FileOutputStream(outputFile));

            try
            {
                IOUtils.copy(inputStream, outputStream);
                if (makeFilesExecutable)
                {
                    outputFile.setExecutable(true, false);
                }
            }
            finally
            {
                outputStream.close();
                inputStream.close();
            }
        }

        private void createDir(File dir)
        {
            log.debug("Creating dir "+dir.getName());
            if(!dir.mkdirs()) throw new RuntimeException("Can not create dir " + dir);
        }
    }
}
