package com.atlassian.maven.plugins.sourcerelease.mojos;

//Portions Copyright Apache Software Foundation

import org.apache.commons.lang.StringUtils;
import org.apache.maven.artifact.Artifact;
import org.apache.maven.artifact.factory.ArtifactFactory;
import org.apache.maven.artifact.metadata.ArtifactMetadataRetrievalException;
import org.apache.maven.artifact.metadata.ArtifactMetadataSource;
import org.apache.maven.artifact.metadata.ResolutionGroup;
import org.apache.maven.artifact.repository.ArtifactRepository;
import org.apache.maven.artifact.resolver.ArtifactNotFoundException;
import org.apache.maven.artifact.resolver.ArtifactResolutionException;
import org.apache.maven.artifact.resolver.ArtifactResolutionResult;
import org.apache.maven.artifact.resolver.ArtifactResolver;
import org.apache.maven.model.Model;
import org.apache.maven.model.Scm;
import org.apache.maven.model.io.xpp3.MavenXpp3Writer;
import org.apache.maven.plugin.AbstractMojo;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.project.InvalidProjectModelException;
import org.apache.maven.project.MavenProject;
import org.apache.maven.project.MavenProjectBuilder;
import org.apache.maven.project.ProjectBuildingException;
import org.apache.maven.settings.Server;
import org.apache.maven.settings.Settings;
import org.apache.maven.scm.ScmException;
import org.apache.maven.scm.ScmFileSet;
import org.apache.maven.scm.ScmResult;
import org.apache.maven.scm.manager.ScmManager;
import org.apache.maven.scm.provider.ScmProviderRepositoryWithHost;
import org.apache.maven.scm.repository.ScmRepository;
import org.apache.maven.scm.repository.ScmRepositoryException;

import com.atlassian.maven.plugins.scm.ScmInfo;

import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Set;

/**
 * Goal checkes out dependencies from source control for a given groupId mask
 *
 * @goal source
 * 
 * @phase package
 */
public class SourceMojo
    extends AbstractMojo
{    
    /**
     * @component
     */
    private ArtifactMetadataSource metadataSource;

    /**
     * @component
     */
    private MavenProjectBuilder projectBuilder;

    /**
     * @component
     */
    private ArtifactResolver resolver;
	
    /**
     * @parameter expression="${component.org.apache.maven.scm.manager.ScmManager}"
     * @required
     * @readonly
     */
    private ScmManager manager;
    
    /**
     * @parameter expression="${settings}"
     * @required
     * @readonly
     */
    private Settings settings;
	
    /**
     * @parameter default-value="${localRepository}"
     * @required
     * @readonly
     */
    private ArtifactRepository localRepository;

    /**
     * @parameter default-value="${project.remoteArtifactRepositories}"
     * @required
     * @readonly
     */
    private List remoteRepositories;
	
    /**
     * @parameter expression="${project}"
     * @required
     */
    private MavenProject project;
    
    /**
     * Location of the file.
     * @parameter expression="${project.build.directory}"
     * @required
     */
    private File outputDirectory;
    
    /**
     * Upload a POM for this artifact.  Will generate a default POM if none is
     * supplied with the pomFile argument.
     *
     * @parameter expression="${sourcedistribution.generatePom}" default-value="true"
     */
    private boolean generatePom;
    
    /**
     * GroupId of the generated pom
     * @parameter
     * @required
     */
    private String groupId;
    
    /**
     * ArtifactId of the generated source release pom
     * @parameter
     * @required
     */
    private String artifactId;
    
    /**
     * Version of the generated pom
     * @parameter
     * @required
     */
    private String version;
    
    /**
     * Product Name of the generated pom (eg. Atlassian Confluence)
     * @parameter
     * @required
     */
    private String productName;
    
    /**
     * Directory name of the location in the output directory where the source release project will be generated
     * @parameter default-value="checkouts"
     */
    private String checkoutDirectoryName;
    
    /**
     * Mask of groupId to include in source release (eg. "com.atlassian." or "org.apache.maven.plugins." )
     * @parameter
     * @required
     */
    private String groupIdMask;
    
    /**
     * List of artifacts to exclude from the source release. Format "groupId:artifactId"
     * @parameter
     */
    private List exclusions;
    
    /**
     * Skips the execution
     * @parameter expression="${sourcedistribution.skip}" default-value="false"
     */
    private boolean skip;
    
    /**
     * All projects listed in the reactor
     * @parameter expression="${reactorProjects}"
     * @required
     */  
    private List reactorProjects;
    
    public void execute()
        throws MojoExecutionException
    {
        if (skip)
        {
            getLog().info("Skipping source distribution execution");
            return;
        }
        
    	final File checkoutDirectory = createCheckoutDirectory();
        final Set projects = getProjectsForArtifacts(getReactorProjectDependencies());
        
        if (projects == null)
        {
            throw new MojoExecutionException("Could not resolve dependent projects");
        }
        
        //Remove undesirables
    	removeExcludedArtifacts(projects);
        
        //Add root reactor project
        MavenProject rootReactorProject = (MavenProject)reactorProjects.get(0);
        projects.add(rootReactorProject);
        getLog().info("Root reactor project is " + rootReactorProject.getGroupId() + ":" + rootReactorProject.getArtifactId());
    	
    	List modules = checkoutProjects(projects, checkoutDirectory);
    	
    	if (generatePom && modules.size() != 0)
    	{
            writeSourceDistributionPom(modules, checkoutDirectory);
    	}
    }
    
    private Set getReactorProjectDependencies() throws MojoExecutionException
    {
        Set dependencies = new HashSet();
        for (Iterator it = reactorProjects.iterator(); it.hasNext();) {
            MavenProject  mavenProject = (MavenProject)it.next();
            Set projectDependencies = mavenProject.getDependencyArtifacts();
            dependencies.addAll(projectDependencies);
        }
        return resolveTransitiveArtifacts(dependencies);
    }
    
    private Set getProjectsForArtifacts(Set artifacts) throws MojoExecutionException
    {
        Set projects = new HashSet();
        for (Iterator it = artifacts.iterator(); it.hasNext();) {
            Artifact artifact = (Artifact)it.next();
            MavenProject currentProject = getProjectForArtifact(artifact);
            
            if (artifact.getGroupId().startsWith(groupIdMask))
            {
                projects.add(currentProject);
            }
        }
        return projects;
    }
    
    private void removeExcludedArtifacts(Set projects)
    	throws MojoExecutionException
    {
        List exclusionPairs = getExclusions();
        List projectList = new ArrayList(projects);
        for (Iterator iter = projectList.iterator(); iter.hasNext();) 
        {
            MavenProject currentProject = (MavenProject) iter.next();

            if (currentProject == null)
            {
                getLog().info("Project in project list is null");
                return;
            }

            if (matchesExclusion(currentProject, exclusionPairs))
            {
                projects.remove(currentProject);
            }
        }
    }
    
    private List getExclusions()
    {
        List excludes = new ArrayList(exclusions);
        List reactorExclusions = reactorProjects.subList(1, reactorProjects.size());
        for (Iterator it = reactorExclusions.iterator(); it.hasNext();) {
            MavenProject mavenProject = (MavenProject)it.next();
            excludes.add(mavenProject.getGroupId() + ":" + mavenProject.getArtifactId());
        }
        return excludes;
    }
    
    private boolean matchesExclusion(MavenProject project, List exclusionPairs)
        throws MojoExecutionException
    {
        for (Iterator it = exclusionPairs.iterator(); it.hasNext();) {
            String exclusion = (String)it.next();
            String[] parts = StringUtils.split(exclusion, ':');
            if (parts.length == 2)
            {
                String currentGroupId = project.getGroupId();
                String currentArtifactId = project.getArtifactId();

                if (parts[0].equals(currentGroupId) && parts[1].equals(currentArtifactId))
                {
                    return true;
                }
            }
            else
            {
                throw new MojoExecutionException("Exclusion is not in the format of groupId:artifactId. Value:" + exclusion);
            }
        }
    	return false;
    }
    
    /*
     * Checks out artifacts and returns a list of their artifactIds
     */
    private List checkoutProjects(Set projects, File checkoutDirectory)
    	throws MojoExecutionException
    {
        getLog().info("Will checkout the following artifacts:");
        for (Iterator iter = projects.iterator(); iter.hasNext();) {
            MavenProject currentProject = (MavenProject) iter.next();
            getLog().info(currentProject.getGroupId() + ":" + currentProject.getArtifactId());
        }
    	
    	List modules = new ArrayList();
    	for (Iterator iter = projects.iterator(); iter.hasNext();) {
            MavenProject currentProject = (MavenProject) iter.next();
            doCheckout(currentProject, checkoutDirectory);
            if (generatePom)
            {
                String projectArtifactId = currentProject.getArtifactId();
                if (modules.contains(projectArtifactId))
                {
                    throw new MojoExecutionException("Cannot have two modules checked out with the same directory name " + projectArtifactId);
                }
                modules.add(currentProject.getArtifactId());
            }
	}
    	
    	return modules;
    }
    
    /*
     * Writes out the generated project pom
     */
    private void writeSourceDistributionPom(List modules, File checkoutDirectory)
    	throws MojoExecutionException
    {
    	Exception ex = null;
    	
        Model model = new Model();
        model.setModelVersion( "4.0.0" );
        model.setGroupId( groupId );
        model.setArtifactId( artifactId );
        model.setVersion( version );
        model.setName( productName + " Source Release"  );
        model.setPackaging( "pom" );
        model.setDescription( "Source for " + productName );
        model.setModules(modules);
        
        FileWriter writer;
        try
        {
            writer = new FileWriter(new File(checkoutDirectory, "pom.xml"));
            new MavenXpp3Writer().write(writer, model);
            writer.flush();
            writer.close();
        }
        catch (IOException e)
        {
            throw new MojoExecutionException("Could not write source release pom", ex);
        }
    }
    
    /*
     * Checks out a MavenProject to the given checkoutDirectory location
     */
    private void doCheckout(MavenProject project, File checkoutDirectory)
    	throws MojoExecutionException
    {
        Scm scm = project.getScm();
        if (scm != null && scm.getConnection() != null)
        {
            ScmResult result = null;
            scm.getConnection();
            ScmRepository scmRepository = null;
            try
            {
                scmRepository = getScmRepository(scm.getConnection());
                getLog().info("Checking out " + project.getGroupId() + ":" + project.getArtifactId());
                result = getScmManager().export(scmRepository, new ScmFileSet(checkoutDirectory), project.getArtifactId());
                checkResult(result);
            }
            catch (ScmException e)
            {
                throw new MojoExecutionException(e.getMessage(), e);
            }
        }
        else
        {
            getLog().warn("Skipping " + project.getArtifactId() + " as it did not have a valid <scm> tag");
        }
    }
    
    /*
     * Returns the MavenProject for a given Artifact
     */
    private MavenProject getProjectForArtifact(Artifact artifact)
    	throws MojoExecutionException
    {
        try
        {
            return projectBuilder.buildFromRepository(artifact, remoteRepositories, localRepository);
        }
        catch (InvalidProjectModelException e)
        {
            return null;
        }
        catch (ProjectBuildingException e)
        {
            throw new MojoExecutionException(e.getMessage(), e);
        }
    }
    
    /*
     * Resolves transitive dependencies for a set of given artifacts
     * Returns a Set of all artifacts resolved
     */
    private Set resolveTransitiveArtifacts(Set artifacts)
    	throws MojoExecutionException
    {
        try 
        {
            ArtifactResolutionResult result = resolver.resolveTransitively(artifacts, project.getArtifact(), 
                localRepository, remoteRepositories, metadataSource, null);
            return result.getArtifacts();
        } 
        catch (ArtifactResolutionException e) 
        {
            throw new MojoExecutionException(e.getMessage(), e);
        } 
        catch (ArtifactNotFoundException e)
        {
            throw new MojoExecutionException(e.getMessage(), e);
        }
    }
    
    private void checkResult( ScmResult result )
    	throws MojoExecutionException
    {
        if ( !result.isSuccess() )
        {
            getLog().error( "Provider message:" );
            getLog().error( result.getProviderMessage() == null ? "" : result.getProviderMessage() );
            getLog().error( "Command output:" );
            getLog().error( result.getCommandOutput() == null ? "" : result.getCommandOutput() );
            throw new MojoExecutionException(
                "Command failed. " + StringUtils.defaultString( result.getProviderMessage() ) );
        }
    }
    
    private File createCheckoutDirectory()
    	throws MojoExecutionException
    {
    	String dirName = checkoutDirectoryName;

    	File checkoutDirectory = new File(outputDirectory, dirName);
        if (!checkoutDirectory.exists() && !checkoutDirectory.mkdirs())
        {
            throw new MojoExecutionException("Could not create directory " + checkoutDirectory.getAbsolutePath());
        }
    	return checkoutDirectory;
    }
    
    private ScmManager getScmManager()
    {
        return manager;
    }
    
    private ScmRepository getScmRepository(String connectionUrl)
    	throws ScmException
    {
        ScmRepository repository;

        try
        {
            repository = getScmManager().makeScmRepository( connectionUrl );

            if ( repository.getProviderRepository() instanceof ScmProviderRepositoryWithHost )
            {
                ScmProviderRepositoryWithHost repo = (ScmProviderRepositoryWithHost) repository.getProviderRepository();
                ScmInfo info = loadInfosFromSettings( repo );

                if ( !StringUtils.isEmpty( info.getUsername() ) )
                {
                    repo.setUser( info.getUsername() );
                }

                if ( !StringUtils.isEmpty( info.getPassword() ) )
                {
                    repo.setPassword( info.getPassword() );
                }

                if ( !StringUtils.isEmpty( info.getPrivateKey()) )
                {
                    repo.setPrivateKey( info.getPrivateKey() );
                }

                if ( !StringUtils.isEmpty( info.getPassphrase() ) )
                {
                    repo.setPassphrase( info.getPassphrase() );
                }
            }
        }
        catch ( ScmRepositoryException e )
        {
            if ( !e.getValidationMessages().isEmpty() )
            {
                for ( Iterator i = e.getValidationMessages().iterator(); i.hasNext(); )
                {
                    String message = (String) i.next();
                    getLog().error( message );
                }
            }
            throw new ScmException( "Can't load the scm provider.", e );
        }
        catch ( Exception e )
        {
            throw new ScmException( "Can't load the scm provider.", e );
        }

        return repository;
    }
    
    private ScmInfo loadInfosFromSettings( ScmProviderRepositoryWithHost repo )
    {
    	ScmInfo info = new ScmInfo();
        String host = repo.getHost();
        int port = repo.getPort();

        if ( port > 0 )
        {
            host += ":" + port;
        }
        
        Server server = this.settings.getServer( host );
        if (server != null)
        {
            info.setUsername(this.settings.getServer( host ).getUsername());
            info.setPassword(this.settings.getServer( host ).getPassword());
            info.setPrivateKey(this.settings.getServer( host ).getPrivateKey());
            info.setPassphrase(this.settings.getServer( host ).getPassphrase());
        }
        
        return info;
    }
}
