package com.atlassian.maven.plugins.pdk;

import com.atlassian.core.util.FileUtils;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugin.MojoFailureException;
import org.apache.maven.project.MavenProject;
import org.apache.maven.model.Dependency;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.zip.ZipEntry;
import java.util.zip.ZipFile;

/**
 * @goal generate-war
 * @requiresDependencyResolution runtime
 * @description Create Confluence War
 */
public class GenerateWarMojo extends BasePdkMojo
{
    /**
     * The maven project.
     *
     * @parameter expression="${project}"
     */
    private MavenProject project;

    /**
     * @parameter expression="${tempBaseDir}" default-value="${project.build.directory}/atlassian";
     */
    private String tempBaseDir;

    /**
     * @parameter expression="${overwrite}" default-value="false"
     */
    private String overwrite;

    /**
     * @parameter expression="${warInstallPath}"
     * @required
     */
    private String warInstallPath;
    /**
     * @parameter expression="${warFileName}"
     */
    private String warFileName;

    private static final String ATLASSIAN = "exploded";
    private static final String WEB_INF = "WEB-INF";
    private static final String LIB = "lib";

    public void execute() throws MojoExecutionException, MojoFailureException
    {
        LinkedList dependencies = findDependencies(project, "war");
        if (dependencies.size() > 0)
        {
            Dependency dependency = (Dependency) dependencies.getFirst();

            File src = getDependencyAsFile(dependency.getGroupId(), dependency.getArtifactId(), dependency.getVersion(), dependency.getType());

            File warTempPath = new File(tempBaseDir + File.separator + dependency.getArtifactId() + File.separator + dependency.getVersion());
            if (warFileName == null)
            {
                warFileName = dependency.getArtifactId() + "-" + dependency.getVersion() + ".war";
            }

            getLog().info("Path to the Atlassian WAR is = " + warInstallPath);

            if (overwrite.equals("true"))
            {
                //TODO: Nuke please
                /**
                 nukeFolders();
                 **/
            }

            //TODO: Take overwrite condition when done with nuking method
            if (!(new File(warTempPath, warFileName).exists()) || overwrite.equals("true"))
            {
                if ((src.exists()))
                {
                    getLog().info("Preparing Atlassian WAR");

                    unJarFiles(src, warTempPath);
                    installFunctestRPC(warTempPath);
                    createConfluenceWAR(warTempPath);
                } else
                {
                    getLog().warn("An Atlassian WAR not specified as a dependency. The Atlasian Test WAR cannot be created.");
                }
            } else
            {
                getLog().info("The Atlassian Test WAR has already been built.");
            }
        } else
        {
            getLog().warn("Dependency not found... Skipped Downloading Atlassian WAR.");
            getLog().warn("Please add a dependency of the form: \n" +
                    "<dependency>\n" +
                    "    <groupId>com.atlassian.something</groupId>\n" +
                    "    <artifactId>something-webapp</artifactId>\n" +
                    "    <version>[version number]</version>\n" +
                    "    <type>war</type>\n" +
                    "    <scope>provided</scope>\n" +
                    "</dependency>");
        }
    }

    private Dependency findDependency(MavenProject project, String groupId, String artifactId)
    {
        Iterator i = project.getDependencies().iterator();
        while (i.hasNext())
        {
            Dependency a = (Dependency) i.next();
            if (a.getGroupId().equals(groupId) && a.getArtifactId().equals(artifactId))
                return a;
        }

        return null;
    }

    private LinkedList findDependencies(MavenProject project, String type)
    {
        LinkedList dependencies = new LinkedList();
        Iterator i = project.getDependencies().iterator();
        while (i.hasNext())
        {
            Dependency a = (Dependency) i.next();
            if (a.getType().equals(type))
                dependencies.add(a);
        }

        return dependencies;
    }

    private void createConfluenceWAR(File explodedWar)
    {
        File target = new File(warInstallPath, warFileName);
        String zipDirPath = ATLASSIAN;
        getLog().info("Creating Atlassian WAR ");
        try
        {
            FileUtils.createZipFile(new File(explodedWar, zipDirPath), target);
        }
        catch (Exception e)
        {
            getLog().error(e);

        }
    }

    private void installFunctestRPC(File installDir)
    {
        String groupId = "com.atlassian.confluence.extra";
        String artifactId = "functestrpc";

        Dependency dependency = findDependency(project, groupId, artifactId);
        if (dependency != null)
        {
            File dest = new File(installDir, ATLASSIAN + File.separator + WEB_INF + File.separator + LIB);
            getLog().info("Installing functestrpc");
            copyFiles(getDependencyAsFile(groupId, artifactId, dependency.getVersion(), dependency.getType()), dest);
        }
    }

    /**
     * safely performs a recursive delete on a directory
     *
     * @param dir The directory to delete.
     * @return True if the directory was deleted.
     */
    private static boolean deleteDir(File dir)
    {
        if (dir == null)
        {
            return false;
        }

        // to see if this directory is actually a symbolic link to a directory,
        // we want to get its canonical path - that is, we follow the link to
        // the file it's actually linked to
        File candir;
        try
        {
            candir = dir.getCanonicalFile();
        }
        catch (IOException e)
        {
            return false;
        }

        // a symbolic link has a different canonical path than its actual path,
        // unless it's a link to itself
        if (!candir.equals(dir.getAbsoluteFile()))
        {
            // this file is a symbolic link, and there's no reason for us to
            // follow it, because then we might be deleting something outside of
            // the directory we were told to delete
            return false;
        }

        // now we go through all of the files and subdirectories in the
        // directory and delete them one by one
        File[] files = candir.listFiles();
        if (files != null)
        {
            for (int i = 0; i < files.length; i++)
            {
                File file = files[i];

                // in case this directory is actually a symbolic link, or it's
                // empty, we want to try to delete the link before we try
                // anything
                boolean deleted = !file.delete();
                if (deleted)
                {
                    // deleting the file failed, so maybe it's a non-empty
                    // directory
                    if (file.isDirectory()) deleteDir(file);

                    // otherwise, there's nothing else we can do
                }
            }
        }

        // now that we tried to clear the directory out, we can try to delete it
        // again
        return dir.delete();
    }

    private void unJarFiles(File source, File target)
    {
        try
        {
            final int BUFFER = 2048;
            String fileName = ATLASSIAN;
            //String fileName = source.getName();
            //int prefixSuffixPoint = fileName.lastIndexOf('.');
            //fileName = fileName.substring(0, prefixSuffixPoint);
            target = new File(target, fileName);
            deleteDir(target);
            target.mkdirs();
            BufferedOutputStream dest;
            BufferedInputStream is;
            ZipEntry entry;
            ZipFile jarfile = new ZipFile(source);
            Enumeration e = jarfile.entries();
            while (e.hasMoreElements())
            {
                entry = (ZipEntry) e.nextElement();
                if (!entry.isDirectory())
                {
                    is = new BufferedInputStream
                            (jarfile.getInputStream(entry));
                    int count;
                    byte data[] = new byte[BUFFER];
                    FileOutputStream fos = new
                            FileOutputStream(new File(target, entry.getName()));
                    dest = new
                            BufferedOutputStream(fos, BUFFER);
                    while ((count = is.read(data, 0, BUFFER)) != -1)
                    {
                        dest.write(data, 0, count);
                    }
                    dest.flush();
                    dest.close();
                    is.close();
                } else
                {
                    File file = new File(target, entry.getName());
                    file.mkdirs();
                }
            }
            jarfile.close();
        }
        catch (IOException e)
        {
            getLog().error(e);
        }
    }

    protected File getDependencyAsFile(String groupId, String artifactId, String version, String type)
    {
        String archetypeArtifactPath = localRepository.getBasedir() + "/"
                + localRepository.pathOf(getArtifact(groupId, artifactId, version, type));
        return new File(archetypeArtifactPath);
    }

    public void setTempBaseDir(String tempBaseDir)
    {
        this.tempBaseDir = tempBaseDir;
    }

    public void setDependencies(ArrayList dependencies)
    {
    }

    public MavenProject getProject()
    {
        return project;
    }

    public void setProject(MavenProject project)
    {
        this.project = project;
    }

    public String getWarInstallPath()
    {
        return warInstallPath;
    }

    public void setWarInstallPath(String warInstallPath)
    {
        this.warInstallPath = warInstallPath;
    }

    public String getWarFileName()
    {
        return warFileName;
    }

    public void setWarFileName(String warFileName)
    {
        this.warFileName = warFileName;
    }

    public void setOverwrite(String overwrite)
    {
        this.overwrite = overwrite;
    }
}

