package com.atlassian.maven.plugins.pdk;

import org.apache.commons.httpclient.Header;
import org.apache.commons.httpclient.HttpClient;
import org.apache.commons.httpclient.DefaultMethodRetryHandler;
import org.apache.commons.httpclient.HttpState;
import org.apache.commons.httpclient.HttpStatus;
import org.apache.commons.httpclient.HttpException;
import org.apache.commons.httpclient.UsernamePasswordCredentials;
import org.apache.commons.httpclient.methods.GetMethod;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugin.MojoFailureException;

import java.net.ConnectException;
import java.io.IOException;
import java.net.URI;
import java.net.URISyntaxException;

/**
 * Abstract base class for plugins which interact with plugins on Atlassian
 * servers (Confluence and JIRA).
 */
public abstract class BasePluginServerMojo extends BasePdkMojo {

    /**
     * @parameter default-value="${atlassian.pdk.server.username}"
     * @required
     */
    protected String username;

    /**
     * @parameter expression="${atlassian.pdk.server.password}"
     * @required
     */
    protected String password;

    /**
     * @parameter expression="${atlassian.pdk.server.url}"
     * @required
     */
    protected String serverUrl;

    /**
     * @parameter expression="${atlassian.plugin.key}"
     * @required
     */
    protected String pluginKey;

    protected void init( String user, String pass, String confUrl, String pluginKey ) {

        this.username = user;
        this.password = pass;
        this.serverUrl = confUrl;
        this.pluginKey = pluginKey;
    }

    public String getPluginKey() {

        return pluginKey;
    }

    public String getServerUrl() {

        return serverUrl;
    }

    public String getUsername() {

        return username;
    }

    public String getPassword() {

        return password;
    }

    protected void init( String user, String pass, String confUrl ) {

        this.username = user;
        this.password = pass;
        this.serverUrl = confUrl;
    }

    public void execute() throws MojoFailureException, MojoExecutionException {

        if ( checkProperties() )
            return;

        // Create an instance of HttpClient.
        HttpClient client = new HttpClient();

        // Create a method instance.
        String url = getActionUrl();

        GetMethod method = new GetMethod( url );

        // bypass anti xsrf protection
        method.setRequestHeader("X-Atlassian-Token", "no-check");

        // Provide custom retry handler is necessary
        DefaultMethodRetryHandler retryhandler = new DefaultMethodRetryHandler();
        retryhandler.setRequestSentRetryEnabled( false );
        retryhandler.setRetryCount( 3 );
        method.setMethodRetryHandler( retryhandler );

        try {
            // Execute the method.
            getLog().info( getTitle() + ": Connecting to server: " + serverUrl );
            int status = client.executeMethod(method.getHostConfiguration(),  method , getHttpState());

            if ( status == HttpStatus.SC_MOVED_TEMPORARILY ) {
                Header location = method.getResponseHeader( "Location" );
                if ( location == null ) {
                    throw new MojoExecutionException( getTitle() + ": Execution failed[" + status
                            + "]: Redirecing to an unknown location" );
                } else if ( location.getValue().indexOf( "/login.action" ) >= 0 )
                    throw new MojoExecutionException( getTitle() + ": Execution failed[" + status
                            + "]: This is likely due to a non-existent or underpriviledged user: " + username );
                else
                    throw new MojoExecutionException( getTitle() + ": Execution failed[" + status
                            + "]: Redirecting to " + location.getValue() );
            } else if ( status != HttpStatus.SC_OK ) {

                throw new MojoExecutionException( getTitle() + ": Method failed: " + method.getStatusLine() );
            } else {
                getLog().info( getTitle() + ": Completed successfully" );
            }
        } catch ( HttpException e ) {
            getLog().error( getTitle() + ": Fatal protocol violation: " + e.getMessage(), e );
        } catch ( ConnectException e ) {
            getLog().error( getTitle() + ": Unable to connect to '" + serverUrl + "': " + e.getMessage() );
            // getLog().debug(e);
        } catch ( IOException e ) {
            getLog().error( getTitle() + ": Fatal transport error: " + e.getMessage(), e );
        } finally {
            // Release the connection.
            method.releaseConnection();
        }
    }

    protected abstract String getTitle();

    protected boolean checkProperties() {

        // Check I have the bits I need
        if ( serverUrl == null || serverUrl.equals( "" ) ) {
            getLog().error( "Please specify the 'atlassian.pdk.server.url' property." );
            return true;
        }
        if ( username == null || username.equals( "" ) ) {
            getLog().error( "Please specify the 'atlassian.pdk.server.username' property." );
            return true;
        }
        if ( password == null || password.equals( "" ) ) {
            getLog().error( "Please specify the 'atlassian.pdk.server.password' property." );
            return true;
        }
        if ( pluginKey == null || pluginKey.equals( "" ) ) {
            getLog().error( "Please specify the 'atlassian.plugin.key' property." );
            return true;
        }
        return false;
    }

    protected String getActionUrl() {

        return serverUrl + "/admin/plugins.action?os_username=" + urlEncode( username ) + "&os_password="
                + password + "&pluginKey=" + urlEncode( pluginKey ) + "&mode=" + urlEncode( getMode() );
    }

    protected HttpState getHttpState() {
        HttpState httpState = new HttpState();
        try {
            httpState.setCredentials(null, new URI(serverUrl).getHost(), new UsernamePasswordCredentials(username, password));
        } catch (URISyntaxException e) {
            throw new RuntimeException("Bad serverUrl:" + serverUrl, e);
        }
        httpState.setAuthenticationPreemptive(true);
        return httpState;
    }

    protected abstract String getMode();
}
