package com.atlassian.maven.plugins.pdk;

import org.apache.commons.httpclient.Header;
import org.apache.commons.httpclient.HttpClient;
import org.apache.commons.httpclient.HttpException;
import org.apache.commons.httpclient.HttpStatus;
import org.apache.commons.httpclient.methods.MultipartPostMethod;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugin.MojoFailureException;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.net.ConnectException;

/**
 * @goal install
 * @execute goal="uninstall"
 * @description Install a plugin into Confluence
 */
public class InstallPluginMojo extends BasePluginServerMojo {

    private static final String UPLOAD = "upload";

    private static final String COPY = "copy";

    /**
     * @parameter expression="${atlassian.pdk.install.method}"
     *            default-value="upload"
     * @required
     */
    protected String installMethod = UPLOAD;

    /**
     * @parameter expression="${atlassian.pdk.server.home}/plugins"
     */
    protected File pluginsHome;

    /**
     * @parameter expression="${project.build.directory}/${project.build.finalName}.jar"
     * @required
     */
    protected File pluginFile;

    protected String getTitle() {

        return "Install Plugin";
    }

    protected boolean checkProperties() {

        if ( installMethod == null
                || ( !UPLOAD.equalsIgnoreCase( installMethod ) && !COPY.equalsIgnoreCase( installMethod ) ) ) {
            getLog().error(
                    "Please specify either '" + UPLOAD + "' or '" + COPY
                            + "' for the 'atlassian.pdk.install.method' property." );
            return true;
        }

        if ( COPY.equals( installMethod ) && pluginsHome == null ) {
            getLog().error( "Please specify the 'atlassian.pdk.server.home' property." );
            return true;
        }

        if ( pluginFile == null ) {
            getLog().error( "The plugin file could not be found." );
            return true;
        } else if ( !pluginFile.exists() || !pluginFile.isFile() ) {
            getLog().error( "The required plugin file does not exist: " + pluginFile.getAbsolutePath() );
            return true;
        }

        return super.checkProperties();
    }

    public void execute() throws MojoFailureException, MojoExecutionException {

        if ( checkProperties() )
            return;

//        // Uninstall Plugin
//        UninstallPluginMojo up = new UninstallPluginMojo();
//        up.init( username, password, serverUrl, pluginKey, installMethod, pluginsHome );
//        up.execute();

        if ( COPY.equals( installMethod ) ) {
            // Copy Plugin to plugin Home Folder
            copyFiles( pluginFile, pluginsHome );
            getLog().info( getTitle() + ": Copied the plugin to '" + pluginsHome + "'" );

            // Rescan for Plugins
            RescanPluginsMojo rp = new RescanPluginsMojo();
            rp.init( username, password, serverUrl, pluginKey );
            rp.execute();
        } else if ( UPLOAD.equals( installMethod ) ) {
            uploadFile();
        }
    }

    private void uploadFile() throws MojoExecutionException {

        // Create an instance of HttpClient.
        HttpClient client = new HttpClient();

        // Create a method instance.
        String url = serverUrl + "/admin/uploadplugin.action?os_username=" + urlEncode( username )
                + "&os_password=" + urlEncode( password );

        MultipartPostMethod filePost = new MultipartPostMethod( url );
        filePost.setFollowRedirects( false );

        try {
            filePost.addParameter( "file_0", pluginFile.getName(), pluginFile );
            client.setConnectionTimeout( 5000 );
            // Execute the method.

            getLog().info( getTitle() + ": Uploading '" + pluginFile.getName() + "' to server: " + serverUrl );
            int status = client.executeMethod( filePost );

            if ( status == HttpStatus.SC_MOVED_TEMPORARILY ) {
                Header location = filePost.getResponseHeader( "Location" );
                if ( location == null ) {
                    throw new MojoExecutionException( getTitle() + ": Upload failed[" + status
                            + "]: Redirecing to an unknown location" );
                } else if ( location.getValue().indexOf( "/login.action" ) >= 0 )
                    throw new MojoExecutionException( getTitle() + ": Upload failed[" + status
                            + "]: This is likely due to a non-existent or underpriviledged user: " + username );
                else if ( location.getValue().indexOf( "/plugins.action" ) < 0 )
                    throw new MojoExecutionException( getTitle() + ": Upload failed[" + status
                            + "]: Redirecting to " + location.getValue() );
            } else if ( status != HttpStatus.SC_OK ) {

                throw new MojoExecutionException( getTitle() + ": Upload failed[" + status + "]: "
                        + HttpStatus.getStatusText( status ) );
            }
            getLog().info( getTitle() + ": Completed successfully[" + status + "]." );
        } catch ( ConnectException e ) {
            getLog().error( getTitle() + ": Unable to connect to '" + serverUrl + "': " + e.getMessage() );
            // getLog().debug(e);
        } catch ( FileNotFoundException e ) {
            getLog().error( getTitle() + ": Unable to find file to upload: " + e.getMessage() );
        } catch ( HttpException e ) {
            // TODO Auto-generated catch block
            e.printStackTrace();
        } catch ( IOException e ) {
            // TODO Auto-generated catch block
            e.printStackTrace();
        } finally {
            filePost.releaseConnection();
        }
    }

    protected String getMode() {

        return null;
    }
}
