package com.atlassian.maven.plugins.updater;

import java.io.BufferedInputStream;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.ConnectException;
import java.net.HttpURLConnection;
import java.net.MalformedURLException;
import java.net.URL;
import java.net.UnknownHostException;
import java.nio.charset.StandardCharsets;
import java.util.function.Function;

import org.apache.commons.io.IOUtils;
import org.codehaus.plexus.logging.AbstractLogEnabled;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.annotations.VisibleForTesting;

/** Implements an SdkResource for the Atlassian Marketplace. */
public class MarketplaceSdkResource extends AbstractLogEnabled implements SdkResource {

    private static final String MARKETPLACE_API_URL_ROOT = "https://marketplace.atlassian.com/rest/1/";
    private static final String SDK_APP_KEY_PREFIX = "atlassian-plugin-sdk-";

    private final ObjectMapper mapper;

    public MarketplaceSdkResource() {
        mapper = new ObjectMapper();
    }

    @Override
    public String getLatestSdkVersion(SdkPackageType packageType) {
        if (!packageType.isObsolete()) {
            MpacData.Version version = getLatestSdkVersionDetails(packageType);
            if (version != null) {
                return version.getVersionNumber();
            }
        }
        return "";
    }

    @Override
    public File downloadSdk(SdkPackageType packageType, String version) {
        if (!packageType.isObsolete()) {
            String appSoftwareId = getAppSoftwareId(packageType);
            MpacData.Version sdkVersion = getAppSoftwareVersion(appSoftwareId, version);
            return downloadSdkVersion(sdkVersion, packageType, version);
        }
        return null;
    }

    @Override
    public File downloadLatestSdk(SdkPackageType packageType) {
        if (!packageType.isObsolete()) {
            String appSoftwareId = getAppSoftwareId(packageType);
            MpacData.Version sdkVersion = getAppSoftwareLatestVersion(appSoftwareId);
            return downloadSdkVersion(sdkVersion, packageType, "LATEST");
        }
        return null;
    }

    private File downloadSdkVersion(MpacData.Version sdkVersion, SdkPackageType packageType, String versionNum) {
        if (sdkVersion == null
                || sdkVersion.getVersionNumber() == null
                || sdkVersion.getFrameworkDetails() == null
                || sdkVersion.getFrameworkDetails().getAttributes() == null
                || sdkVersion.getFrameworkDetails().getAttributes().getBinaryUrl() == null) {
            throw new UpdaterException("Couldn't find SDK version for " + packageType.key()
                    + " on marketplace with version " + versionNum);
        }

        File sdkDownloadTempFile;
        HttpURLConnection conn = null;
        try {
            sdkDownloadTempFile =
                    File.createTempFile(SDK_APP_KEY_PREFIX + sdkVersion.getVersionNumber(), packageType.getSuffix());

            URL url;
            try {
                url = new URL(sdkVersion.getFrameworkDetails().getAttributes().getBinaryUrl());
            } catch (MalformedURLException e) {
                throw new UpdaterException(e);
            }
            conn = (HttpURLConnection) url.openConnection();

            try (InputStream inputStream = conn.getInputStream()) {
                copyResponseStreamToFile(inputStream, sdkDownloadTempFile);
            }
        } catch (IOException e) {
            throw new UpdaterException(e);
        } finally {
            if (conn != null) {
                conn.disconnect();
            }
        }
        return sdkDownloadTempFile;
    }

    private void copyResponseStreamToFile(InputStream stream, File file) {
        try (FileOutputStream fos = new FileOutputStream(file)) {
            IOUtils.copy(stream, fos);
        } catch (IOException ioe) {
            throw new UpdaterException(ioe);
        }
    }

    private MpacData.Version getLatestSdkVersionDetails(SdkPackageType packageType) {
        String appSoftwareId = getAppSoftwareId(packageType);
        return appSoftwareId != null ? getAppSoftwareLatestVersion(appSoftwareId) : null;
    }

    @VisibleForTesting
    String parseAppSoftwareIdResponse(final String json) {
        try {
            MpacData.AppKey[] response = mapper.readValue(json, MpacData.AppKey[].class);
            return response.length > 0 ? response[0].getAppSoftwareId() : null;
        } catch (Exception e) {
            throw new UpdaterException(e);
        }
    }

    private String getAppSoftwareId(SdkPackageType packageType) {
        return doGetMarketplaceApi(
                String.format(
                        "internal/app-software/app-key/%s%s?hosting=datacenter", SDK_APP_KEY_PREFIX, packageType.key()),
                this::parseAppSoftwareIdResponse);
    }

    @VisibleForTesting
    MpacData.Version parseAppSoftwareLatestVersionResponse(final String json) {
        try {
            MpacData.AppSoftwareVersion response = mapper.readValue(json, MpacData.AppSoftwareVersion.class);
            return response.getVersions().length > 0 ? response.getVersions()[0] : null;
        } catch (Exception e) {
            throw new UpdaterException(e);
        }
    }

    private MpacData.Version getAppSoftwareLatestVersion(String appSoftwareId) {
        return doGetMarketplaceApi(
                String.format("app-software/%s/versions?limit=1", appSoftwareId),
                this::parseAppSoftwareLatestVersionResponse);
    }

    @VisibleForTesting
    MpacData.Version parseAppSoftwareVersionResponse(final String json) {
        try {
            return mapper.readValue(json, MpacData.Version.class);
        } catch (Exception e) {
            throw new UpdaterException(e);
        }
    }

    private MpacData.Version getAppSoftwareVersion(String appSoftwareId, String version) {
        return doGetMarketplaceApi(
                String.format("internal/app-software/%s/version-number/%s", appSoftwareId, version),
                this::parseAppSoftwareVersionResponse);
    }

    private <T> T doGetMarketplaceApi(String uri, Function<String, T> transform) {
        URL url;
        try {
            url = new URL(MARKETPLACE_API_URL_ROOT + uri);
        } catch (MalformedURLException e) {
            throw new UpdaterException(e);
        }
        HttpURLConnection conn = null;
        try {
            conn = (HttpURLConnection) url.openConnection();
            try (InputStream jsonStream = new BufferedInputStream(conn.getInputStream())) {
                String json = IOUtils.toString(jsonStream, StandardCharsets.UTF_8);
                return transform.apply(json);
            }
        } catch (UnknownHostException e) {
            this.getLogger().info("Unknown host " + url.getHost());
        } catch (ConnectException e) {
            this.getLogger().info("Fail to connect to host " + url.getHost());
        } catch (FileNotFoundException e) {
            this.getLogger().info("Not found " + uri);
        } catch (Exception e) {
            throw new UpdaterException(e);
        } finally {
            if (conn != null) {
                conn.disconnect();
            }
        }
        return null;
    }
}
