package com.atlassian.maven.plugins.amps.pdk;

import java.io.File;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.util.Base64;
import java.util.Optional;
import java.util.concurrent.TimeUnit;

import org.apache.http.Header;
import org.apache.http.HttpHeaders;
import org.apache.http.HttpMessage;
import org.apache.http.HttpResponse;
import org.apache.http.HttpStatus;
import org.apache.http.auth.AuthScope;
import org.apache.http.auth.UsernamePasswordCredentials;
import org.apache.http.client.CredentialsProvider;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpDelete;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpHead;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.entity.ContentType;
import org.apache.http.entity.mime.MultipartEntityBuilder;
import org.apache.http.entity.mime.content.FileBody;
import org.apache.http.entity.mime.content.StringBody;
import org.apache.http.impl.client.BasicCredentialsProvider;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClients;
import org.apache.http.message.BasicHeader;
import org.apache.http.util.EntityUtils;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugin.logging.Log;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;

import com.atlassian.maven.plugins.amps.InstallParams;

public class PluginInstaller {

    public static final String PENDING_TASK_JSON = "application/vnd.atl.plugins.pending-task+json";

    private final Log log;
    private final ObjectMapper objectMapper;

    public PluginInstaller(Log log) {
        this.log = log;
        this.objectMapper = new ObjectMapper();
    }

    <T extends HttpMessage> T addAuthHeader(T message, InstallParams installParams) {
        String authString = installParams.getUsername() + ":" + installParams.getPassword();
        String auth = Base64.getEncoder().encodeToString(authString.getBytes(StandardCharsets.US_ASCII));
        message.addHeader(new BasicHeader(HttpHeaders.AUTHORIZATION, auth));
        return message;
    }

    private static String getHeaderValue(String name, HttpResponse response) {
        return Optional.ofNullable(response.getFirstHeader(name))
                .map(Header::getValue)
                .orElse(null);
    }

    private static String getLocation(HttpResponse response) {
        return getHeaderValue(HttpHeaders.LOCATION, response);
    }

    private static String getContentType(HttpResponse response) {
        return getHeaderValue(HttpHeaders.CONTENT_TYPE, response);
    }

    private String getUpmToken(CloseableHttpClient httpClient, InstallParams installParams) throws IOException {
        HttpHead head = addAuthHeader(new HttpHead(installParams.getUpmUrl()), installParams);
        head.setHeader(HttpHeaders.ACCEPT, "*/*");
        HttpResponse response = httpClient.execute(head);
        return getHeaderValue("upm-token", response);
    }

    private CloseableHttpClient createHttpClient(String username, String password) {
        CredentialsProvider credsProvider = new BasicCredentialsProvider();
        credsProvider.setCredentials(AuthScope.ANY, new UsernamePasswordCredentials(username, password));
        RequestConfig requestConfig = RequestConfig.custom()
                .setConnectTimeout(5000)
                .setSocketTimeout(5000)
                .build();
        return HttpClients.custom()
                .setDefaultCredentialsProvider(credsProvider)
                .setDefaultRequestConfig(requestConfig)
                .evictIdleConnections(30, TimeUnit.SECONDS)
                .build();
    }

    public void installPlugin(InstallParams installParams) throws MojoExecutionException {
        File pluginFile = installParams.getPluginFile();

        try (CloseableHttpClient httpClient =
                createHttpClient(installParams.getUsername(), installParams.getPassword())) {
            String token = getUpmToken(httpClient, installParams);
            if (token == null) {
                log.error(getTitle() + ": Couldn't get the token from upm.");
                return;
            }
            HttpPost uploadPost =
                    addAuthHeader(new HttpPost(installParams.getUpmUrl() + "?token=" + token), installParams);
            uploadPost.setHeader("X-Atlassian-Token", "no-check");
            MultipartEntityBuilder builder = MultipartEntityBuilder.create();
            builder.addPart(
                    "plugin", new FileBody(pluginFile, ContentType.APPLICATION_OCTET_STREAM, pluginFile.getName()));
            String signature = installParams.getSignature();
            if (signature != null) {
                builder.addPart(
                        "signature",
                        new StringBody("{\"signature\":\"" + signature + "\"}", ContentType.APPLICATION_JSON));
            } else {
                log.info("No signature provided : Signature check must be disabled on the server.");
            }
            uploadPost.setEntity(builder.build());
            try (CloseableHttpResponse response = httpClient.execute(uploadPost)) {
                int statusCode = response.getStatusLine().getStatusCode();
                String responseBody = EntityUtils.toString(response.getEntity(), StandardCharsets.UTF_8);

                switch (statusCode) {
                    case HttpStatus.SC_OK:
                        log.info(getTitle() + ": Completed successfully[" + statusCode + "].");
                        break;
                    case HttpStatus.SC_ACCEPTED:
                        String location = getLocation(response);
                        waitForCompletion(httpClient, location, installParams);
                        break;
                    default:
                        throw new MojoExecutionException(
                                getTitle() + " : Upload failed --- " + statusCode + ": " + responseBody);
                }
            }
        } catch (AsynchronousTaskException | IOException e) {
            throw new MojoExecutionException(e);
        }
    }

    private void waitForCompletion(CloseableHttpClient httpClient, String taskLocation, InstallParams installParams)
            throws AsynchronousTaskException {
        if (taskLocation == null) {
            throw new AsynchronousTaskException("No location provided!");
        }
        for (int i = 0; i < 30; i++) {
            HttpGet get = addAuthHeader(new HttpGet(taskLocation), installParams);
            get.setHeader(HttpHeaders.ACCEPT, PENDING_TASK_JSON);
            try (CloseableHttpResponse response = httpClient.execute(get)) {
                int status = response.getStatusLine().getStatusCode();
                String contentType = getContentType(response);

                String body = EntityUtils.toString(response.getEntity());
                if (isAdditionalTaskRequested(status, contentType)) {
                    JsonNode entity = readJson(body);
                    executeNextTask(httpClient, entity, installParams);
                    break;
                }
                if (status != 200) {
                    break;
                }
                if (isTaskErrorRepresentation(contentType)) {
                    throw new AsynchronousTaskException("Task error: " + body);
                }
                if (isTaskDone(contentType)) {
                    break;
                }
            } catch (IOException ioe) {
                throw new AsynchronousTaskException("Error while waiting for installation completion : ", ioe);
            }
            try {
                Thread.sleep(1000);
            } catch (InterruptedException e) {
                log.info("Interrupted!");
            }
        }
    }

    private JsonNode readJson(String payload) throws JsonProcessingException {
        return objectMapper.readTree(payload);
    }

    private static boolean isAdditionalTaskRequested(int status, String contentType) {
        return status == 202 && contentType != null && contentType.endsWith("next-task+json");
    }

    private static boolean isTaskErrorRepresentation(String contentType) {
        return contentType != null && contentType.endsWith("err+json");
    }

    private static boolean isTaskDone(String contentType) {
        return contentType != null && contentType.endsWith("complete+json");
    }

    private void executeNextTask(CloseableHttpClient httpClient, JsonNode object, InstallParams installParams)
            throws AsynchronousTaskException {
        try {
            JsonNode status = object.get("status");
            if (status.has("nextTaskPostUri")) {
                String uri = status.get("nextTaskPostUri").asText();
                HttpPost post = new HttpPost(uri);
                try (CloseableHttpResponse response = httpClient.execute(post)) {
                    Header location = response.getFirstHeader("Location");
                    if (location != null) {
                        waitForCompletion(httpClient, location.getValue(), installParams);
                    }
                }
            } else if (status.has("cleanupDeleteUri")) {
                String uri = status.get("cleanupDeleteUri").asText();
                HttpDelete delete = new HttpDelete(uri);
                try (CloseableHttpResponse response = httpClient.execute(delete)) {
                    // No further action needed
                }
            } else {
                throw new AsynchronousTaskException(
                        "status should contain nextTaskPostUri or cleanupDeleteUri, but was: " + status);
            }
        } catch (IOException e) {
            throw new AsynchronousTaskException("Error executing next task", e);
        }
    }

    protected String getTitle() {
        return "Install Plugin";
    }

    private static class AsynchronousTaskException extends Exception {

        public AsynchronousTaskException(String s, Throwable cause) {
            super(s, cause);
        }

        public AsynchronousTaskException(String s) {
            super(s);
        }
    }
}
