package com.atlassian.maven.plugins.amps.minifier.strategies.googleclosure;

import static com.google.javascript.jscomp.CheckLevel.OFF;
import static com.google.javascript.jscomp.CompilationLevel.SIMPLE_OPTIMIZATIONS;
import static com.google.javascript.jscomp.CompilerOptions.LanguageMode.ECMASCRIPT_2021;
import static com.google.javascript.jscomp.CompilerOptions.LanguageMode.STABLE;
import static com.google.javascript.jscomp.DiagnosticGroup.forType;
import static com.google.javascript.jscomp.DiagnosticGroups.DEPRECATED_ANNOTATIONS;
import static com.google.javascript.jscomp.DiagnosticGroups.JSDOC_MISSING_TYPE;
import static com.google.javascript.jscomp.DiagnosticGroups.MISPLACED_MSG_ANNOTATION;
import static com.google.javascript.jscomp.DiagnosticGroups.MISPLACED_SUPPRESS;
import static com.google.javascript.jscomp.DiagnosticGroups.MISPLACED_TYPE_ANNOTATION;
import static com.google.javascript.jscomp.DiagnosticGroups.NON_STANDARD_JSDOC;
import static com.google.javascript.jscomp.DiagnosticGroups.forName;
import static com.google.javascript.jscomp.ImplicitNullabilityCheck.IMPLICITLY_NONNULL_JSDOC;
import static com.google.javascript.jscomp.ImplicitNullabilityCheck.IMPLICITLY_NULLABLE_JSDOC;
import static com.google.javascript.jscomp.lint.CheckConstantCaseNames.MISSING_CONST_PROPERTY;
import static com.google.javascript.jscomp.lint.CheckInterfaces.MISSING_JSDOC_IN_DECLARATION_STATEMENT;
import static com.google.javascript.jscomp.lint.CheckJSDocStyle.LINT_DIAGNOSTICS;
import static com.google.javascript.jscomp.lint.CheckNullabilityModifiers.MISSING_NULLABILITY_MODIFIER_JSDOC;
import static com.google.javascript.jscomp.lint.CheckNullabilityModifiers.NULL_MISSING_NULLABILITY_MODIFIER_JSDOC;
import static com.google.javascript.jscomp.lint.CheckNullabilityModifiers.REDUNDANT_NULLABILITY_MODIFIER_JSDOC;

import java.io.IOException;
import java.util.Map;

import org.apache.maven.plugin.logging.Log;
import com.google.javascript.jscomp.Compiler;
import com.google.javascript.jscomp.CompilerOptions;
import com.google.javascript.jscomp.Result;
import com.google.javascript.jscomp.SourceFile;
import com.google.javascript.jscomp.SourceMap;

import com.atlassian.maven.plugins.amps.code.Sources;

public class GoogleClosureJsMinifier {

    protected static CompilerOptions getOptions(
            Map<String, String> closureOptions, Log log, boolean closureJsdocWarningsEnabled) {
        GoogleClosureOptionsHandler googleClosureOptionsHandler = new GoogleClosureOptionsHandler(log);
        googleClosureOptionsHandler.getCompilerOptions().setStrictModeInput(false);
        googleClosureOptionsHandler.getCompilerOptions().setLanguageIn(STABLE);
        googleClosureOptionsHandler.getCompilerOptions().setLanguageOut(ECMASCRIPT_2021);
        SIMPLE_OPTIMIZATIONS.setOptionsForCompilationLevel(googleClosureOptionsHandler.getCompilerOptions());

        // allow for overrides
        if (closureOptions != null) {
            closureOptions.forEach(googleClosureOptionsHandler::setOption);
        }

        if (!closureJsdocWarningsEnabled) {
            googleClosureOptionsHandler.getCompilerOptions().setWarningLevel(NON_STANDARD_JSDOC, OFF);
            googleClosureOptionsHandler.getCompilerOptions().setWarningLevel(MISPLACED_TYPE_ANNOTATION, OFF);
            googleClosureOptionsHandler.getCompilerOptions().setWarningLevel(MISPLACED_MSG_ANNOTATION, OFF);
            googleClosureOptionsHandler.getCompilerOptions().setWarningLevel(MISPLACED_SUPPRESS, OFF);
            googleClosureOptionsHandler.getCompilerOptions().setWarningLevel(DEPRECATED_ANNOTATIONS, OFF);
            googleClosureOptionsHandler.getCompilerOptions().setWarningLevel(JSDOC_MISSING_TYPE, OFF);
            googleClosureOptionsHandler.getCompilerOptions().setWarningLevel(LINT_DIAGNOSTICS, OFF);
            googleClosureOptionsHandler
                    .getCompilerOptions()
                    .setWarningLevel(forType(MISSING_NULLABILITY_MODIFIER_JSDOC), OFF);
            googleClosureOptionsHandler
                    .getCompilerOptions()
                    .setWarningLevel(forType(NULL_MISSING_NULLABILITY_MODIFIER_JSDOC), OFF);
            googleClosureOptionsHandler
                    .getCompilerOptions()
                    .setWarningLevel(forType(REDUNDANT_NULLABILITY_MODIFIER_JSDOC), OFF);
            googleClosureOptionsHandler.getCompilerOptions().setWarningLevel(forType(IMPLICITLY_NONNULL_JSDOC), OFF);
            googleClosureOptionsHandler.getCompilerOptions().setWarningLevel(forType(IMPLICITLY_NULLABLE_JSDOC), OFF);
            googleClosureOptionsHandler
                    .getCompilerOptions()
                    .setWarningLevel(forType(MISSING_JSDOC_IN_DECLARATION_STATEMENT), OFF);
            googleClosureOptionsHandler.getCompilerOptions().setWarningLevel(forType(MISSING_CONST_PROPERTY), OFF);
            // Non-public fields
            googleClosureOptionsHandler
                    .getCompilerOptions()
                    .setWarningLevel(forName("JSC_JSDOC_MISSING_BRACES_WARNING"), OFF);
        }

        return googleClosureOptionsHandler.getCompilerOptions();
    }

    public static Sources compile(
            String code,
            Map<String, String> closureOptions,
            Log log,
            boolean closureJsdocWarningsEnabled,
            String filenameForLogging) {
        Compiler compiler = new Compiler();
        CompilerOptions options = getOptions(closureOptions, log, closureJsdocWarningsEnabled);

        // Dummy file paths used for source map, all those paths will be replaced at runtime.
        options.setSourceMapFormat(SourceMap.Format.V3);
        options.setSourceMapOutputPath("/dummy-file-path");

        SourceFile extern = SourceFile.fromCode("externs.js", "function alert(x) {}");

        SourceFile input = SourceFile.fromCode(filenameForLogging, code);

        // compile() returns a Result, but it is not needed here.
        Result result = compiler.compile(extern, input, options);
        // Why? To avoid introducing bugs silently
        // If there's too much screaming we can use the original source, but...
        // I'm betting that our consumers care about the bundle size more
        // Also a good forcing function to periodically update GCC
        if (result.errors != null && !result.errors.isEmpty()) {
            throw new RuntimeException("There was a minification error");
        }
        String min = compiler.toSource();

        // Getting the source map.
        // Note that it should be called after the `compiler.toSource` otherwise it would be empty.
        StringBuilder sourceMapStream = new StringBuilder();
        try {
            result.sourceMap.appendTo(sourceMapStream, "/dummy-file-path");
        } catch (IOException e) {
            log.warn("can't create source map", e);
        }

        // The compiler is responsible for generating the compiled code; it is not
        // accessible via the Result.
        return new Sources(min, sourceMapStream.toString());
    }
}
