package com.atlassian.maven.plugins.amps.osgi;

import static com.atlassian.maven.plugins.amps.util.FileUtils.file;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.HashMap;
import java.util.Map;

import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugin.MojoFailureException;
import org.apache.maven.plugins.annotations.Mojo;
import org.apache.maven.plugins.annotations.Parameter;

import com.atlassian.maven.plugins.amps.AbstractAmpsMojo;

/**
 * Validates the current project's {@code target/classes/META-INF/MANIFEST.MF} file, if present. Also, if the BND
 * instructions contain an {@code Atlassian-Plugin-Key} header, indicating a transformerless plugin, this goal checks
 * that {@code atlassian-plugin.xml} contains no {@code <component>}, {@code <component-import>}, or
 * {@code <module-type>} elements.
 */
@Mojo(name = "validate-manifest")
public class ValidateManifestMojo extends AbstractAmpsMojo {
    /** Whether to skip validation or not */
    @Parameter(property = "manifest.validation.skip")
    protected boolean skipManifestValidation = false;

    /** The BND instructions for the bundle. */
    @Parameter
    private Map<String, String> instructions = new HashMap<>();

    public void execute() throws MojoExecutionException, MojoFailureException {
        final File mfile =
                file(getMavenContext().getProject().getBuild().getOutputDirectory(), "META-INF", "MANIFEST.MF");

        // Only valid if the manifest exists
        if (!skipManifestValidation && mfile.exists()) {
            getLog().info("Manifest found, validating...");
            try {
                checkManifestEndsWithNewLine(mfile);

                if (instructions.containsKey("Atlassian-Plugin-Key")) {
                    AtlassianPluginContentValidator validator = new AtlassianPluginContentValidator();
                    validator.validate(file(
                            getMavenContext().getProject().getBuild().getOutputDirectory(), "atlassian-plugin.xml"));
                }
            } catch (IOException e) {
                throw new MojoExecutionException("Unable to read manifest", e);
            }
            getLog().info("Manifest validated");
        } else {
            getLog().info("No manifest found or validation skip flag specified, skipping validation");
        }
    }

    private void checkManifestEndsWithNewLine(final File mfile)
            throws IOException, MojoExecutionException, MojoFailureException {
        try (InputStream is = new FileInputStream(mfile)) {
            final long bytesToSkip = mfile.length() - 1;
            long bytesSkipped = is.skip(bytesToSkip);
            if (bytesSkipped != bytesToSkip) {
                throw new MojoExecutionException(
                        "Could not skip " + bytesToSkip + " bytes reading " + mfile.getAbsolutePath());
            } else if (is.read() != '\n') {
                throw new MojoFailureException(
                        "Manifests must end with a new line. " + mfile.getAbsolutePath() + " doesn't.");
            }
        }
    }
}
