package com.atlassian.maven.plugins.amps.frontend.association.verification;

import com.atlassian.maven.plugins.amps.frontend.association.mapping.ReportFeManifestAssociationsMojo;
import com.atlassian.maven.plugins.amps.frontend.association.verification.impl.ArtifactScannerImpl;
import com.atlassian.maven.plugins.amps.frontend.association.verification.model.ArtifactScanResults;
import com.atlassian.maven.plugins.amps.frontend.association.verification.model.FailurePreferences;
import com.atlassian.maven.plugins.amps.frontend.association.verification.model.ModuleScanResults;
import com.atlassian.maven.plugins.amps.frontend.association.verification.utils.ArtifactPathUtils;
import com.atlassian.maven.plugins.amps.frontend.association.verification.utils.TrueZipUtils;
import com.google.common.annotations.VisibleForTesting;
import de.schlichtherle.truezip.file.TFile;
import org.apache.maven.plugin.AbstractMojo;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugins.annotations.Component;
import org.apache.maven.plugins.annotations.Mojo;
import org.apache.maven.plugins.annotations.Parameter;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.ArrayDeque;
import java.util.Map;
import java.util.Queue;
import java.util.Set;

/**
 * Verifies if all javascript files in a given artifact have corresponding manifest association
 * created by {@link ReportFeManifestAssociationsMojo}.
 * It accepts all standard archive types: {@link ArtifactScannerImpl#SUPPORTED_ARCHIVE_EXTENSIONS}.
 * And also works with directories.
 *
 * @since 8.17.0
 */
@Mojo(name = "verify-fe-manifest-associations", requiresProject = false)
public class VerifyFeManifestAssociationsMojo extends AbstractMojo {

    /**
     * Path to input artifact or directory which will be verified
     *
     * @since 8.17.0
     */
    @Parameter(property = "verify.fe.manifest.associations.inputEntrypoint", alias = "verifyFeManifestAssociationsInputEntrypoint", required = true)
    private String inputEntrypoint;

    /**
     * Flag controlling whether to fail when undeclared files were found
     *
     * @since 8.17.0
     */
    @Parameter(property = "verify.fe.manifest.associations.failOnUndeclaredFiles", alias = "verifyFeManifestAssociationsFailOnUndeclaredFiles", defaultValue = "true")
    private Boolean failOnUndeclaredFiles;

    /**
     * Flag controlling whether to fail when any declarations not matching existing files were found
     *
     * @since 8.17.0
     */
    @Parameter(property = "verify.fe.manifest.associations.failOnExtraDeclarations", alias = "verifyFeManifestAssociationsFailOnExtraDeclarations", defaultValue = "true")
    private Boolean failOnExtraDeclarations;

    private final Logger logger = LoggerFactory.getLogger(Logger.ROOT_LOGGER_NAME);

    @Component
    private ArtifactScanner artifactScanner;
    @Component
    private DeclarationsReader declarationsReader;
    @Component
    private DeclarationsChecker declarationsChecker;

    public void execute() throws MojoExecutionException {
        logger.info("Verifying frontend manifest associations for an artifact: {}", inputEntrypoint);

        FailurePreferences failurePreferences = new FailurePreferences(failOnUndeclaredFiles, failOnExtraDeclarations);
        verifyManifestAssociations(inputEntrypoint, failurePreferences);
    }

    @VisibleForTesting
    protected void verifyManifestAssociations(String inputEntrypoint, FailurePreferences failurePreferences) throws MojoExecutionException {
        TrueZipUtils.configureTrueZip();
        declarationsChecker.setFailurePreferences(failurePreferences);

        try {
            scanEntrypoint(inputEntrypoint);
            declarationsChecker.finalizeChecks();
        } finally {
            TrueZipUtils.unmountTrueZip();
        }
    }

    private void scanEntrypoint(String inputEntrypoint) throws MojoExecutionException {
        Queue<TFile> queue = new ArrayDeque<>();
        TFile root = new TFile(inputEntrypoint);
        queue.add(root);

        while (!queue.isEmpty()) {
            TFile currentArtifact = queue.poll();
            ArtifactScanResults scanResults = artifactScanner.scan(currentArtifact);
            queue.addAll(scanResults.getArchiveFiles());

            verifyModules(scanResults.getModules());
        }
    }

    private void verifyModules(Map<TFile, ModuleScanResults> modules) throws MojoExecutionException {
        for (Map.Entry<TFile, ModuleScanResults> module : modules.entrySet()) {
            TFile moduleRoot = module.getKey();
            ModuleScanResults moduleScanResults = module.getValue();

            Set<String> moduleRelativeDeclarations = declarationsReader
                .readRelativeDeclarations(moduleScanResults.getAssociationFiles(), moduleRoot);
            Set<String> moduleRelativeJsFiles = ArtifactPathUtils
                .getRelativePaths(moduleScanResults.getJavascriptFiles(), moduleRoot);

            declarationsChecker.verifyModule(moduleRoot, moduleRelativeDeclarations, moduleRelativeJsFiles);
        }
    }
}
