package com.atlassian.maven.plugins.amps.frontend.association.mapping;

import com.atlassian.maven.plugins.amps.AbstractAmpsMojo;
import com.atlassian.maven.plugins.amps.MavenContext;
import com.atlassian.maven.plugins.amps.frontend.association.mapping.model.FeManifestAssociation;
import com.atlassian.maven.plugins.amps.frontend.association.mapping.model.FeManifestAssociationParameter;
import com.atlassian.maven.plugins.amps.frontend.association.mapping.model.FeManifestAssociationsConfiguration;
import com.atlassian.maven.plugins.amps.frontend.association.mapping.model.FeOutputJsFileDeclaration;
import com.atlassian.maven.plugins.amps.frontend.association.mapping.utils.DirectoryHelper;
import com.atlassian.maven.plugins.amps.frontend.association.mapping.utils.JsonParser;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugin.MojoFailureException;
import org.apache.maven.plugins.annotations.Mojo;
import org.apache.maven.plugins.annotations.Parameter;

import java.util.List;
import java.util.Map;

import static java.lang.Boolean.TRUE;
import static org.apache.maven.plugins.annotations.LifecyclePhase.PREPARE_PACKAGE;

/**
 * Creates a mapping association between output js files and fe module manifest file provided in configuration
 * or from WRM webpack plugin. Prepared association is output to
 * {@code META-INF/fe-manifest-associations/fe-manifest-association.json}.
 *
 * @since 8.15.0
 */
@Mojo(name = "report-fe-manifest-associations", defaultPhase = PREPARE_PACKAGE)
public class ReportFeManifestAssociationsMojo extends AbstractAmpsMojo {

    /**
     * Mapping configuration providing manifest file and a list of output js files
     *
     * @since 8.15.0
     */
    @Parameter
    private List<FeManifestAssociationParameter> feManifestAssociations;

    @Parameter
    private FeManifestAssociationsConfiguration feManifestAssociationsConfiguration = new FeManifestAssociationsConfiguration();

    /**
     * When set to true the mojo will not execute
     *
     * @since 8.17.2
     */
    @Parameter(property = "skipReportFeManifestAssociations")
    private Boolean skipReportFeManifestAssociations;

    @Override
    public void execute() throws MojoExecutionException, MojoFailureException {
        if (TRUE.equals(skipReportFeManifestAssociations)) {
            getLog().info("Skipping `report-fe-manifest-associations` execution");
            return;
        }

        MavenContext mavenContext = getMavenContext();
        DirectoryHelper directoryHelper = new DirectoryHelper(mavenContext);
        JsonParser jsonParser = new JsonParser();

        FeManifestAssociationReader reader = new FeManifestAssociationReader(directoryHelper, jsonParser);
        List<FeManifestAssociation> manifestAssociations = reader.getManifestAssociations(feManifestAssociations);
        if (manifestAssociations.isEmpty()) {
            getLog().info("There were no frontend manifest associations found. Skipping generating final mapping.");
            return;
        }

        FeManifestAssociationProcessor processor = new FeManifestAssociationProcessor(directoryHelper, jsonParser);
        Map<String, FeOutputJsFileDeclaration> outputJsFileDeclarations = processor
                .transformManifestAssociationsToFileDeclarations(manifestAssociations);

        String packaging = mavenContext.getProject().getPackaging();

        if (packaging.equals("war")) {
            getLog().warn("Cannot verify declared FE associations because the project specifies 'war' packaging. This is a limitation of the report-fe-manifest-associations mojo. To ensure correctness, verify fe-manifest-association.json exists in the output and declares all FE files.");
        } else {
            FeManifestAssociationArbiter arbiter = new FeManifestAssociationArbiter(
                directoryHelper, feManifestAssociationsConfiguration, getLog());
            arbiter.verifyOutput(outputJsFileDeclarations);
        }

        FeManifestAssociationWriter writer = new FeManifestAssociationWriter(directoryHelper, jsonParser);
        writer.writeManifestAssociation(outputJsFileDeclarations);
    }
}
