package com.atlassian.maven.plugins.amps.frontend.association.mapping;

import com.atlassian.maven.plugins.amps.frontend.association.mapping.model.ExternalFeManifestAssociationConfiguration;
import com.atlassian.maven.plugins.amps.frontend.association.mapping.model.FeManifestAssociation;
import com.atlassian.maven.plugins.amps.frontend.association.mapping.model.FeManifestAssociationParameter;
import com.atlassian.maven.plugins.amps.frontend.association.mapping.model.WebpackFeManifestAssociationConfiguration;
import com.atlassian.maven.plugins.amps.frontend.association.mapping.utils.DirectoryHelper;
import com.atlassian.maven.plugins.amps.frontend.association.mapping.utils.JsonParser;
import com.google.common.annotations.VisibleForTesting;
import org.apache.maven.plugin.MojoExecutionException;

import javax.annotation.Nonnull;
import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import static com.atlassian.maven.plugins.amps.frontend.association.mapping.utils.DirectoryHelper.WEBPACK_CONFIGURATIONS_DIR;

public class FeManifestAssociationReader {

    private static final String WEBPACK_CONFIGURATIONS_SUFFIX = "-webpack.intermediary.json";

    private final DirectoryHelper directoryHelper;
    private final JsonParser jsonParser;

    public FeManifestAssociationReader(DirectoryHelper directoryHelper, JsonParser jsonParser) {
        this.directoryHelper = directoryHelper;
        this.jsonParser = jsonParser;
    }

    public List<FeManifestAssociation> getManifestAssociations(List<FeManifestAssociationParameter> manifestAssociationConfigurations) throws MojoExecutionException {
        List<FeManifestAssociation> fromConfiguration = getManifestAssociationsFromConfiguration(manifestAssociationConfigurations);
        List<FeManifestAssociation> fromWebpackAssociations = getManifestAssociationsFromWebpackConfigurations(
                directoryHelper.getOutputFileAbsolutePath(WEBPACK_CONFIGURATIONS_DIR)
        );

        return Stream.of(
                fromConfiguration.stream(),
                fromWebpackAssociations.stream()
        ).flatMap(i -> i).collect(Collectors.toList());
    }


    @VisibleForTesting
    @Nonnull
    protected List<FeManifestAssociation> getManifestAssociationsFromWebpackConfigurations(String webpackConfigurationsPath) throws MojoExecutionException {
        File webpackDirectory = new File(webpackConfigurationsPath);
        List<FeManifestAssociation> manifestAssociations = new ArrayList<>();

        if (webpackDirectory.exists() && webpackDirectory.isDirectory()) {
            File[] directoryFiles = webpackDirectory.listFiles();

            if (directoryFiles != null) {
                for (final File fileEntry : directoryFiles) {
                    if (!fileEntry.isDirectory() && fileEntry.getName().endsWith(WEBPACK_CONFIGURATIONS_SUFFIX)) {
                        manifestAssociations.add(
                                getManifestAssociationFromWebpackConfiguration(fileEntry.getAbsolutePath())
                        );
                    }
                }
            }
        }

        return manifestAssociations;
    }

    private FeManifestAssociation getManifestAssociationFromWebpackConfiguration(String webpackConfigurationAbsPath) throws MojoExecutionException {
        File webpackConfigurationFile = new File(webpackConfigurationAbsPath);

        WebpackFeManifestAssociationConfiguration webpackConfiguration;
        try {
            webpackConfiguration = jsonParser.readFile(webpackConfigurationFile, WebpackFeManifestAssociationConfiguration.class);
        } catch (IOException e) {
            throw new MojoExecutionException(String.format("Webpack manifest association couldn't be parsed. Path: %s", webpackConfigurationAbsPath));
        }

        return new FeManifestAssociation(
                webpackConfiguration.getPackageName(),
                webpackConfiguration.getOutputDirectoryFiles(),
                directoryHelper.getPathRelativeToSourceDirectory(webpackConfigurationFile.getAbsolutePath())
        );
    }

    @VisibleForTesting
    @Nonnull
    protected FeManifestAssociation getManifestAssociationsFromExternalConfiguration(String externalConfigurationPath) throws MojoExecutionException {
        String externalConfigurationAbsolutePath = directoryHelper.getSourceFileAbsolutePath(externalConfigurationPath);
        File externalConfigurationFile = new File(externalConfigurationAbsolutePath);

        ExternalFeManifestAssociationConfiguration externalConfiguration;
        try {
            externalConfiguration = jsonParser.readFile(externalConfigurationFile, ExternalFeManifestAssociationConfiguration.class);
        } catch (IOException e) {
            throw new MojoExecutionException(String.format("External manifest association couldn't be parsed. Path: %s", externalConfigurationAbsolutePath));
        }

        return new FeManifestAssociation(
                externalConfiguration.getPackageName(),
                externalConfiguration.getManuallyVerifiedEveryFileHasItsDependenciesDeclaredInTheManifest().getOutputDirectoryFiles(),
                directoryHelper.getPathRelativeToSourceDirectory(externalConfigurationPath)
        );
    }

    @Nonnull
    private List<FeManifestAssociation> getManifestAssociationsFromConfiguration(List<FeManifestAssociationParameter> manifestAssociationConfigurations) throws MojoExecutionException {
        List<FeManifestAssociation> manifestAssociations = new ArrayList<>();
        Set<String> externalConfigurations = new HashSet<>();

        if (manifestAssociationConfigurations != null) {
            for (FeManifestAssociationParameter associationConfiguration : manifestAssociationConfigurations) {
                if (associationConfiguration.getOutputDirectoryFilesDeclaration() != null) {
                    externalConfigurations.add(associationConfiguration.getOutputDirectoryFilesDeclaration());
                } else {
                    manifestAssociations.add(createManifestAssociationFromConfiguration(associationConfiguration));
                }
            }
        }

        for (String externalConfiguration : externalConfigurations) {
            FeManifestAssociation manifestAssociation = getManifestAssociationsFromExternalConfiguration(externalConfiguration);
            manifestAssociations.add(manifestAssociation);
        }

        return manifestAssociations;
    }

    private FeManifestAssociation createManifestAssociationFromConfiguration(FeManifestAssociationParameter associationConfiguration) {
        return new FeManifestAssociation(
                associationConfiguration.getPackageName(),
                associationConfiguration
                        .getFilesDeclaration()
                        .getOutputDirectoryFiles()
                        .stream()
                        .filter(path -> path != null && !path.isEmpty())
                        .map(String::trim)
                        .collect(Collectors.toList()),
                "pom.xml"
        );
    }
}
