package com.atlassian.maven.plugins.amps.product.jira;

import com.atlassian.maven.plugins.amps.DataSource;
import com.atlassian.maven.plugins.amps.LibArtifact;
import org.apache.maven.model.Dependency;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugin.logging.Log;
import org.codehaus.plexus.util.xml.Xpp3Dom;

import javax.annotation.Nonnull;
import java.util.ArrayList;
import java.util.List;

import static java.util.Collections.emptyList;
import static java.util.Objects.requireNonNull;
import static org.twdata.maven.mojoexecutor.MojoExecutor.configuration;
import static org.twdata.maven.mojoexecutor.MojoExecutor.element;
import static org.twdata.maven.mojoexecutor.MojoExecutor.name;

/**
 * Base class for concrete {@link JiraDatabase} implementations.
 */
public abstract class AbstractJiraDatabase implements JiraDatabase {
    protected final DataSource dataSource;
    protected final Log log;

    protected LibArtifact lib;

    protected AbstractJiraDatabase(final DataSource dataSource, final Log log) {
        this.dataSource = requireNonNull(dataSource);
        this.log = requireNonNull(log);
    }

    @Nonnull
    public final DataSource getDataSource() {
        return dataSource;
    }

    protected abstract String dropDatabase() throws MojoExecutionException;

    protected abstract String createDatabase() throws MojoExecutionException;

    protected abstract String dropUser();

    protected abstract String createUser();

    protected abstract String grantPermissionForUser() throws MojoExecutionException;

    protected Xpp3Dom systemDatabaseConfiguration() {
        return configuration(
                element(name("driver"), dataSource.getDriver()),
                element(name("url"), dataSource.getDefaultDatabase()),
                element(name("username"), dataSource.getSystemUsername()),
                element(name("password"), dataSource.getSystemPassword()),
                element(name("autocommit"), "true")
        );
    }

    protected Xpp3Dom productDatabaseConfiguration() {
        return configuration(
                element(name("driver"), dataSource.getDriver()),
                element(name("url"), dataSource.getUrl()),
                element(name("username"), dataSource.getUsername()),
                element(name("password"), dataSource.getPassword()),
                // we need commit transaction for drop database and then create them again
                element(name("autocommit"), "true")
        );
    }

    protected abstract String getDatabaseName(String url) throws MojoExecutionException;

    @Override
    @Nonnull
    public List<Dependency> getSqlMavenDependencies() {
        if (dataSource.getLibArtifacts() == null) {
            return emptyList();
        }
        final List<Dependency> dependencies = new ArrayList<>();
        for (LibArtifact libArtifact : dataSource.getLibArtifacts()) {
            final Dependency dependency = new Dependency();
            dependency.setGroupId(libArtifact.getGroupId());
            dependency.setArtifactId(libArtifact.getArtifactId());
            dependency.setVersion(libArtifact.getVersion());
            dependencies.add(dependency);
        }
        return dependencies;
    }

    @Override
    @Nonnull
    public Xpp3Dom getSqlMavenFileImportConfiguration() {
        Xpp3Dom pluginConfiguration = productDatabaseConfiguration();
        pluginConfiguration.addChild(
                element(name("srcFiles"),
                        element(name("srcFile"), dataSource.getDumpFilePath())).toDom()
        );
        return pluginConfiguration;
    }

    @Override
    public Xpp3Dom getExecMavenToolImportConfiguration() throws MojoExecutionException {
        return null;
    }
}
