package com.atlassian.maven.plugins.amps;

import aQute.bnd.osgi.Constants;
import com.atlassian.maven.plugins.amps.minifier.MinifierParameters;
import com.atlassian.maven.plugins.amps.minifier.ResourcesMinifier;
import com.atlassian.maven.plugins.amps.util.AmpsCreatePluginPrompter;
import com.atlassian.maven.plugins.amps.util.CreatePluginProperties;
import com.atlassian.maven.plugins.amps.util.MojoExecutorWrapper;
import com.atlassian.maven.plugins.amps.util.VersionUtils;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.collect.ImmutableMap;
import com.googlecode.htmlcompressor.compressor.XmlCompressor;
import org.apache.commons.io.FileUtils;
import org.apache.commons.io.filefilter.FileFilterUtils;
import org.apache.commons.io.filefilter.IOFileFilter;
import org.apache.commons.lang3.StringUtils;
import org.apache.maven.archetype.common.DefaultPomManager;
import org.apache.maven.archetype.common.MavenJDOMWriter;
import org.apache.maven.archetype.common.util.Format;
import org.apache.maven.model.FileSet;
import org.apache.maven.model.Model;
import org.apache.maven.model.Plugin;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugin.logging.Log;
import org.apache.maven.project.MavenProject;
import org.codehaus.plexus.components.interactivity.PrompterException;
import org.codehaus.plexus.util.IOUtil;
import org.codehaus.plexus.util.xml.Xpp3Dom;
import org.jdom.Document;
import org.jdom.input.SAXBuilder;
import org.twdata.maven.mojoexecutor.MojoExecutor.Element;
import org.twdata.maven.mojoexecutor.MojoExecutor.ExecutionEnvironment;

import javax.annotation.Nullable;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.FilenameFilter;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStreamWriter;
import java.io.UncheckedIOException;
import java.io.Writer;
import java.nio.charset.Charset;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Objects;
import java.util.Properties;
import java.util.Set;
import java.util.TreeSet;
import java.util.jar.Manifest;
import java.util.regex.Matcher;
import java.util.stream.Stream;

import static com.atlassian.maven.plugins.amps.BannedDependencies.getBannedElements;
import static com.atlassian.maven.plugins.amps.util.FileUtils.file;
import static java.io.File.createTempFile;
import static java.lang.Boolean.FALSE;
import static java.lang.Boolean.TRUE;
import static java.lang.String.format;
import static java.nio.charset.StandardCharsets.UTF_8;
import static java.nio.file.Files.createDirectories;
import static java.nio.file.Files.createTempDirectory;
import static java.nio.file.Files.walk;
import static java.util.Map.Entry.comparingByKey;
import static java.util.Objects.requireNonNull;
import static java.util.stream.Collectors.groupingBy;
import static java.util.stream.Collectors.joining;
import static java.util.stream.Collectors.mapping;
import static java.util.stream.Collectors.toCollection;
import static java.util.stream.Collectors.toMap;
import static org.apache.commons.io.FileUtils.copyInputStreamToFile;
import static org.apache.commons.io.FileUtils.deleteDirectory;
import static org.apache.commons.lang3.StringUtils.isNotBlank;
import static org.twdata.maven.mojoexecutor.MojoExecutor.artifactId;
import static org.twdata.maven.mojoexecutor.MojoExecutor.configuration;
import static org.twdata.maven.mojoexecutor.MojoExecutor.element;
import static org.twdata.maven.mojoexecutor.MojoExecutor.goal;
import static org.twdata.maven.mojoexecutor.MojoExecutor.groupId;
import static org.twdata.maven.mojoexecutor.MojoExecutor.name;
import static org.twdata.maven.mojoexecutor.MojoExecutor.plugin;
import static org.twdata.maven.mojoexecutor.MojoExecutor.version;

/**
 * Executes specific maven goals
 */
public class MavenGoals {

    @VisibleForTesting
    static final String AJP_PORT_PROPERTY = "cargo.tomcat.ajp.port";

    /**
     * Defines a Failsafe/Surefire {@code %regex} pattern which can be used to exclude/include integration tests.
     * <p>
     * Pattern handling for excludes and includes changed in Surefire 2.22, and patterns that don't start with
     * "**&#47;" are now prefixed with it automatically. That means "it/**" becomes "**&#47;it/**", which matches
     * any test with an "it" package anywhere, not just at the root. Regexes are <i>not</i> automatically prefixed,
     * so using a regex pattern here allows us to continue only matching tests with an "it" package at the root.
     * <p>
     * <b>Warning</b>: Surefire's documentation states that all slashes are forward, even on Windows. However,
     * <a href="https://github.com/bturner/surefire-slashes">a simple test</a> proves that that's not the case.
     * <a href="https://issues.apache.org/jira/browse/SUREFIRE-1599">SUREFIRE-1599</a> has been created to track
     * this mismatch between the documentation and the implementation.
     *
     * @since 8.0
     */
    public static final String REGEX_INTEGRATION_TESTS = "%regex[it[/\\\\].*]";

    private static final String ABSTRACT_CLASSES = "**/Abstract*";

    private static final String INNER_CLASSES = "**/*$*";

    private static final String REPORTS_DIRECTORY = "reportsDirectory";

    private final MavenContext ctx;

    private final Map<String, Container> idToContainerMap = ImmutableMap.<String, Container>builder()
            .put("tomcat5x", new Container("tomcat5x", "org.apache.tomcat", "apache-tomcat", "5.5.36"))
            .put("tomcat6x", new Container("tomcat6x", "org.apache.tomcat", "apache-tomcat", "6.0.41"))
            .put("tomcat7x", new Container("tomcat7x", "org.apache.tomcat", "apache-tomcat", "7.0.73-atlassian-hosted", "windows-x64"))
            .put("tomcat8x", new Container("tomcat8x", "org.apache.tomcat", "apache-tomcat", "8.0.53-atlassian-hosted", "windows-x64"))
            .put("tomcat85x", new Container("tomcat8x", "org.apache.tomcat", "apache-tomcat", "8.5.40-atlassian-hosted", "windows-x64"))
            .put("tomcat85_6", new Container("tomcat8x", "org.apache.tomcat", "apache-tomcat", "8.5.6-atlassian-hosted", "windows-x64"))
            .put("tomcat85_29", new Container("tomcat8x", "org.apache.tomcat", "apache-tomcat", "8.5.29-atlassian-hosted", "windows-x64"))
            .put("tomcat85_32", new Container("tomcat8x", "org.apache.tomcat", "apache-tomcat", "8.5.32-atlassian-hosted", "windows-x64"))
            .put("tomcat85_35", new Container("tomcat8x", "org.apache.tomcat", "apache-tomcat", "8.5.35-atlassian-hosted", "windows-x64"))
            .put("tomcat9x", new Container("tomcat9x", "org.apache.tomcat", "apache-tomcat", "9.0.11-atlassian-hosted", "windows-x64"))
            .put("jetty6x", new Container("jetty6x"))
            .put("jetty7x", new Container("jetty7x"))
            .put("jetty8x", new Container("jetty8x"))
            .put("jetty9x", new Container("jetty9x"))
            .build();

    private final Log log;
    private final MojoExecutorWrapper mojoExecutorWrapper;

    public MavenGoals(final MavenContext ctx, final MojoExecutorWrapper mojoExecutorWrapper) {
        this.ctx = requireNonNull(ctx);
        this.log = ctx.getLog();
        this.mojoExecutorWrapper = requireNonNull(mojoExecutorWrapper);
    }

    private ExecutionEnvironment executionEnvironment() {
        return ctx.getExecutionEnvironment();
    }

    public MavenProject getContextProject() {
        return ctx.getProject();
    }

    public void executeAmpsRecursively(final String ampsVersion, final String ampsGoal, final Xpp3Dom configuration)
            throws MojoExecutionException {
        mojoExecutorWrapper.executeWithMergedConfig(
                plugin(
                        groupId("com.atlassian.maven.plugins"),
                        artifactId("amps-maven-plugin"),
                        version(ampsVersion)
                ),
                goal(ampsGoal),
                configuration,
                executionEnvironment());
    }

    public void createPlugin(final String productId, AmpsCreatePluginPrompter createPrompter) throws MojoExecutionException {
        CreatePluginProperties props = null;
        Properties systemProps = System.getProperties();

        if (systemProps.containsKey("groupId")
                && systemProps.containsKey("artifactId")
                && systemProps.containsKey("version")
                && systemProps.containsKey("package")) {
            props = new CreatePluginProperties(systemProps.getProperty("groupId"),
                    systemProps.getProperty("artifactId"), systemProps.getProperty("version"),
                    systemProps.getProperty("package"), systemProps.getProperty("useOsgiJavaConfig", "N"));
        }
        if (props == null) {
            try {
                props = createPrompter.prompt();
            } catch (PrompterException e) {
                throw new MojoExecutionException("Unable to gather properties", e);
            }
        }

        if (props != null) {
            ExecutionEnvironment execEnv = executionEnvironment();

            Properties userProperties = execEnv.getMavenSession().getUserProperties();
            userProperties.setProperty("groupId", props.getGroupId());
            userProperties.setProperty("artifactId", props.getArtifactId());
            userProperties.setProperty("version", props.getVersion());
            userProperties.setProperty("package", props.getThePackage());
            userProperties.setProperty("useOsgiJavaConfig", props.getUseOsgiJavaConfigInMavenInvocationFormat());

            mojoExecutorWrapper.executeWithMergedConfig(
                    ctx.getPlugin("org.apache.maven.plugins", "maven-archetype-plugin"),
                    goal("generate"),
                    configuration(
                            element(name("archetypeGroupId"), "com.atlassian.maven.archetypes"),
                            element(name("archetypeArtifactId"), (productId.equals("all") ? "" : productId + "-") + "plugin-archetype"),
                            element(name("archetypeVersion"), VersionUtils.getVersion()),
                            element(name("interactiveMode"), "false")
                    ),
                    execEnv);

            /*
            The problem is if plugin is sub of multiple module project, then the pom file will be add parent section.
            When add parent section to module pom file, maven use the default Format with lineEnding is \r\n.
            This step add \r\n character as the line ending.
            Call the function below to remove cr (\r) character
            */
            correctCrlf(props.getArtifactId());

            File pluginDir = new File(ctx.getProject().getBasedir(), props.getArtifactId());

            if (pluginDir.exists()) {
                File src = new File(pluginDir, "src");
                File test = new File(src, "test");
                File java = new File(test, "java");

                String packagePath = props.getThePackage().replaceAll("\\.", Matcher.quoteReplacement(File.separator));
                File packageFile = new File(java, packagePath);
                File packageUT = new File(packageFile, "ut");
                File packageIT = new File(packageFile, "it");

                File ut = new File(new File(java, "ut"), packagePath);
                File it = new File(new File(java, "it"), packagePath);

                if (packageFile.exists()) {
                    try {
                        if (packageUT.exists()) {
                            FileUtils.copyDirectory(packageUT, ut);
                        }

                        if (packageIT.exists()) {
                            FileUtils.copyDirectory(packageIT, it);
                        }

                        IOFileFilter filter = FileFilterUtils.and(FileFilterUtils.notFileFilter(FileFilterUtils.nameFileFilter("it")), FileFilterUtils.notFileFilter(FileFilterUtils.nameFileFilter("ut")));

                        com.atlassian.maven.plugins.amps.util.FileUtils.cleanDirectory(java, filter);

                    } catch (IOException e) {
                        //for now just ignore
                    }
                }
            }
        }
    }

    /**
     * Helper function to scan all generated folder and list all pom.xml files that need to be re-write to remove \r
     */
    private void correctCrlf(String artifactId) {
        if (ctx != null && ctx.getProject() != null
                && ctx.getProject().getBasedir() != null && ctx.getProject().getBasedir().exists()) {
            File outputDirectoryFile = new File(ctx.getProject().getBasedir(), artifactId);

            if (outputDirectoryFile.exists()) {
                FilenameFilter pomFilter = (dir, name) -> "pom.xml".equals(name);

                File[] pomFiles = outputDirectoryFile.listFiles(pomFilter);
                DefaultPomManager pomManager = new DefaultPomManager();

                for (File pom : pomFiles) {
                    processCorrectCrlf(pomManager, pom);
                }
            }
        }
    }

    /**
     * Helper function to re-write pom.xml file with lineEnding \n instead of \r\n
     */
    protected void processCorrectCrlf(DefaultPomManager pomManager, File pom) {
        InputStream inputStream = null;
        Writer outputStreamWriter = null;
        final Model model;
        try {
            model = pomManager.readPom(pom);
            String fileEncoding = StringUtils.isEmpty(model.getModelEncoding()) ? model.getModelEncoding() : "UTF-8";

            inputStream = new FileInputStream(pom);

            SAXBuilder builder = new SAXBuilder();
            Document doc = builder.build(inputStream);

            // The cdata parts of the pom are not preserved from initial to target
            MavenJDOMWriter writer = new MavenJDOMWriter();

            outputStreamWriter = new OutputStreamWriter(new FileOutputStream(pom), fileEncoding);

            Format form = Format.getRawFormat().setEncoding(fileEncoding);
            form.setLineSeparator("\n");
            writer.write(model, doc, outputStreamWriter, form);
        } catch (Exception e) {
            log.error("Have exception when try correct line ending.", e);
        } finally {
            IOUtil.close(inputStream);
            IOUtil.close(outputStreamWriter);
        }
    }

    public void copyBundledDependencies() throws MojoExecutionException {
        mojoExecutorWrapper.executeWithMergedConfig(
                ctx.getPlugin("org.apache.maven.plugins", "maven-dependency-plugin"),
                goal("copy-dependencies"),
                configuration(
                        element(name("includeScope"), "runtime"),
                        element(name("excludeScope"), "provided"),
                        element(name("excludeScope"), "test"),
                        element(name("includeTypes"), "jar"),
                        element(name("outputDirectory"), "${project.build.outputDirectory}/META-INF/lib")
                ),
                executionEnvironment()
        );
    }

    /**
     * Uses the Enforcer plugin to check that the current project has no platform modules in {@code compile} scope,
     * even transitively, i.e. it will not bundle any such artifacts.
     *
     * @param banExcludes any dependencies to be allowed, in the form {@code groupId:artifactId[:version][:type]}
     * @throws MojoExecutionException if something goes wrong
     */
    public void validateBannedDependencies(final Set<String> banExcludes) throws MojoExecutionException {
        log.info("validate banned dependencies");
        mojoExecutorWrapper.executeWithMergedConfig(
                plugin(
                        groupId("org.apache.maven.plugins"),
                        artifactId("maven-enforcer-plugin"),
                        version("3.0.0-M3")
                ),
                goal("enforce"),
                configuration(
                        element(name("rules"),
                                element(name("bannedDependencies"),
                                        element(name("searchTransitive"), "true"),
                                        element(name("message"), "make sure platform artifacts are not bundled into plugin"),
                                        element(name("excludes"), getBannedElements(banExcludes).toArray(new Element[0])))
                        )
                ),
                executionEnvironment()
        );
    }

    public void copyTestBundledDependencies(List<ProductArtifact> testBundleExcludes) throws MojoExecutionException {
        StringBuilder sb = new StringBuilder();

        for (ProductArtifact artifact : testBundleExcludes) {
            log.info("excluding artifact from test jar: " + artifact.getArtifactId());
            sb.append(",").append(artifact.getArtifactId());
        }

        String customExcludes = sb.toString();

        mojoExecutorWrapper.executeWithMergedConfig(
                ctx.getPlugin("org.apache.maven.plugins", "maven-dependency-plugin"),
                goal("copy-dependencies"),
                configuration(
                        element(name("includeScope"), "test"),
                        element(name("excludeScope"), "provided"),
                        element(name("excludeArtifactIds"), "junit" + customExcludes),
                        element(name("useSubDirectoryPerScope"), "true"),
                        element(name("outputDirectory"), "${project.build.directory}/testlibs")
                ),
                executionEnvironment()
        );

        File targetDir = new File(ctx.getProject().getBuild().getDirectory());
        File testlibsDir = new File(targetDir, "testlibs");
        File compileLibs = new File(testlibsDir, "compile");
        File testLibs = new File(testlibsDir, "test");


        File testClassesDir = new File(ctx.getProject().getBuild().getTestOutputDirectory());
        File metainfDir = new File(testClassesDir, "META-INF");
        File libDir = new File(metainfDir, "lib");

        try {
            compileLibs.mkdirs();
            testLibs.mkdirs();
            libDir.mkdirs();

            FileUtils.copyDirectory(compileLibs, libDir);
            FileUtils.copyDirectory(testLibs, libDir);
        } catch (IOException e) {
            throw new MojoExecutionException("unable to copy test libs", e);
        }
    }

    public void copyTestBundledDependenciesExcludingTestScope(List<ProductArtifact> testBundleExcludes) throws MojoExecutionException {
        StringBuilder sb = new StringBuilder();

        for (ProductArtifact artifact : testBundleExcludes) {
            log.info("excluding artifact from test jar: " + artifact.getArtifactId());
            sb.append(",").append(artifact.getArtifactId());
        }

        String customExcludes = sb.toString();

        mojoExecutorWrapper.executeWithMergedConfig(
                ctx.getPlugin("org.apache.maven.plugins", "maven-dependency-plugin"),
                goal("copy-dependencies"),
                configuration(
                        element(name("includeScope"), "runtime"),
                        element(name("excludeScope"), "provided"),
                        element(name("excludeScope"), "test"),
                        element(name("includeTypes"), "jar"),
                        element(name("excludeArtifactIds"), "junit" + customExcludes),
                        element(name("outputDirectory"), "${project.build.testOutputDirectory}/META-INF/lib")
                ),
                executionEnvironment()
        );
    }

    private void extractDependencies(final Xpp3Dom configuration) throws MojoExecutionException {
        // Save a copy of the given config (it's mutated when we do the first extraction)
        final Xpp3Dom copyOfConfiguration = new Xpp3Dom(configuration);

        // Do the extraction they asked for ...
        doExtractDependencies(configuration);

        // ... but check whether that caused any files to be overwritten
        warnAboutOverwrites(copyOfConfiguration);
    }

    private void warnAboutOverwrites(final Xpp3Dom configuration) throws MojoExecutionException {
        final Path tempDirectory = createTempDirectoryForOverwriteDetection();
        configuration.getChild("outputDirectory").setValue(tempDirectory.toString());
        configuration.addChild(element("useSubDirectoryPerArtifact", "true").toDom());
        // We set these overWrite flags so that Maven will allow each dependency to be unpacked again (see MDEP-586)
        configuration.addChild(element("overWriteReleases", "true").toDom());
        configuration.addChild(element("overWriteSnapshots", "true").toDom());
        configuration.addChild(element("silent", "true").toDom());
        doExtractDependencies(configuration);
        checkForOverwrites(tempDirectory);
        try {
            deleteDirectory(tempDirectory.toFile());
        } catch (final IOException ignored) {
            // Ignore; it's in the temp folder anyway
        }
    }

    private void checkForOverwrites(final Path dependencyDirectory) {
        try (final Stream<Path> fileStream = walk(dependencyDirectory)) {
            // Map all dependency files to the artifacts that contain them
            final Map<Path, Set<Path>> artifactsByPath = fileStream
                    .filter(Files::isRegularFile)
                    .map(dependencyDirectory::relativize)
                    .collect(groupingBy(MavenGoals::tail, mapping(MavenGoals::head, toCollection(TreeSet::new))));
            // Find any clashes
            final Map<Path, Set<Path>> clashes = artifactsByPath.entrySet().stream()
                    .filter(e -> e.getValue().size() > 1)
                    .collect(toMap(Entry::getKey, Entry::getValue));
            if (!clashes.isEmpty()) {
                logWarnings(clashes);
            }
        } catch (IOException e) {
            throw new UncheckedIOException(e);
        }
    }

    private void logWarnings(final Map<Path, Set<Path>> clashes) {
        log.warn("Extracting your plugin's dependencies caused the following file(s) to overwrite each other:");
        clashes.entrySet().stream()
                .sorted(comparingByKey())
                .forEach(e -> log.warn(format("-- %s from %s", e.getKey(), e.getValue())));
        log.warn("To prevent this, set <extractDependencies> to false in your AMPS configuration");
    }

    private static Path head(final Path path) {
        return path.subpath(0, 1);
    }

    private static Path tail(final Path path) {
        return path.subpath(1, path.getNameCount());
    }

    private Path createTempDirectoryForOverwriteDetection() {
        final Path targetDirectory = Paths.get(ctx.getProject().getBuild().getDirectory());
        try {
            createDirectories(targetDirectory);
            return createTempDirectory(targetDirectory, "amps-overwrite-detection-");
        } catch (final IOException e) {
            throw new UncheckedIOException(e);
        }
    }

    private void doExtractDependencies(final Xpp3Dom configuration) throws MojoExecutionException {
        mojoExecutorWrapper.executeWithMergedConfig(
                ctx.getPlugin("org.apache.maven.plugins", "maven-dependency-plugin"),
                goal("unpack-dependencies"),
                configuration,
                executionEnvironment()
        );
    }

    public void extractBundledDependencies() throws MojoExecutionException {
        extractDependencies(configuration(
                element(name("includeScope"), "runtime"),
                element(name("excludeScope"), "provided"),
                element(name("excludeScope"), "test"),
                element(name("includeTypes"), "jar"),
                element(name("excludes"), "atlassian-plugin.xml, module-info.class, META-INF/MANIFEST.MF, META-INF/*.DSA, META-INF/*.SF"),
                element(name("outputDirectory"), "${project.build.outputDirectory}")
        ));
    }

    public void extractTestBundledDependenciesExcludingTestScope(List<ProductArtifact> testBundleExcludes)
            throws MojoExecutionException {
        StringBuilder sb = new StringBuilder();

        for (ProductArtifact artifact : testBundleExcludes) {
            sb.append(",").append(artifact.getArtifactId());
        }

        String customExcludes = sb.toString();

        extractDependencies(configuration(
                element(name("includeScope"), "runtime"),
                element(name("excludeScope"), "provided"),
                element(name("excludeScope"), "test"),
                element(name("includeTypes"), "jar"),
                element(name("excludeArtifactIds"), "junit" + customExcludes),
                element(name("excludes"), "atlassian-plugin.xml, module-info.class, META-INF/MANIFEST.MF, META-INF/*.DSA, META-INF/*.SF"),
                element(name("outputDirectory"), "${project.build.testOutputDirectory}")
        ));
    }

    public void extractTestBundledDependencies(List<ProductArtifact> testBundleExcludes) throws MojoExecutionException {
        StringBuilder sb = new StringBuilder();

        for (ProductArtifact artifact : testBundleExcludes) {
            sb.append(",").append(artifact.getArtifactId());
        }

        String customExcludes = sb.toString();

        extractDependencies(configuration(
                element(name("includeScope"), "test"),
                element(name("excludeScope"), "provided"),
                element(name("excludeArtifactIds"), "junit" + customExcludes),
                element(name("includeTypes"), "jar"),
                element(name("useSubDirectoryPerScope"), "true"),
                element(name("excludes"), "atlassian-plugin.xml, module-info.class, META-INF/MANIFEST.MF, META-INF/*.DSA, META-INF/*.SF"),
                element(name("outputDirectory"), "${project.build.directory}/testlibs")
        ));

        File targetDir = new File(ctx.getProject().getBuild().getDirectory());
        File testlibsDir = new File(targetDir, "testlibs");
        File compileLibs = new File(testlibsDir, "compile");
        File testLibs = new File(testlibsDir, "test");


        File testClassesDir = new File(ctx.getProject().getBuild().getTestOutputDirectory());

        try {
            compileLibs.mkdirs();
            testLibs.mkdirs();

            FileUtils.copyDirectory(compileLibs, testClassesDir, FileFilterUtils.notFileFilter(FileFilterUtils.nameFileFilter("META-INF")));
            FileUtils.copyDirectory(testLibs, testClassesDir, FileFilterUtils.notFileFilter(FileFilterUtils.nameFileFilter("META-INF")));
        } catch (IOException e) {
            throw new MojoExecutionException("unable to copy test libs", e);
        }
    }

    public void compressResources(boolean compressJs, boolean compressCss, boolean useClosureForJs, Charset cs, Map<String, String> closureOptions) throws MojoExecutionException {
        MinifierParameters minifierParameters = new MinifierParameters(compressJs,
                compressCss,
                useClosureForJs,
                cs,
                log,
                closureOptions
        );
        new ResourcesMinifier(minifierParameters).minify(ctx.getProject().getBuild().getResources(), ctx.getProject().getBuild().getOutputDirectory());
    }

    public void filterPluginDescriptor() throws MojoExecutionException {
        mojoExecutorWrapper.executeWithMergedConfig(
                ctx.getPlugin("org.apache.maven.plugins", "maven-resources-plugin"),
                goal("copy-resources"),
                configuration(
                        element(name("encoding"), "UTF-8"),
                        element(name("resources"),
                                element(name("resource"),
                                        element(name("directory"), "src/main/resources"),
                                        element(name("filtering"), "true"),
                                        element(name("includes"),
                                                element(name("include"), "atlassian-plugin.xml"))
                                )
                        ),
                        element(name("outputDirectory"), "${project.build.outputDirectory}")
                ),
                executionEnvironment()
        );


        XmlCompressor compressor = new XmlCompressor();
        File pluginXmlFile = new File(ctx.getProject().getBuild().getOutputDirectory(), "atlassian-plugin.xml");

        if (pluginXmlFile.exists()) {
            try {
                String source = FileUtils.readFileToString(pluginXmlFile, UTF_8);
                String min = compressor.compress(source);
                FileUtils.writeStringToFile(pluginXmlFile, min, UTF_8);
            } catch (IOException e) {
                throw new MojoExecutionException("IOException while minifying plugin XML file", e);
            }
        }
    }

    public void filterTestPluginDescriptor() throws MojoExecutionException {
        mojoExecutorWrapper.executeWithMergedConfig(
                ctx.getPlugin("org.apache.maven.plugins", "maven-resources-plugin"),
                goal("copy-resources"),
                configuration(
                        element(name("encoding"), "UTF-8"),
                        element(name("resources"),
                                element(name("resource"),
                                        element(name("directory"), "src/test/resources"),
                                        element(name("filtering"), "true"),
                                        element(name("includes"),
                                                element(name("include"), "atlassian-plugin.xml"))
                                )
                        ),
                        element(name("outputDirectory"), "${project.build.testOutputDirectory}")
                ),
                executionEnvironment()
        );

    }

    /**
     * Runs the current project's unit tests via the Surefire plugin. Excludes tests whose names match the pattern
     * {@value #REGEX_INTEGRATION_TESTS} and inner classes.
     *
     * @param systemProperties see SureFire's systemPropertyVariables parameter
     * @param excludedGroups   see SureFire's excludedGroups parameter
     * @param category         see SureFire's category parameter
     * @throws MojoExecutionException if something goes wrong
     */
    public void runUnitTests(
            final Map<String, Object> systemProperties, final String excludedGroups, final String category)
            throws MojoExecutionException {
        final Element systemProps = convertPropsToElements(systemProperties);
        final Xpp3Dom config = configuration(
                systemProps,
                element(name("excludes"),
                        element(name("exclude"), REGEX_INTEGRATION_TESTS),
                        element(name("exclude"), "**/*$*")),
                element(name("excludedGroups"), excludedGroups)
        );

        if (isNotBlank(category)) {
            config.addChild(groupsElement(category));
        }

        final Plugin surefirePlugin =
                ctx.getPlugin("org.apache.maven.plugins", "maven-surefire-plugin");
        log.info("Surefire " + surefirePlugin.getVersion() + " test configuration:");
        log.info(config.toString());

        mojoExecutorWrapper.executeWithMergedConfig(
                surefirePlugin,
                goal("test"),
                config,
                executionEnvironment()
        );
    }

    public File copyWebappWar(final String productId, final File targetDirectory, final ProductArtifact artifact)
            throws MojoExecutionException {
        final File webappWarFile = new File(targetDirectory, productId + "-original.war");
        mojoExecutorWrapper.executeWithMergedConfig(
                ctx.getPlugin("org.apache.maven.plugins", "maven-dependency-plugin"),
                goal("copy"),
                configuration(
                        element(name("artifactItems"),
                                element(name("artifactItem"),
                                        element(name("groupId"), artifact.getGroupId()),
                                        element(name("artifactId"), artifact.getArtifactId()),
                                        element(name("type"), "war"),
                                        element(name("version"), artifact.getVersion()),
                                        element(name("destFileName"), webappWarFile.getName()))),
                        element(name("outputDirectory"), targetDirectory.getPath())
                ),
                executionEnvironment()
        );
        return webappWarFile;
    }

    public void unpackWebappWar(final File targetDirectory, final ProductArtifact artifact)
            throws MojoExecutionException {
        mojoExecutorWrapper.executeWithMergedConfig(
                ctx.getPlugin("org.apache.maven.plugins", "maven-dependency-plugin"),
                goal("unpack"),
                configuration(
                        element(name("artifactItems"),
                                element(name("artifactItem"),
                                        element(name("groupId"), artifact.getGroupId()),
                                        element(name("artifactId"), artifact.getArtifactId()),
                                        element(name("type"), "war"),
                                        element(name("version"), artifact.getVersion()))),
                        element(name("outputDirectory"), targetDirectory.getPath()),
                        //Overwrite needs to be enabled or unpack won't unpack multiple copies
                        //of the same dependency GAV, _even to different output directories_
                        element(name("overWriteReleases"), "true"),
                        element(name("overWriteSnapshots"), "true"),
                        //Use the JVM's chmod; it's faster than forking
                        element(name("useJvmChmod"), "true")
                ),
                executionEnvironment()
        );
    }

    /**
     * Copies {@code artifacts} to the {@code outputDirectory}. Artifacts are looked up in this order:
     *
     * <ol>
     *     <li>in the Maven reactor</li>
     *     <li>in the Maven repositories</li>
     * </ol>
     * <p>
     * This can't be used in a goal that happens before the <em>package</em> phase, as artifacts in the reactor will not
     * be packaged (and therefore copyable) until this phase.
     *
     * @param outputDirectory the directory to copy artifacts to
     * @param artifacts       the list of artifact to copy to the given directory
     */
    public void copyPlugins(final File outputDirectory, final List<ProductArtifact> artifacts)
            throws MojoExecutionException {
        for (ProductArtifact artifact : artifacts) {
            final MavenProject artifactReactorProject = getReactorProjectForArtifact(artifact);
            if (artifactReactorProject != null) {

                log.debug(artifact + " will be copied from reactor project " + artifactReactorProject);
                final File artifactFile = artifactReactorProject.getArtifact().getFile();
                if (artifactFile == null) {
                    log.warn("The plugin " + artifact + " is in the reactor but not the file hasn't been attached.  Skipping.");
                } else {
                    log.debug("Copying " + artifactFile + " to " + outputDirectory);
                    try {
                        FileUtils.copyFile(artifactFile, new File(outputDirectory, artifactFile.getName()));
                    } catch (IOException e) {
                        throw new MojoExecutionException("Could not copy " + artifact + " to " + outputDirectory, e);
                    }
                }

            } else {
                mojoExecutorWrapper.executeWithMergedConfig(
                        ctx.getPlugin("org.apache.maven.plugins", "maven-dependency-plugin"),
                        goal("copy"),
                        configuration(
                                element(name("artifactItems"),
                                        element(name("artifactItem"),
                                                element(name("groupId"), artifact.getGroupId()),
                                                element(name("artifactId"), artifact.getArtifactId()),
                                                element(name("version"), artifact.getVersion()))),
                                element(name("outputDirectory"), outputDirectory.getPath())
                        ),
                        executionEnvironment());
            }
        }
    }

    private MavenProject getReactorProjectForArtifact(ProductArtifact artifact) {
        for (final MavenProject project : ctx.getReactor()) {
            if (project.getGroupId().equals(artifact.getGroupId())
                    && project.getArtifactId().equals(artifact.getArtifactId())
                    && project.getVersion().equals(artifact.getVersion())) {
                return project;
            }
        }
        return null;
    }

    private void unpackContainer(final Container container) throws MojoExecutionException {
        mojoExecutorWrapper.executeWithMergedConfig(
                ctx.getPlugin("org.apache.maven.plugins", "maven-dependency-plugin"),
                goal("unpack"),
                configuration(
                        element(name("artifactItems"),
                                element(name("artifactItem"),
                                        element(name("groupId"), container.getGroupId()),
                                        element(name("artifactId"), container.getArtifactId()),
                                        element(name("version"), container.getVersion()),
                                        element(name("classifier"), container.getClassifier()),
                                        element(name("type"), "zip"))),  // see AMPS-1527
                        element(name("outputDirectory"), container.getRootDirectory(getBuildDirectory()))
                ),
                executionEnvironment());
    }

    private String getBuildDirectory() {
        return ctx.getProject().getBuild().getDirectory();
    }

    private static Xpp3Dom configurationWithoutNullElements(Element... elements) {
        return configuration(removeNullElements(elements));
    }

    private static Element[] removeNullElements(final Element... elements) {
        return Arrays.stream(elements)
                .filter(Objects::nonNull)
                .toArray(Element[]::new);
    }

    private Plugin bndPlugin() {
        final Plugin bndPlugin = ctx.getPlugin("org.apache.felix", "maven-bundle-plugin");
        log.info("using maven-bundle-plugin v" + bndPlugin.getVersion());
        return bndPlugin;
    }

    private ContainerConfig getContainerConfig(final Product product) throws MojoExecutionException {
        try {
            return new ContainerConfig(
                    product.resolve(product.getAjpPort(), "AJP"),
                    product.resolve(product.getRmiPort(), "RMI"),
                    product.resolve(product.getWebPort(), "HTTP"),
                    product.getProtocol());
        } catch (MojoExecutionException e) {
            log.error(e.getMessage());
            throw e;
        }
    }

    /**
     * The network configuration of the Cargo container.
     */
    static class ContainerConfig {

        private final int ajpPort;
        private final int rmiPort;
        private final int webPort;
        private final String protocol;

        private ContainerConfig(final int ajpPort, final int rmiPort, final int webPort, final String protocol) {
            this.ajpPort = ajpPort;
            this.rmiPort = rmiPort;
            this.webPort = webPort;
            this.protocol = requireNonNull(protocol);
        }

        public int getAjpPort() {
            return ajpPort;
        }

        public int getRmiPort() {
            return rmiPort;
        }

        public int getWebPort() {
            return webPort;
        }

        public String getProtocol() {
            return protocol;
        }
    }

    /**
     * Uses the {@code cargo-maven2-plugin:start} goal to start the servlet container and deploy the given WAR file to
     * it.
     *
     * @param productWar                 the product's WAR file, to deploy to the servlet container
     * @param systemProperties           any system properties to be passed to the servlet container; these will also be passed to
     *                                   Cargo as <a href="https://codehaus-cargo.github.io/cargo/Configuration+properties.html">
     *                                   configuration properties</a>
     * @param extraContainerDependencies any extra artifacts to deploy to the container's own {@code lib} directory
     * @param extraProductDeployables    any artifacts to deploy to the container, in addition to the given WAR
     * @param product                    the product being deployed
     * @return the HTTP/HTTPS port of the deployed product
     * @throws MojoExecutionException if something goes wrong
     */
    public int startWebapp(final File productWar, final Map<String, String> systemProperties,
                           final List<ProductArtifact> extraContainerDependencies,
                           final List<ProductArtifact> extraProductDeployables,
                           final Product product) throws MojoExecutionException {
        final Container container = getContainerFromWebappContext(product);
        final ContainerConfig containerConfig = getContainerConfig(product);

        if (!container.isEmbedded()) {
            unpackOrReuse(container);
        }

        log.info("Starting " + product.getInstanceId() + " on the " + container.getId() +
                " container on ports " + containerConfig.getWebPort() + " (" + containerConfig.getProtocol() + "), " +
                containerConfig.getRmiPort() + " (rmi) and " + containerConfig.getAjpPort() + " (ajp)");

        mojoExecutorWrapper.execute(
                cargoPlugin(),
                goal("start"),
                getCargoStartConfiguration(productWar, systemProperties, extraContainerDependencies,
                        extraProductDeployables, product, container, containerConfig),
                executionEnvironment()
        );

        return containerConfig.getWebPort();
    }

    private Xpp3Dom getCargoStartConfiguration(
            final File productWar, final Map<String, String> systemProperties,
            final List<ProductArtifact> extraContainerDependencies, final List<ProductArtifact> extraProductDeployables,
            final Product product, final Container container, final ContainerConfig containerConfig)
            throws MojoExecutionException {
        return configurationWithoutNullElements(
                element(name("deployables"), getDeployables(productWar, extraProductDeployables, product)),
                element(name("container"),
                        element(name("containerId"), container.getId()),
                        element(name("type"), container.getType()),
                        element(name("home"), container.getInstallDirectory(getBuildDirectory())),
                        element(name("output"), product.getOutput()),
                        element(name("systemProperties"),
                                getSystemProperties(systemProperties, product, containerConfig)),
                        element(name("dependencies"), getContainerDependencies(extraContainerDependencies, product)),
                        element(name("timeout"), String.valueOf(getStartupTimeout(product)))
                ),
                element(name("configuration"), removeNullElements(
                        element(name("configfiles"), getExtraContainerConfigurationFiles()),
                        element(name("home"),
                                container.getConfigDirectory(getBuildDirectory(), product.getInstanceId())),
                        element(name("type"), "standalone"),
                        element(name("properties"),
                                getConfigurationProperties(systemProperties, product, containerConfig)),
                        xmlReplacementsElement(product.getCargoXmlOverrides()))
                ),
                // Fix issue AMPS copy 2 War files to container
                // Refer to Cargo documentation: when project's packaging is war, ear, ejb
                // the generated artifact will automatic copy to target container.
                // For avoiding this behavior, just add an empty <deployer/> element
                element(name("deployer"))
        );
    }

    private void unpackOrReuse(final Container container) throws MojoExecutionException {
        final File containerDir = new File(container.getInstallDirectory(getBuildDirectory()));
        if (containerDir.exists()) {
            log.info("Reusing unpacked container '" + container.getId() + "' from " + containerDir.getPath());
        } else {
            log.info("Unpacking container '" + container.getId() + "' from container artifact: " + container);
            unpackContainer(container);
        }
    }

    private Element[] getSystemProperties(final Map<String, String> systemProperties,
                                          final Product product, final ContainerConfig containerConfig) {
        final List<Element> sysPropElements = new ArrayList<>();
        systemProperties.forEach((key, value) -> sysPropElements.add(element(name(key), value)));

        final String baseUrl = getBaseUrl(product, containerConfig.getWebPort());
        sysPropElements.add(element(name("baseurl"), baseUrl));
        return sysPropElements.toArray(new Element[0]);
    }

    private int getStartupTimeout(final Product product) {
        if (FALSE.equals(product.getSynchronousStartup())) {
            return 0;
        }
        return product.getStartupTimeout();
    }

    private Element[] getDeployables(
            final File productWar, final List<ProductArtifact> extraDeployables, final Product webappContext) {
        final List<Element> deployables = new ArrayList<>();
        // Add the given product WAR file
        deployables.add(element(name("deployable"),
                element(name("groupId"), "foo"),
                element(name("artifactId"), "bar"),
                element(name("type"), "war"),
                element(name("location"), productWar.getPath()),
                element(name("properties"),
                        element(name("context"), webappContext.getContextPath())
                )
        ));

        for (final ProductArtifact artifact : extraDeployables) {
            deployables.add(element(name("deployable"),
                    element(name("groupId"), artifact.getGroupId()),
                    element(name("artifactId"), artifact.getArtifactId()),
                    element(name("type"), artifact.getType()),
                    element(name("location"), artifact.getPath())
            ));
        }
        return deployables.toArray(new Element[0]);
    }

    private Container getContainerFromWebappContext(Product webappContext) {
        if (webappContext.getCustomContainerArtifact() == null) {
            return findContainer(webappContext.getContainerId());
        } else {
            return convertCustomContainerStringToContainerObject(webappContext);
        }
    }

    private Container convertCustomContainerStringToContainerObject(final Product product) {
        final String customContainerArtifactStr = product.getCustomContainerArtifact().trim();
        final String[] containerData = customContainerArtifactStr.split(":");
        if (containerData.length < 3 || containerData.length > 5) {
            throw new IllegalArgumentException(format("Container artifact string must have the format" +
                    " groupId:artifactId:version[:packaging:classifier] or groupId:artifactId:version:classifier;" +
                    " actual string = '%s'", customContainerArtifactStr));
        }
        final String cargoContainerId = findContainer(product.getContainerId()).getId();
        final String groupId = containerData[0];
        final String artifactId = containerData[1];
        final String version = containerData[2];
        switch (containerData.length) {
            case 3:
                return new Container(cargoContainerId, groupId, artifactId, version);
            case 4:
                return new Container(cargoContainerId, groupId, artifactId, version, containerData[3]);
            case 5:
                // cause unpack have hardcoded packaging
                return new Container(cargoContainerId, groupId, artifactId, version, containerData[4]);
            default:
                // We checked the array length earlier, this should never happen
                throw new IllegalStateException(format("Unexpected container data %s", Arrays.toString(containerData)));
        }
    }

    // See https://codehaus-cargo.github.io/cargo/Configuration+files+option.html
    private Element[] getExtraContainerConfigurationFiles() throws MojoExecutionException {
        return new Element[]{
                // For AMPS-1429, apply a custom context.xml with a correctly configured JarScanFilter
                element("configfile",
                        element("file", getContextXml().getAbsolutePath()),
                        element("todir", "conf"),
                        element("tofile", "context.xml"),
                        element("configfile", "true")
                )
        };
    }

    /**
     * Returns the <code>context.xml</code> file to be copied into the Tomcat instance.
     *
     * @return an extant file
     */
    private File getContextXml() throws MojoExecutionException {
        try {
            // Because Cargo needs an absolute file path, we copy context.xml from the AMPS JAR to a temp file
            final File tempContextXml = createTempFile("context.xml", null);
            final InputStream contextXmlToCopy = requireNonNull(getClass().getResourceAsStream("context.xml"));
            copyInputStreamToFile(contextXmlToCopy, tempContextXml);
            return tempContextXml;
        } catch (IOException e) {
            throw new MojoExecutionException("Failed to create Tomcat context.xml", e);
        }
    }

    private Element[] getContainerDependencies(
            final List<ProductArtifact> extraContainerDependencies, final Product product)
            throws MojoExecutionException {
        final List<Element> dependencyElements = new ArrayList<>();
        final List<ProductArtifact> containerDependencies = new ArrayList<>(extraContainerDependencies);
        product.getDataSources().forEach(ds -> containerDependencies.addAll(ds.getLibArtifacts()));
        for (final ProductArtifact containerDependency : containerDependencies) {
            dependencyElements.add(element(name("dependency"),
                    element(name("location"), product.getArtifactRetriever().resolve(containerDependency))
            ));
        }
        return dependencyElements.toArray(new Element[0]);
    }

    @Nullable
    private Element xmlReplacementsElement(final Collection<XmlOverride> cargoXmlOverrides) {
        if (cargoXmlOverrides == null) {
            return null;
        }

        final Element[] xmlReplacementsElements = cargoXmlOverrides.stream().map(xmlOverride ->
                element(name("xmlReplacement"),
                        element(name("file"), xmlOverride.getFile()),
                        element(name("xpathExpression"), xmlOverride.getxPathExpression()),
                        element(name("attributeName"), xmlOverride.getAttributeName()),
                        element(name("value"), xmlOverride.getValue()))
        ).toArray(Element[]::new);

        return element(name("xmlReplacements"), xmlReplacementsElements);
    }

    /**
     * Returns the <a href="https://codehaus-cargo.github.io/cargo/Configuration+properties.html">properties</a> to
     * apply to the Cargo configuration.
     *
     * @param systemProperties any user-provided properties
     * @param product          the product being deployed
     * @param containerConfig  the container configuration
     * @return XML elements whose names are the property keys and value are the property values
     */
    @VisibleForTesting
    Element[] getConfigurationProperties(final Map<String, String> systemProperties, final Product product,
                                         final ContainerConfig containerConfig) {
        final List<Element> props = new ArrayList<>();
        for (final Entry<String, String> entry : systemProperties.entrySet()) {
            props.add(element(name(entry.getKey()), entry.getValue()));
        }
        props.add(element(name("cargo.servlet.port"), String.valueOf(containerConfig.getWebPort())));

        if (TRUE.equals(product.getUseHttps())) {
            log.debug("starting tomcat using Https via cargo with the following parameters:");
            log.debug("cargo.servlet.port = " + containerConfig.getWebPort());
            log.debug("cargo.protocol = " + containerConfig.getProtocol());
            props.add(element(name("cargo.protocol"), containerConfig.getProtocol()));

            log.debug("cargo.tomcat.connector.clientAuth = " + product.getHttpsClientAuth());
            props.add(element(name("cargo.tomcat.connector.clientAuth"), product.getHttpsClientAuth()));

            log.debug("cargo.tomcat.connector.sslProtocol = " + product.getHttpsSSLProtocol());
            props.add(element(name("cargo.tomcat.connector.sslProtocol"), product.getHttpsSSLProtocol()));

            log.debug("cargo.tomcat.connector.keystoreFile = " + product.getHttpsKeystoreFile());
            props.add(element(name("cargo.tomcat.connector.keystoreFile"), product.getHttpsKeystoreFile()));

            log.debug("cargo.tomcat.connector.keystorePass = " + product.getHttpsKeystorePass());
            props.add(element(name("cargo.tomcat.connector.keystorePass"), product.getHttpsKeystorePass()));

            log.debug("cargo.tomcat.connector.keyAlias = " + product.getHttpsKeyAlias());
            props.add(element(name("cargo.tomcat.connector.keyAlias"), product.getHttpsKeyAlias()));

            log.debug("cargo.tomcat.httpSecure = " + product.getHttpsHttpSecure());
            props.add(element(name("cargo.tomcat.httpSecure"), product.getHttpsHttpSecure().toString()));
        }

        props.add(element(name(AJP_PORT_PROPERTY), String.valueOf(containerConfig.getAjpPort())));
        props.add(element(name("cargo.rmi.port"), String.valueOf(containerConfig.getRmiPort())));
        props.add(element(name("cargo.jvmargs"), product.getJvmArgs() + product.getDebugArgs()));
        return props.toArray(new Element[0]);
    }

    public void stopWebapp(final String productId, final String containerId, final Product webappContext)
            throws MojoExecutionException {
        final Container container = getContainerFromWebappContext(webappContext);

        final String actualShutdownTimeout = webappContext.getSynchronousStartup() ? "0" : String.valueOf(webappContext.getShutdownTimeout());

        mojoExecutorWrapper.execute(
                cargoPlugin(),
                goal("stop"),
                configuration(
                        element(name("container"),
                                element(name("containerId"), container.getId()),
                                element(name("type"), container.getType()),
                                element(name("timeout"), actualShutdownTimeout),
                                // org.codehaus.cargo
                                element(name("home"), container.getInstallDirectory(getBuildDirectory()))
                        ),
                        element(name("configuration"),
                                // org.twdata.maven
                                element(name("home"), container.getConfigDirectory(getBuildDirectory(), productId)),
                                //we don't need that atm. since timeout is 0 for org.codehaus.cargo
                                //hoping this will fix AMPS-987
                                element(name("properties"), createShutdownPortsPropertiesConfiguration(webappContext))
                        )
                ),
                executionEnvironment()
        );
    }

    /**
     * Cargo waits (AbstractCatalinaInstalledLocalContainer#waitForCompletion(boolean)) for the HTTP and AJP ports, to
     * close before it decides the container is stopped.
     * <p>
     * Since {@link #startWebapp} can use random ports for HTTP or AJP, it's possible the container isn't using the
     * ports defined on the {@link Product}. For the HTTP port we just accept that risk, on the assumption it's likely
     * to be a minority case. For the AJP port, rather than waiting for {@link Product#getAjpPort}, we configure it to
     * use the HTTP port as the AJP port.
     * <p>
     * Note that the RMI port <i>is intentionally not configured here</i>. Earlier versions of AMPS set the HTTP port
     * as the RMI port as well, but <a href="https://codehaus-cargo.atlassian.net/browse/CARGO-1337">CARGO-1337</a>
     * resulted in a change in Cargo that causes it to no longer wait for the RMI port to stop. That means if we set
     * the AJP and HTTP ports to a value that matches the RMI port Cargo doesn't wait for <i>any</i> of the sockets
     * to be closed and just immediately concludes the container has stopped.
     */
    private Element[] createShutdownPortsPropertiesConfiguration(final Product webappContext) {
        final String webPort = String.valueOf(webappContext.getWebPort());
        final List<Element> properties = new ArrayList<>();
        properties.add(element(name("cargo.servlet.port"), webPort));
        properties.add(element(name(AJP_PORT_PROPERTY), webPort));
        return properties.toArray(new Element[0]);
    }

    /**
     * Returns the Cargo plugin to be used.
     */
    private Plugin cargoPlugin() {
        final Plugin cargoPlugin = ctx.getPlugin("org.codehaus.cargo", "cargo-maven2-plugin");
        log.info("using codehaus cargo v" + cargoPlugin.getVersion());
        return cargoPlugin;
    }

    public static String getBaseUrl(final Product product, final int actualHttpPort) {
        return getBaseUrl(product.getServer(), actualHttpPort, product.getContextPath());
    }

    private static String getBaseUrl(String server, final int actualHttpPort, String contextPath) {
        final String port = actualHttpPort != 80 ? ":" + actualHttpPort : "";
        server = server.startsWith("http") ? server : "http://" + server;
        if (!contextPath.startsWith("/") && isNotBlank(contextPath)) {
            contextPath = "/" + contextPath;
        }
        return server + port + contextPath;
    }

    public static String getReportsDirectory(
            final File targetDirectory, final String testGroupId, final String containerId) {
        return targetDirectory.getAbsolutePath() + "/" + testGroupId + "/" + containerId + "/surefire-reports";
    }

    /**
     * Runs the plugin's integration tests by invoking {@code maven-failsafe-plugin:integration-test}.
     *
     * @param reportsDirectory the directory in which to write the test reports
     * @param includes         the filename patterns of the tests to be run, e.g. "&#8727;&#8727;/IT&#8727;.java"
     * @param excludes         the filename patterns of the tests <em>not</em> to be run
     * @param systemProperties the System properties to pass to the JUnit tests
     * @param category         if specified, only classes/methods/etc decorated with one of these groups/categories/tags will be
     *                         included in test run. In JUnit4, this affects tests annotated with the
     *                         {@link org.junit.experimental.categories.Category Category} annotation.
     * @param debug            see the {@code debugForkedProcess} parameter of the {@code failsafe:integration-test} goal
     * @throws MojoExecutionException if something goes wrong, or any tests fail
     */
    public void runIntegrationTests(
            final String reportsDirectory, final List<String> includes, final List<String> excludes,
            final Map<String, Object> systemProperties, final String category, @Nullable final String debug)
            throws MojoExecutionException {
        final Xpp3Dom integrationTestConfig =
                getIntegrationTestConfig(includes, excludes, systemProperties, category, debug, reportsDirectory);
        final Plugin failsafePlugin =
                ctx.getPlugin("org.apache.maven.plugins", "maven-failsafe-plugin");
        log.info("Failsafe " + failsafePlugin.getVersion() + " integration-test configuration:");
        log.info(integrationTestConfig.toString());
        mojoExecutorWrapper.executeWithMergedConfig(
                failsafePlugin,
                goal("integration-test"),
                integrationTestConfig,
                executionEnvironment()
        );
    }

    /**
     * Runs {@code maven-failsafe-plugin:verify}.
     *
     * @param reportsDirectory the directory containing the test reports
     * @throws MojoExecutionException if something goes wrong or a test failed
     */
    public void runVerify(final String reportsDirectory)
            throws MojoExecutionException {
        mojoExecutorWrapper.executeWithMergedConfig(
                ctx.getPlugin("org.apache.maven.plugins", "maven-failsafe-plugin"),
                goal("verify"),
                configuration(element(name(REPORTS_DIRECTORY), reportsDirectory)),
                executionEnvironment()
        );
    }

    private Xpp3Dom getIntegrationTestConfig(
            final List<String> includes, final List<String> excludes, final Map<String, Object> systemProperties,
            final String category, final @Nullable String debug, final String reportsDirectory) {
        final Element[] includeElements = includes.stream()
                .map(include -> element(name("include"), include))
                .toArray(Element[]::new);

        systemProperties.put(REPORTS_DIRECTORY, reportsDirectory);
        final Element systemPropsElement = convertPropsToElements(systemProperties);

        final Xpp3Dom config = configuration(
                element(name("includes"), includeElements),
                element(name("excludes"), getTestExcludes(excludes)),
                systemPropsElement,
                element(name(REPORTS_DIRECTORY), reportsDirectory)
        );

        if (debug != null) {
            config.addChild(element(name("debugForkedProcess"), debug).toDom());
        }

        if (isNotBlank(category)) {
            config.addChild(groupsElement(category));
        }

        return config;
    }

    private Element[] getTestExcludes(final List<String> userSpecifiedExcludes) {
        final Set<String> allExcludes = new HashSet<>(userSpecifiedExcludes);
        allExcludes.add(ABSTRACT_CLASSES);
        allExcludes.add(INNER_CLASSES);
        return allExcludes.stream()
                .map(exclude -> element(name("exclude"), exclude))
                .toArray(Element[]::new);
    }

    private Xpp3Dom groupsElement(final String category) {
        return element(name("groups"), category).toDom();
    }

    /**
     * Converts a map of System properties to a Maven {@code systemPropertyVariables} config element.
     */
    private Element convertPropsToElements(final Map<String, Object> systemProperties) {
        final List<Element> properties = new ArrayList<>();
        for (Entry<String, Object> entry : systemProperties.entrySet()) {
            log.info("adding system property to configuration: " + entry.getKey() + "::" + entry.getValue());

            properties.add(element(name(entry.getKey()), entry.getValue().toString()));
        }

        return element(name("systemPropertyVariables"), properties.toArray(new Element[0]));
    }

    private Container findContainer(final String containerId) {
        final Container container = idToContainerMap.get(containerId);
        if (container == null) {
            throw new IllegalArgumentException("Container " + containerId + " not supported");
        }
        return container;
    }

    /**
     * Installs a P2 plugin into a running Atlassian product.
     *
     * @param pdkParams describes the intended host product and the plugin to be uploaded
     * @throws MojoExecutionException if the installation fails
     */
    public void installPlugin(PdkParams pdkParams) throws MojoExecutionException {
        final String baseUrl = getBaseUrl(pdkParams.getServer(), pdkParams.getPort(), pdkParams.getContextPath());
        // We delegate the plugin installation to https://bitbucket.org/atlassian/atlassian-pdk-maven-plugin
        mojoExecutorWrapper.executeWithMergedConfig(
                ctx.getPlugin("com.atlassian.maven.plugins", "atlassian-pdk"),
                goal("install"),
                configuration(
                        element(name("pluginFile"), pdkParams.getPluginFile()),
                        element(name("username"), pdkParams.getUsername()),
                        element(name("password"), pdkParams.getPassword()),
                        element(name("serverUrl"), baseUrl),
                        element(name("pluginKey"), pdkParams.getPluginKey())
                ),
                executionEnvironment()
        );
    }

    public void installIdeaPlugin() throws MojoExecutionException {
        // See https://github.com/atlassian/maven-cli-plugin/blob/master/maven/src/main/java/org/twdata/maven/cli/IdeaMojo.java
        mojoExecutorWrapper.executeWithMergedConfig(
                ctx.getPlugin("org.twdata.maven", "maven-cli-plugin"),
                goal("idea"),
                configuration(),
                executionEnvironment()
        );
    }

    public File copyDist(final File targetDirectory, final ProductArtifact artifact) throws MojoExecutionException {
        return copyZip(targetDirectory, artifact, "test-dist.zip");
    }

    public File copyHome(final File targetDirectory, final ProductArtifact artifact) throws MojoExecutionException {
        return copyZip(targetDirectory, artifact, artifact.getArtifactId() + ".zip");
    }

    public File copyZip(final File targetDirectory, final ProductArtifact artifact, final String localName) throws MojoExecutionException {
        final File artifactZip = new File(targetDirectory, localName);
        mojoExecutorWrapper.executeWithMergedConfig(
                ctx.getPlugin("org.apache.maven.plugins", "maven-dependency-plugin"),
                goal("copy"),
                configuration(
                        element(name("artifactItems"),
                                element(name("artifactItem"),
                                        element(name("groupId"), artifact.getGroupId()),
                                        element(name("artifactId"), artifact.getArtifactId()),
                                        element(name("type"), "zip"),
                                        element(name("version"), artifact.getVersion()),
                                        element(name("destFileName"), artifactZip.getName()))),
                        element(name("outputDirectory"), artifactZip.getParent())
                ),
                executionEnvironment()
        );
        return artifactZip;
    }

    public void generateBundleManifest(final Map<String, String> instructions, final Map<String, String> basicAttributes) throws MojoExecutionException {
        final List<Element> instlist = new ArrayList<>();
        for (final Entry<String, String> entry : instructions.entrySet()) {
            instlist.add(element(entry.getKey(), entry.getValue()));
        }
        if (!instructions.containsKey(Constants.IMPORT_PACKAGE)) {
            instlist.add(element(Constants.IMPORT_PACKAGE, "*;resolution:=optional"));
            // BND will expand the wildcard to a list of actually-used packages, but this tells it to mark
            // them all as optional
        }
        for (final Entry<String, String> entry : basicAttributes.entrySet()) {
            instlist.add(element(entry.getKey(), entry.getValue()));
        }
        mojoExecutorWrapper.executeWithMergedConfig(
                bndPlugin(),
                goal("manifest"),
                configuration(
                        element(name("supportedProjectTypes"),
                                element(name("supportedProjectType"), "jar"),
                                element(name("supportedProjectType"), "bundle"),
                                element(name("supportedProjectType"), "war"),
                                element(name("supportedProjectType"), "atlassian-plugin")),
                        element(name("instructions"), instlist.toArray(new Element[0]))
                ),
                executionEnvironment()
        );
    }

    public void generateTestBundleManifest(final Map<String, String> instructions, final Map<String, String> basicAttributes) throws MojoExecutionException {
        final List<Element> instlist = new ArrayList<>();
        for (final Entry<String, String> entry : instructions.entrySet()) {
            instlist.add(element(entry.getKey(), entry.getValue()));
        }
        if (!instructions.containsKey(Constants.IMPORT_PACKAGE)) {
            instlist.add(element(Constants.IMPORT_PACKAGE, "*;resolution:=optional"));
            // BND will expand the wildcard to a list of actually-used packages, but this tells it to mark
            // them all as optional
        }
        for (final Entry<String, String> entry : basicAttributes.entrySet()) {
            instlist.add(element(entry.getKey(), entry.getValue()));
        }
        mojoExecutorWrapper.executeWithMergedConfig(
                bndPlugin(),
                goal("manifest"),
                configuration(
                        element(name("manifestLocation"), "${project.build.testOutputDirectory}/META-INF"),
                        element(name("supportedProjectTypes"),
                                element(name("supportedProjectType"), "jar"),
                                element(name("supportedProjectType"), "bundle"),
                                element(name("supportedProjectType"), "war"),
                                element(name("supportedProjectType"), "atlassian-plugin")),
                        element(name("instructions"), instlist.toArray(new Element[0]))
                ),
                executionEnvironment()
        );
    }

    public void generateMinimalManifest(final Map<String, String> basicAttributes) throws MojoExecutionException {
        File metaInf = file(ctx.getProject().getBuild().getOutputDirectory(), "META-INF");
        if (!metaInf.exists()) {
            metaInf.mkdirs();
        }
        File mf = file(ctx.getProject().getBuild().getOutputDirectory(), "META-INF", "MANIFEST.MF");
        Manifest m = new Manifest();
        m.getMainAttributes().putValue("Manifest-Version", "1.0");
        for (Entry<String, String> entry : basicAttributes.entrySet()) {
            m.getMainAttributes().putValue(entry.getKey(), entry.getValue());
        }

        try (FileOutputStream fos = new FileOutputStream(mf)) {
            m.write(fos);
        } catch (IOException e) {
            throw new MojoExecutionException("Unable to create manifest", e);
        }
    }

    public void generateTestMinimalManifest(final Map<String, String> basicAttributes) throws MojoExecutionException {
        File metaInf = file(ctx.getProject().getBuild().getTestOutputDirectory(), "META-INF");
        if (!metaInf.exists()) {
            metaInf.mkdirs();
        }
        File mf = file(ctx.getProject().getBuild().getTestOutputDirectory(), "META-INF", "MANIFEST.MF");
        Manifest m = new Manifest();
        m.getMainAttributes().putValue("Manifest-Version", "1.0");
        for (Entry<String, String> entry : basicAttributes.entrySet()) {
            m.getMainAttributes().putValue(entry.getKey(), entry.getValue());
        }

        try (FileOutputStream fos = new FileOutputStream(mf)) {
            m.write(fos);
        } catch (IOException e) {
            throw new MojoExecutionException("Unable to create manifest", e);
        }
    }

    /**
     * JARs up the current project.
     *
     * @param manifestExists whether there is a {@code target/classes/META-INF/MANIFEST.MF} file to include in the JAR.
     * @throws MojoExecutionException if the operation fails
     */
    public void jarWithOptionalManifest(final boolean manifestExists) throws MojoExecutionException {
        Element[] archive = new Element[0];
        if (manifestExists) {
            archive = new Element[]{element(name("manifestFile"), "${project.build.outputDirectory}/META-INF/MANIFEST.MF")};
        }

        mojoExecutorWrapper.executeWithMergedConfig(
                ctx.getPlugin("org.apache.maven.plugins", "maven-jar-plugin"),
                goal("jar"),
                configuration(
                        element(name("archive"), archive)
                ),
                executionEnvironment()
        );

    }

    public void jarTests(String finalName) throws MojoExecutionException {
        mojoExecutorWrapper.executeWithMergedConfig(
                ctx.getPlugin("org.apache.maven.plugins", "maven-jar-plugin"),
                goal("test-jar"),
                configuration(
                        element(name("finalName"), finalName),
                        element(name("archive"),
                                element(name("manifestFile"), "${project.build.testOutputDirectory}/META-INF/MANIFEST.MF"))
                ),
                executionEnvironment()
        );
    }

    public void generateObrXml(File dep, File obrXml) throws MojoExecutionException {
        mojoExecutorWrapper.executeWithMergedConfig(
                bndPlugin(),
                goal("install-file"),
                configuration(
                        element(name("obrRepository"), obrXml.getPath()),

                        // the following three settings are required but not really used
                        element(name("groupId"), "doesntmatter"),
                        element(name("artifactId"), "doesntmatter"),
                        element(name("version"), "doesntmatter"),

                        element(name("packaging"), "jar"),
                        element(name("file"), dep.getPath())

                ),
                executionEnvironment()
        );
    }

    /**
     * Adds the file to the artifacts of this build.
     * The artifact will be deployed using the name and version of the current project,
     * as in if your artifactId is 'MyProject', it will be MyProject-1.0-SNAPSHOT.jar,
     * overriding any artifact created at compilation time.
     * <p>
     * Attached artifacts get installed (at install phase) and deployed (at deploy phase)
     *
     * @param file the file
     * @param type the type of the file, default 'jar'
     */
    public void attachArtifact(File file, String type) throws MojoExecutionException {
        mojoExecutorWrapper.executeWithMergedConfig(
                ctx.getPlugin("org.codehaus.mojo", "build-helper-maven-plugin"),
                goal("attach-artifact"),
                configuration(
                        element(name("artifacts"),
                                element(name("artifact"),
                                        element(name("file"), file.getAbsolutePath()),
                                        element(name("type"), type)
                                )
                        )
                ),
                executionEnvironment());
    }

    /**
     * Performs a Maven release of the current project.
     *
     * @param mavenArgs the {@code arguments} (if any) to pass to the {@code maven-release-plugin}
     * @throws MojoExecutionException if the release fails
     */
    public void release(String mavenArgs) throws MojoExecutionException {
        String args = "";

        if (isNotBlank(mavenArgs)) {
            args = mavenArgs;
        }

        mojoExecutorWrapper.executeWithMergedConfig(
                ctx.getPlugin("org.apache.maven.plugins", "maven-release-plugin"),
                goal("prepare"),
                configuration(
                        element(name("arguments"), args)
                        , element(name("autoVersionSubmodules"), "true")
                ),
                executionEnvironment()
        );

        mojoExecutorWrapper.executeWithMergedConfig(
                ctx.getPlugin("org.apache.maven.plugins", "maven-release-plugin"),
                goal("perform"),
                configuration(
                        element(name("arguments"), args),
                        element(name("useReleaseProfile"), "true")
                ),
                executionEnvironment()
        );
    }

    /**
     * Starts the remotable plugins container in debug mode (on port 5004) and loads the given plugin into it.
     *
     * @param pluginFile the plugin to load
     * @throws MojoExecutionException if something goes wrong
     */
    public void debugStandaloneContainer(final File pluginFile) throws MojoExecutionException {
        final String pluginResourceDirs = getContextProject().getResources().stream()
                .map(FileSet::getDirectory)
                .collect(joining(","));

        mojoExecutorWrapper.executeWithMergedConfig(
                ctx.getPlugin("org.codehaus.mojo", "exec-maven-plugin"),
                goal("exec"),
                configuration(
                        element(name("executable"), "java"),
                        element(name("arguments"),
                                element(name("argument"), "-Datlassian.dev.mode=true"),
                                element(name("argument"), "-Datlassian.allow.insecure.url.parameter.login=true"),
                                element(name("argument"), "-Dplugin.resource.directories=" + pluginResourceDirs),
                                element(name("argument"), "-Xdebug"),
                                element(name("argument"), "-Xrunjdwp:transport=dt_socket,server=y,suspend=n,address=5004"),
                                element(name("argument"), "-jar"),
                                element(name("argument"), "${project.build.directory}/remotable-plugins-container-standalone.jar"),
                                element(name("argument"), pluginFile.getPath()))
                ),
                executionEnvironment()
        );
    }

    public File generateEffectivePom(ProductArtifact artifact, File parentDir) throws MojoExecutionException {
        File effectivePom = new File(parentDir, "effectivePom.xml");
        mojoExecutorWrapper.executeWithMergedConfig(
                ctx.getPlugin("org.apache.maven.plugins", "maven-help-plugin"),
                goal("effective-pom"),
                configuration(
                        element(name("artifact"), artifact.getGroupId() + ":" + artifact.getArtifactId() + ":" + artifact.getVersion()),
                        element(name("output"), effectivePom.getAbsolutePath())
                ),
                executionEnvironment()
        );
        return effectivePom;
    }

    private static class Container extends ProductArtifact {
        private final String id;
        private final String type;
        private final String classifier;

        /**
         * Installable container that can be downloaded by Maven.
         *
         * @param id         identifier of container, eg. "tomcat5x".
         * @param groupId    groupId of container.
         * @param artifactId artifactId of container.
         * @param version    version number of container.
         */
        Container(final String id, final String groupId, final String artifactId, final String version) {
            super(groupId, artifactId, version);
            this.id = id;
            this.type = "installed";
            this.classifier = "";
        }

        /**
         * Installable container that can be downloaded by Maven.
         *
         * @param id         identifier of container, eg. "tomcat5x".
         * @param groupId    groupId of container.
         * @param artifactId artifactId of container.
         * @param version    version number of container.
         * @param classifier classifier of the container.
         */
        Container(final String id, final String groupId, final String artifactId, final String version, final String classifier) {
            super(groupId, artifactId, version);
            this.id = id;
            this.type = "installed";
            this.classifier = classifier;
        }

        /**
         * Embedded container packaged with Cargo.
         *
         * @param id identifier of container, eg. "jetty6x".
         */
        Container(final String id) {
            this.id = id;
            this.type = "embedded";
            this.classifier = "";
        }

        /**
         * @return identifier of container.
         */
        public String getId() {
            return id;
        }

        /**
         * @return "installed" or "embedded".
         */
        @Override
        public String getType() {
            return type;
        }

        /**
         * @return classifier the classifier of the ProductArtifact
         */
        public String getClassifier() {
            return classifier;
        }

        /**
         * @return <code>true</code> if the container type is "embedded".
         */
        public boolean isEmbedded() {
            return "embedded".equals(type);
        }

        /**
         * @param buildDir project.build.directory.
         * @return root directory of the container that will house the container installation and configuration.
         */
        public String getRootDirectory(String buildDir) {
            return buildDir + File.separator + "container" + File.separator + getId();
        }

        /**
         * @param buildDir project.build.directory.
         * @return directory housing the installed container.
         */
        public String getInstallDirectory(String buildDir) {
            String installDirectory = getRootDirectory(buildDir) + File.separator + getArtifactId() + "-";
            String version = getVersion();
            if (version.endsWith("-atlassian-hosted") && !new File(installDirectory + version).exists()) {
                version = version.substring(0, version.indexOf("-atlassian-hosted"));
            }
            return installDirectory + version;
        }

        /**
         * @param buildDir  project.build.directory.
         * @param productId product name.
         * @return directory to house the container configuration for the specified product.
         */
        public String getConfigDirectory(String buildDir, String productId) {
            return getRootDirectory(buildDir) + File.separator + "cargo-" + productId + "-home";
        }
    }
}
