package com.atlassian.mail.msgraph.util;

import io.atlassian.fugue.Either;
import io.atlassian.fugue.Iterables;

import javax.annotation.Nonnull;
import java.util.Iterator;
import java.util.List;
import java.util.function.BinaryOperator;
import java.util.function.Supplier;

import static io.atlassian.fugue.Either.left;
import static io.atlassian.fugue.Either.right;
import static io.atlassian.fugue.Option.none;
import static io.atlassian.fugue.Option.some;
import static io.atlassian.fugue.Pair.pair;

/**
 * <p>A structure for running a sequence of operations that can fail.</p>
 *
 * <p>The motivating operation is paginated requests to an external system, in which each {@link LazyLinkedListExecutor#next} is
 * constructed based on the paging data returned by the outcome of the prior request.</p>
 *
 * <p> Because the structure is lazy, we can construct the request sequence without running it, and requests are only
 * executed when {@link LazyLinkedListExecutor#reduce} is called or the {@link Iterator} is run.</p>
 *
 * @param <T> type of the result of each successful execution.
 */
public class LazyLinkedListExecutor<T> implements Iterable<T> {
    public final T value;
    public final boolean hasNext;
    private final Supplier<Either<Throwable, LazyLinkedListExecutor<T>>> getNext;

    public LazyLinkedListExecutor(T value) {
        this.value = value;
        hasNext = false;
        getNext = this::next;
    }

    public LazyLinkedListExecutor(T value, Supplier<Either<Throwable, LazyLinkedListExecutor<T>>> getNext) {
        this.value = value;
        this.hasNext = true;
        this.getNext = getNext;
    }

    public Either<Throwable, LazyLinkedListExecutor<T>> next() {
        return hasNext ? getNext.get() : left(new NoItemsAvailable());
    }

    @Nonnull
    @Override
    public Iterator<T> iterator() {
        return Iterables.unfold(
                maybeItem -> maybeItem.flatMap(item -> {
                    if (item.hasNext) {
                        Either<Throwable, LazyLinkedListExecutor<T>> next = item.next();
                        return next.toOption().map(nextItem -> pair(item.value, some(nextItem)));
                    } else {
                        return some(pair(item.value, none()));
                    }
                }), some(this)).iterator();
    }

    public Either<Throwable, T> reduce(BinaryOperator<T> combine) {
        if (hasNext) {
            return next()
                    .flatMap(next -> next.reduce(combine))
                    .map(ts -> combine.apply(value, ts));
        } else {
            return right(value);
        }
    }

    public static class NoItemsAvailable extends Throwable {
        public NoItemsAvailable() {
            super("Tried to request next when there were none left");
        }
    }

    public static <T> LazyLinkedListExecutor<T> fromList(T first, List<T> rest) {
        if (rest.isEmpty()) {
            return new LazyLinkedListExecutor<>(first);
        } else {
            return new LazyLinkedListExecutor<>(
                    first,
                    () -> right(fromList(rest.get(0), rest.subList(1, rest.size()))));
        }
    }
}
