package com.atlassian.mail.msgraph.service;

import com.atlassian.mail.msgraph.settings.providers.MailConnectionSettingsProvider;
import com.atlassian.mail.msgraph.util.LazyLinkedListExecutor;
import com.atlassian.mail.msgraph.util.Lists;
import com.google.common.annotations.VisibleForTesting;
import com.microsoft.graph.models.MailFolder;
import com.microsoft.graph.requests.MailFolderCollectionPage;
import com.microsoft.graph.requests.MessageCollectionPage;
import io.atlassian.fugue.Either;
import io.atlassian.fugue.Unit;
import java.util.Optional;
import org.joda.time.DateTime;

import javax.mail.Message;
import java.util.List;

import static java.util.Collections.emptyList;

public class MicrosoftGraphMailService implements ExternalMailPuller {

    private final MailConnectionSettingsProvider settingsProvider;
    private final MicrosoftGraphMailClient microsoftGraphClient;

    public MicrosoftGraphMailService(MailConnectionSettingsProvider settingsProvider) {
        this.settingsProvider = settingsProvider;
        this.microsoftGraphClient = new MicrosoftGraphMailClient(settingsProvider);
    }

    @VisibleForTesting
    MicrosoftGraphMailService(MailConnectionSettingsProvider settingsProvider, MicrosoftGraphMailClient microsoftGraphMailClient) {
        this.settingsProvider = settingsProvider;
        this.microsoftGraphClient = microsoftGraphMailClient;
    }

    @Override
    public Either<Throwable, Unit> verifyConnection() {
        return microsoftGraphClient.getMessages(new DateTime(settingsProvider.getMailSettings().getPullFromDate()), "inbox")
                        .map(messagePage -> Unit.VALUE);
    }

    @Override
    public Either<Throwable, List<Message>> pullMessages() {
        Either<Throwable, String> folderIdByName = getFolderIdByName();

        if (folderIdByName.isLeft()) {
            return Either.left(folderIdByName.left().get());
        }

        Either<Throwable, MessageCollectionPage> firstPage = microsoftGraphClient.getMessages(
                new DateTime(settingsProvider.getMailSettings().getPullFromDate()), folderIdByName.right().get());

        Either<Throwable, LazyLinkedListExecutor<List<Message>>> allPages =
                firstPage.map(messagePage ->
                        new LazyLinkedListExecutor<>(
                                emptyList(),
                                () -> pullMessagesPage(firstPage)));

        return allPages.flatMap(executor -> executor.reduce(Lists::concat));
    }

    private Either<Throwable, LazyLinkedListExecutor<List<Message>>> pullMessagesPage(Either<Throwable, MessageCollectionPage> current) {
        return current.map(messagesPage -> {
            List<Message> messages = microsoftGraphClient.getMessageStubsFromMessagePage(messagesPage);

            if (messagesPage.getNextPage() == null) {
                return new LazyLinkedListExecutor<>(messages);
            } else {
                return new LazyLinkedListExecutor<>(
                        messages,
                        () -> pullMessagesPage(microsoftGraphClient.getNextMessagesPage(messagesPage)));
            }
        });
    }

    private Either<Throwable, String> getFolderIdByName() {
        String folderName = settingsProvider.getMailSettings().getFolder();

        if (!folderName.equalsIgnoreCase("inbox")) {
            Either<Throwable, MailFolderCollectionPage> either = microsoftGraphClient.getFolderIdByName(folderName);

            if (either.isLeft()) {
                return Either.left(either.left().get());
            }

            Optional<MailFolder> mailFolder = either.right().get().getCurrentPage()
                    .stream()
                    .filter(f -> f.displayName != null && f.displayName.equals(settingsProvider.getMailSettings().getFolder()))
                    .findFirst();
            return mailFolder.<Either<Throwable, String>>map(folder -> Either.right(folder.id))
                    .orElseGet(() -> Either.left(new IllegalArgumentException("Folder with name " + folderName + " cannot be found")));
        } else {
            return Either.right("inbox");
        }
    }

    @Override
    public Either<Throwable, Unit> markMessageRead(Message message) {
        return microsoftGraphClient.markMessageRead(message);
    }
}
