package com.atlassian.mail.config;

import java.io.InputStream;
import java.util.HashMap;
import java.util.Map;
import java.util.Optional;
import java.util.function.BiFunction;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;

import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.w3c.dom.Text;

import com.atlassian.mail.Settings;
import com.atlassian.mail.server.MailServerManager;
import com.atlassian.mail.server.auth.AuthenticationContextFactory;
import com.atlassian.mail.util.ClassLoaderUtils;

public class ConfigLoader {
    private static final String DEFAULT_CONFIG_FILE = "mail-config.xml";

    private MailServerManager loadedManager;
    private Settings loadedSettings;
    private AuthenticationContextFactory loadedAuthCtxFactory;

    /**
     * Constructor for internal use only - it allows ImmutableConfigLoader to be created
     */
    private ConfigLoader() {
        super();
    }

    public ConfigLoader(final String file) {
        // This is a bit of a hack for JBOSS
        try (InputStream configurationFileAsStream = ClassLoaderUtils.getResourceAsStream(file, ConfigLoader.class)) {
            final DocumentBuilder db = DocumentBuilderFactory.newInstance().newDocumentBuilder();
            final Document xmlDoc = db.parse(configurationFileAsStream);
            final Element root = xmlDoc.getDocumentElement();
            final MailServerManager mailServerManager =
                    create(MailServerManager.class, root, "manager", Optional.of((instance, element) -> {
                        final Map<String, String> params = new HashMap<String, String>();
                        final NodeList properties = element.getElementsByTagName("property");
                        if (properties.getLength() > 0) {
                            for (int i = 0; i < properties.getLength(); i++) {
                                final Element property = (Element) properties.item(i);
                                final String name = getContainedText(property, "name");
                                final String value = getContainedText(property, "value");
                                params.put(name, value);
                            }
                        }
                        instance.init(params);
                        return instance;
                    }));
            setLoadedManager(mailServerManager);

            final Settings settingsInstance = create(Settings.class, root, "settings", Optional.empty());
            if (settingsInstance != null) {
                setLoadedSettings(settingsInstance);
            } else {
                setLoadedSettings(new Settings.Default());
            }

            final AuthenticationContextFactory authCtxFactory =
                    create(AuthenticationContextFactory.class, root, "auth-ctx-factory", Optional.empty());

            if (authCtxFactory != null) {
                setLoadedAuthContextFactory(authCtxFactory);
            }

        } catch (Exception e) {
            throw new RuntimeException("Error in mail configuration: " + e.getMessage(), e);
        }
    }

    private static <T> T create(
            final Class<T> expectedClass,
            final Element root,
            final String elementName,
            final Optional<BiFunction<T, Element, T>> createHandlerOpt)
            throws ClassNotFoundException, InstantiationException, IllegalAccessException {
        final Element configurationElement =
                (Element) root.getElementsByTagName(elementName).item(0);
        if (configurationElement != null) {
            final Class<?> factoryClass =
                    ClassLoaderUtils.loadClass(configurationElement.getAttribute("class"), ConfigLoader.class);
            if (expectedClass.isAssignableFrom(factoryClass)) {
                final T instance = expectedClass.cast(factoryClass.newInstance());
                return createHandlerOpt
                        .map(h -> h.apply(instance, configurationElement))
                        .orElse(instance);
            }
        }
        return null;
    }

    /**
     * Returns immutable default configuration stored in ConfigLoader
     * @return immutable ConfigLoader instance
     */
    public static ConfigLoader getImmutableConfigurationLoader() {
        return getImmutableConfigurationLoader(DEFAULT_CONFIG_FILE);
    }

    public static ConfigLoader getImmutableConfigurationLoader(final String file) {
        final ConfigLoader configLoader = new ConfigLoader(file);
        return new ImmutableConfigLoader(configLoader);
    }

    public static MailServerManager getServerManager() {
        return getServerManager(DEFAULT_CONFIG_FILE);
    }

    public static MailServerManager getServerManager(final String file) {
        final ConfigLoader configLoader = new ConfigLoader(file);
        return configLoader.getLoadedManager();
    }

    public static Settings getSettings(final String file) {
        final ConfigLoader configLoader = new ConfigLoader(file);
        return configLoader.getLoadedSettings();
    }

    public static Settings getSettings() {
        return getSettings(DEFAULT_CONFIG_FILE);
    }

    public static AuthenticationContextFactory getAuthenticationContextFactory(final String file) {
        final ConfigLoader configLoader = new ConfigLoader(file);
        return configLoader.getLoadedAuthContextFactory();
    }

    public static AuthenticationContextFactory getAuthenticationContextFactory() {
        return getAuthenticationContextFactory(DEFAULT_CONFIG_FILE);
    }

    public MailServerManager getLoadedManager() {
        return loadedManager;
    }

    public void setLoadedManager(final MailServerManager loadedManager) {
        this.loadedManager = loadedManager;
    }

    public void setLoadedSettings(final Settings loadedSettings) {
        this.loadedSettings = loadedSettings;
    }

    public Settings getLoadedSettings() {
        return loadedSettings;
    }

    public void setLoadedAuthContextFactory(final AuthenticationContextFactory factory) {
        this.loadedAuthCtxFactory = factory;
    }

    public AuthenticationContextFactory getLoadedAuthContextFactory() {
        return this.loadedAuthCtxFactory;
    }

    private static String getContainedText(Element parent, String childTagName) {
        try {
            Node tag = parent.getElementsByTagName(childTagName).item(0);
            return ((Text) tag.getFirstChild()).getData();
        } catch (Exception e) {
            return null;
        }
    }

    /**
     * Represents ConfigLoader that can not be changed after being created
     */
    private static final class ImmutableConfigLoader extends ConfigLoader {

        private final ConfigLoader loader;

        private ImmutableConfigLoader(final ConfigLoader delegate) {
            this.loader = delegate;
        }

        @Override
        public MailServerManager getLoadedManager() {
            return loader.getLoadedManager();
        }

        @Override
        public void setLoadedManager(MailServerManager loadedManager) {}

        @Override
        public void setLoadedSettings(Settings loadedSettings) {}

        @Override
        public Settings getLoadedSettings() {
            return loader.getLoadedSettings();
        }

        @Override
        public void setLoadedAuthContextFactory(AuthenticationContextFactory factory) {}

        @Override
        public AuthenticationContextFactory getLoadedAuthContextFactory() {
            return loader.getLoadedAuthContextFactory();
        }
    }
}
