package com.atlassian.mail.queue;

import java.sql.Timestamp;
import java.util.Queue;

/**
 * <p>The MailQueue is used to asynchronously send mail from within applications.</p>
 *
 * <p>It is populated with {@link MailQueueItem} objects, which know how to send themselves.</p>
 *
 * <p>The queue also keeps track of which messages have attempted to be send, and were erroring.</p>
 */
public interface MailQueue {
    /**
     * Send all the messages in the queue.
     */
    void sendBuffer();

    /**
     * Send all the messages in the queue. Block until all other sendBuffer() operations are finished.
     */
    void sendBufferBlocking();

    /**
     * @return The number of messages in the queue.
     */
    int size();

    /**
     * @return The number of erroring messages in the queue.
     */
    int errorSize();

    /**
     * Add a new item to the mail queue.
     *
     * @param item The item to be added to the queue.
     */
    void addItem(MailQueueItem item);

    /**
     * Add an error item
     *
     * @param item a mail queue item to append to the "errorItems" list
     */
    void addErrorItem(MailQueueItem item);

    /**
     * <p>Get access to the messages in the mail queue.</p>
     *
     * <p>Note: you should synchronize on access to this queue.</p>
     * @return a queue of {@link MailQueueItem} objects representing messages in the mail queue
     */
    Queue<MailQueueItem> getQueue();

    /**
     * <p>Get access to the messages in the error queue.</p>
     *
     * <p>Note: you should synchronize on access to this queue.</p>
     * @return a queue of {@link MailQueueItem} objects representing messages in the error queue
     */
    Queue<MailQueueItem> getErrorQueue();

    /**
     * @return Whether or not the queue is currently sending.
     */
    boolean isSending();

    /**
     * @return The date/time the queue started sending, null if the queue is not sending
     */
    Timestamp getSendingStarted();

    /**
     * @return The date/time the queue attempted to send the last email in the current batch,
     * null if no email was sent yet
     */
    default Timestamp getLastSendingAttempt() {
        throw new UnsupportedOperationException();
    }

    /**
     * Empty the error queue (discard these messages)
     */
    void emptyErrorQueue();

    /**
     * Send all messages in the error queue.
     */
    void resendErrorQueue();

    /**
     * Retrieve the item currently being sent.
     * @return the {@link MailQueueItem} object representing a message currently being sent.
     */
    MailQueueItem getItemBeingSent();

    /**
     * If the queue is sending and has 'stuck' on an item, this lets the queue proceed.
     * Only relevant for synchronous queues.
     */
    void unstickQueue();
}
