package com.atlassian.mail.converters.wiki;

import org.jsoup.nodes.Element;
import org.jsoup.nodes.Node;
import org.jsoup.nodes.TextNode;

import javax.annotation.ParametersAreNonnullByDefault;

import static org.apache.commons.lang3.StringUtils.EMPTY;
import static org.apache.commons.lang3.StringUtils.isNotBlank;
import static org.jsoup.internal.StringUtil.in;

@ParametersAreNonnullByDefault
final class TableHandler {

    public static final String HTML_TABLE = "table";
    public static final String HTML_TH = "th";
    public static final String HTML_TR = "tr";
    public static final String HTML_TD = "td";

    private static final String WIKI_HEADER = "||";
    private static final String WIKI_DATA = "|";
    private static final String WIKI_ROW_END = "|";
    private static final String NEWLINE = "\n";

    private final BlockStyleHandler blockStyleHandler;

    private boolean inTable;
    private boolean inRow;

    private boolean firstTableRowData;

    public TableHandler(final BlockStyleHandler blockStyleHandler) {
        this.blockStyleHandler = blockStyleHandler;
    }

    public String enter(final String name) {
        if (!this.blockStyleHandler.isFormattingPossible()) {
            return EMPTY;
        }

        if (HTML_TABLE.equals(name)) {
            inTable = true;
            return NEWLINE;
        } else if (inTable && HTML_TR.equals(name)) {
            inRow = true;
            firstTableRowData = true;
            return EMPTY;
        } else if (inRow && HTML_TH.equals(name)) {
            return WIKI_HEADER;
        } else if (inRow && HTML_TD.equals(name)) {
            return WIKI_DATA;
        }

        return EMPTY;
    }

    public String exit(final String name) {
        if (!this.blockStyleHandler.isFormattingPossible()) {
            return EMPTY;
        }

        if (HTML_TABLE.equals(name)) {
            inTable = false;
            return NEWLINE;
        } else if (inRow && HTML_TR.equals(name)) {
            inRow = false;
            return WIKI_ROW_END + NEWLINE;
        } else if (in(name, HTML_TD, HTML_TH)) {
            firstTableRowData = false;
        }

        return EMPTY;
    }

    public boolean isFirstTableRowData() {
        return inTable && inRow && firstTableRowData;
    }

    public boolean isEndOfRow() {
        return inTable && !inRow;
    }

    public boolean isInTable() {
        return inTable;
    }

    public boolean isStartOfTableData(final Node node) {
        final Node parent = node.parent();

        // tables are a special case, as having a new line in the wrong place breaks them.
        // new lines are allowed, just not immediately preceding the wiki format | or ||
        //
        // so what we want to do here is find out if the supplied Node is the first child of the parent.
        //
        // In doing this check, if there is a blank text node before it, or a breaking element before it, then
        // these are ignored.
        if (parent != null) {
            if (parent.childNodeSize() > 0) {
                int index = 0;
                Node node1 = null;
                while (node1 == null && index < parent.childNodeSize()) {
                    node1 = parent.childNode(index++);
                    if (node.equals(node1)) {
                        break;
                    } else if (node1 instanceof TextNode) {
                        if (isNotBlank(((TextNode) node1).text())) {
                            break;
                        } else {
                            node1 = null;
                        }
                    } else if (node1 instanceof Element) {
                        final Element element = (Element) node1;
                        if (in(element.nodeName(), "p", "div", "br")) {
                            node1 = null;
                        }
                    }
                }

                // is the first no empty / breaking child what we want? And is the parent actually a table data / header cell?
                return node.equals(node1) && in(parent.nodeName(), HTML_TD, HTML_TH);
            }
        }

        return false;
    }

}
