package com.atlassian.mail.converters.wiki;

import com.atlassian.mail.MailUtils.Attachment;
import com.atlassian.mail.converters.HtmlConverter;
import com.google.common.collect.ImmutableList;
import org.jsoup.nodes.Element;
import org.jsoup.nodes.Node;
import org.jsoup.nodes.TextNode;
import org.jsoup.select.NodeTraversor;
import org.jsoup.select.NodeVisitor;

import javax.annotation.Nonnull;
import java.io.IOException;
import java.util.List;
import java.util.Objects;
import java.util.regex.Pattern;

import static com.atlassian.mail.converters.wiki.BlockStyleHandler.HTML_BLOCKQUOTE;
import static com.atlassian.mail.converters.wiki.BlockStyleHandler.HTML_CODE;
import static com.atlassian.mail.converters.wiki.BlockStyleHandler.HTML_PRE;
import static com.atlassian.mail.converters.wiki.FontStyleHandler.HTML_B;
import static com.atlassian.mail.converters.wiki.FontStyleHandler.HTML_CITE;
import static com.atlassian.mail.converters.wiki.FontStyleHandler.HTML_DEL;
import static com.atlassian.mail.converters.wiki.FontStyleHandler.HTML_EM;
import static com.atlassian.mail.converters.wiki.FontStyleHandler.HTML_I;
import static com.atlassian.mail.converters.wiki.FontStyleHandler.HTML_INS;
import static com.atlassian.mail.converters.wiki.FontStyleHandler.HTML_Q;
import static com.atlassian.mail.converters.wiki.FontStyleHandler.HTML_S;
import static com.atlassian.mail.converters.wiki.FontStyleHandler.HTML_STRIKE;
import static com.atlassian.mail.converters.wiki.FontStyleHandler.HTML_STRONG;
import static com.atlassian.mail.converters.wiki.FontStyleHandler.HTML_U;
import static com.atlassian.mail.converters.wiki.LinkAndImageHandler.HTML_IMG;
import static com.atlassian.mail.converters.wiki.LinkAndImageHandler.HTML_LINK;
import static com.atlassian.mail.converters.wiki.ListHandler.HTML_DD;
import static com.atlassian.mail.converters.wiki.ListHandler.HTML_DL;
import static com.atlassian.mail.converters.wiki.ListHandler.HTML_DT;
import static com.atlassian.mail.converters.wiki.ListHandler.HTML_LI;
import static com.atlassian.mail.converters.wiki.ListHandler.HTML_OL;
import static com.atlassian.mail.converters.wiki.ListHandler.HTML_UL;
import static com.atlassian.mail.converters.wiki.TableHandler.HTML_TABLE;
import static com.atlassian.mail.converters.wiki.TableHandler.HTML_TD;
import static com.atlassian.mail.converters.wiki.TableHandler.HTML_TH;
import static com.atlassian.mail.converters.wiki.TableHandler.HTML_TR;
import static org.apache.commons.lang3.StringUtils.EMPTY;
import static org.apache.commons.lang3.StringUtils.contains;
import static org.apache.commons.lang3.StringUtils.containsAny;
import static org.apache.commons.lang3.StringUtils.defaultIfEmpty;
import static org.apache.commons.lang3.StringUtils.defaultString;
import static org.apache.commons.lang3.StringUtils.isBlank;
import static org.apache.commons.lang3.StringUtils.isNotBlank;
import static org.apache.commons.lang3.StringUtils.isNotEmpty;
import static org.apache.commons.lang3.StringUtils.replace;
import static org.apache.commons.lang3.StringUtils.startsWith;
import static org.apache.commons.lang3.StringUtils.strip;
import static org.apache.commons.lang3.StringUtils.stripStart;
import static org.apache.commons.lang3.StringUtils.substring;
import static org.apache.commons.lang3.StringUtils.trimToEmpty;
import static org.jsoup.internal.StringUtil.in;
import static org.jsoup.internal.StringUtil.isWhitespace;

/**
 * <p>Helper class to convert basic HTML to Wiki macro.</p>
 *
 * <p>For example <code>img</code> to <code>!image.png|thumbnail!</code>.</p>
 *
 * @see <a href="https://extranet.atlassian.com/display/VPORT/CQ+-+Update+Html+To+Wiki+Text+converter+in+Atlassian+Mail">
 *     Update HTML to Wiki Text converter in Atlassian</a> for some more details
 * @since v2.6.0 was moved from <code>com.atlassian.mail.HtmlToWikiTextConverter</code>
 */
public class HtmlToWikiTextConverter implements HtmlConverter {

    private static final char WIKI_LINK_START_CHAR = '[';
    private static final char WIKI_TABLE_CHAR = '|';
    private static final char NEWLINE_CHAR = '\n';

    private static final String NEWLINE = Character.toString(NEWLINE_CHAR);
    private static final String WIKI_TABLE = Character.toString(WIKI_TABLE_CHAR);
    private static final String NON_WIKI_TEXT_REPLACE = ":";
    private static final String WIKI_LINK_START = Character.toString(WIKI_LINK_START_CHAR);
    private static final String WIKI_LINK_END = "]";

    private final List<Attachment> attachments;
    private final boolean thumbnailsAllowed;

    public HtmlToWikiTextConverter(@Nonnull final List<Attachment> attachments) {
        this(attachments, true);
    }

    public HtmlToWikiTextConverter(@Nonnull List<Attachment> attachments, boolean thumbnailsAllowed) {
        this.attachments = ImmutableList.copyOf(attachments);
        this.thumbnailsAllowed = thumbnailsAllowed;
    }

    private enum WhitespaceNewlineHandling {
        NONE,
        ONE,
        UNCHANGED
    }

    @Override
    public String convert(@Nonnull final String html) throws IOException {
        final DocumentUtilities.DocumentElement document = DocumentUtilities.parseHtml(html);

        final DocumentUtilities.BodyElement body = DocumentUtilities.getBody(document);

        FormattingVisitor formatter = new FormattingVisitor();
        NodeTraversor traversor = new NodeTraversor();
        traversor.traverse(formatter, body.getBody()); // walk the DOM, and call .head() and .tail() for each node

        return formatter.toString();
    }

    // the formatting rules, implemented in a breadth-first DOM traverse
    private class FormattingVisitor implements NodeVisitor {
        private final StringBuilder accum = new StringBuilder(); // holds the accumulated text
        private final StringBuilder colorAccum = new StringBuilder();
        private final BlockStyleHandler blockStyleHandler = new BlockStyleHandler();
        private final FontStyleHandler fontStyleHandler = new FontStyleHandler(blockStyleHandler);
        private final ListHandler listHandler = new ListHandler(blockStyleHandler);
        private final TableHandler tableHandler = new TableHandler(blockStyleHandler);
        private final ColorHandler colorHandler = new ColorHandler(fontStyleHandler, blockStyleHandler);
        private final LinkAndImageHandler linkAndImageHandler = new LinkAndImageHandler(blockStyleHandler, colorHandler, attachments, thumbnailsAllowed);

        // hit when the node is first seen
        public void head(Node node, int depth) {
            final String name = node.nodeName();

            // TextNodes carry all user-readable text in the DOM.
            if (node instanceof TextNode) {
                String text = ((TextNode) node).text();
                if (fontStyleHandler.isAnyAtStart(depth)) {
                    text = stripStart(text, null);
                }
                if (tableHandler.isInTable()) {
                    // if we are inside of a table, then to prevent weirdness in the wiki output, replace | with someething else (in our case :)
                    text = replace(text, WIKI_TABLE, NON_WIKI_TEXT_REPLACE);
                }
                if (linkAndImageHandler.isInsideLinkWithText()) {
                    text = replace(text, WIKI_LINK_START, NON_WIKI_TEXT_REPLACE);
                    text = replace(text, WIKI_LINK_END, NON_WIKI_TEXT_REPLACE);
                }

                // now append
                if (tableHandler.isInTable()) {
                    // if we are inside of a table, then to prevent weirdness in the wiki output, replace | with someething else (in our case :)
                    append(node, text, WhitespaceNewlineHandling.ONE);
                } else if (linkAndImageHandler.isInsideLinkWithText()) {
                    append(node, text, WhitespaceNewlineHandling.NONE);
                } else {
                    append(node, text);
                }
            } else if (in(name, HTML_B, HTML_STRONG, HTML_I, HTML_EM, HTML_U, HTML_INS, HTML_STRIKE, HTML_DEL, HTML_S, HTML_Q, HTML_CITE)) {
                final String enterText = fontStyleHandler.enter(node, name, depth);
                if (isNotEmpty(enterText) && node instanceof Element) {
                    final Element element = (Element) node;
                    if (element.hasText()) {
                        String removedWhitespace = EMPTY;

                        final Node firstChildNode = element.childNodeSize() > 0 ? element.childNode(0) : null;

                        String text;
                        if (firstChildNode instanceof TextNode) {
                            text = ((TextNode) firstChildNode).getWholeText();
                        } else {
                            text = element.text();
                        }

                        while (text.length() > 0 && isWhitespace(text.charAt(0))) {
                            removedWhitespace += text.charAt(0);
                            text = substring(text, 1);
                        }

                        if (!fontStyleHandler.isPrecededByStyle(this.accum.toString())) {
                            append(node, defaultIfEmpty(removedWhitespace, " "));
                        }
                    }
                }
                append(node, enterText);
            } else if (in(name, HTML_BLOCKQUOTE, HTML_PRE, HTML_CODE)) {
                appendAroundColor(node, blockStyleHandler.enter(node, name, depth));
            } else if (in(name, "h1", "h2", "h3", "h4", "h5", "h6")) {
                appendAroundColor(node, NEWLINE + name + ". ");
            } else if (in(name, HTML_OL, HTML_UL, HTML_LI, HTML_DL, HTML_DD, HTML_DT)) {
                final String enter = listHandler.enter(name);
                if (in(name, HTML_OL, HTML_UL, HTML_DL)) {
                    append(node, enter, WhitespaceNewlineHandling.ONE);
                } else {
                    String prefix = EMPTY;
                    if (!tableHandler.isInTable()) {
                        prefix = NEWLINE;
                    }
                    appendAroundColor(node, prefix + enter, WhitespaceNewlineHandling.ONE);
                }
            } else if (in(name, HTML_TABLE, HTML_TR, HTML_TH, HTML_TD)) {
                WhitespaceNewlineHandling whitespaceNewlineHandling = WhitespaceNewlineHandling.NONE;
                if (in(name, HTML_TABLE, HTML_TR) || tableHandler.isFirstTableRowData()) {
                    whitespaceNewlineHandling = WhitespaceNewlineHandling.ONE;
                }
                appendAroundColor(node, tableHandler.enter(name), whitespaceNewlineHandling);
            } else if (in(name, HTML_LINK, HTML_IMG)) {
                appendAroundColor(node, linkAndImageHandler.enter(node, name));
            } else if (in(name, "p", "div")) {
                if (name.equals("div")) {
                    for (Node child : node.childNodes()) {
                        final boolean treatNewline;
                        if (child instanceof TextNode) {
                            final String text = ((TextNode) child).text();
                            if (isBlank(text)) {
                                // if blank then check the next node if there is one.
                                // if there is not a next node, then no new line added
                                continue;
                            } else {
                                treatNewline = true;
                            }
                        } else if (child instanceof Element) {
                            treatNewline = !in(((Element) child).tagName(), "br", "p");
                        } else {
                            treatNewline = true;
                        }

                        if (treatNewline) {
                            append(node, NEWLINE);
                        }

                        // stop after processed the first relevant node
                        break;
                    }
                } else if (name.equals("p")) {
                    append(node, NEWLINE + NEWLINE);
                }
            }

            // handle any potential colors that may start here
            final String color = colorHandler.enter(accum, node, name, depth, linkAndImageHandler.isInsideAnyLink(), linkAndImageHandler.isInsideLinkWithText(), linkAndImageHandler.isUrlInLinkText(), tableHandler.isEndOfRow());
            if (isNotBlank(color)) {
                colorAccum.append(color);
            }
        }

        // hit when all of the node's children (if any) have been visited
        public void tail(Node node, int depth) {
            String name = node.nodeName();
            if (name.equals("br")) {
                append(node, NEWLINE);
            } else if (name.equals("hr")) {
                appendAroundColor(node, NEWLINE + "----" + NEWLINE, WhitespaceNewlineHandling.ONE);
            } else if (in(name, HTML_B, HTML_STRONG, HTML_I, HTML_EM, HTML_U, HTML_INS, HTML_STRIKE, HTML_DEL, HTML_S, HTML_Q, HTML_CITE)) {
                String fontExit = fontStyleHandler.exit(name, depth);
                String removedWhitespace = EMPTY;
                if (isNotBlank(fontExit)) {
                    // need to wrap tightly
                    removedWhitespace = DocumentUtilities.removeTrailingWhitespace(accum);

                    while (accum.length() > 0 && fontExit.length() > 0 && accum.charAt(accum.length() - 1) == fontExit.charAt(0)) {
                        accum.deleteCharAt(accum.length() - 1);
                        fontExit = substring(fontExit, 1);

                        removedWhitespace = DocumentUtilities.removeTrailingWhitespace(accum) + removedWhitespace;
                    }
                }

                append(node, fontExit);

                if (node instanceof Element) {
                    final Element element = (Element) node;
                    final Node sibling = element.nextSibling();
                    if (sibling != null && sibling instanceof TextNode) {
                        final String text = ((TextNode) sibling).text();
                        if (isNotEmpty(text) && !isWhitespace(text.charAt(0)) && !Pattern.matches("\\p{Punct}", substring(text, 0, 1))) {
                            append(node, " ");
                        }
                    }
                }

                append(node, removedWhitespace);
            } else if (in(name, HTML_BLOCKQUOTE, HTML_PRE, HTML_CODE)) {
                append(node, blockStyleHandler.exit(name, depth));
            } else if (in(name, "h1", "h2", "h3", "h4", "h5", "h6")) {
                appendAroundColor(node, NEWLINE);
            } else if (in(name, HTML_OL, HTML_UL, HTML_LI, HTML_DL, HTML_DD, HTML_DT)) {
                append(node, listHandler.exit(name), WhitespaceNewlineHandling.ONE);
            } else if (in(name, HTML_TABLE, HTML_TR, HTML_TH, HTML_TD)) {
                final String exit = tableHandler.exit(name);
                if (in(name, HTML_TABLE, HTML_TR)) {
                    appendAroundColor(node, exit, WhitespaceNewlineHandling.ONE);
                } else {
                    appendAroundColor(node, exit, WhitespaceNewlineHandling.NONE);
                }
            } else if (HTML_LINK.equals(name)) {
                append(node, linkAndImageHandler.exit(accum, node, name));
            }

            // check if any colors that this ends
            String precedingWhitespace = DocumentUtilities.removeTrailingWhitespace(accum);

            final String color = colorHandler.exit(accum, name, depth, precedingWhitespace, linkAndImageHandler.isInsideAnyLink(), linkAndImageHandler.isUrlInLinkText());
            if (isNotBlank(color)) {
                append(node, color);
            } else {
                accum.append(precedingWhitespace);
            }
        }

        private void appendAroundColor(Node node, String text) {
            appendAroundColor(node, text, WhitespaceNewlineHandling.UNCHANGED);
        }

        private void appendAroundColor(Node node, String text, WhitespaceNewlineHandling whitespaceNewlineHandling) {
            // color block only allows the standard font styling, such as bold, underline, etc.
            // if you try something like a table, link, horizontal rule ... it acts like no format :(
            String precedingWhitespace = DocumentUtilities.removeTrailingWhitespace(accum);

            final boolean tableRow = HTML_TR.equals(node.nodeName());
            final String s = colorHandler.handleAroundNonSupportedFormatting(accum, text, precedingWhitespace, linkAndImageHandler.isInsideAnyLink(), linkAndImageHandler.isInsideLinkWithText(), linkAndImageHandler.isUrlInLinkText(), tableRow);

            if (isNotEmpty(s)) {
                append(node, s, whitespaceNewlineHandling);
            } else {
                accum.append(precedingWhitespace);
            }
        }

        private void append(Node node, String text) {
            append(node, text, WhitespaceNewlineHandling.UNCHANGED);
        }

        private void append(Node node, String text, WhitespaceNewlineHandling whitespaceNewlineHandling) {
            text = replace(text, "\r" + NEWLINE, NEWLINE);
            text = replace(text, "\f", NEWLINE);
            text = replace(text, "\r", NEWLINE);
            text = replace(text, "\t", " ");

            if (accum.length() == 0) {
                text = stripStart(text, null);
            } else if (whitespaceNewlineHandling == WhitespaceNewlineHandling.NONE || whitespaceNewlineHandling == WhitespaceNewlineHandling.ONE) {
                String removed = DocumentUtilities.removeTrailingWhitespace(accum);

                final ColorHandler.StripResult removedFormatting = colorHandler.removeFormatting(text);
                String result = Objects.toString(removedFormatting.getResult(), EMPTY);
                while (isNotEmpty(result) && isWhitespace(result.charAt(0))) {
                    removed += result.charAt(0);
                    result = substring(result, 1);
                }

                if (containsAny(removed, NEWLINE) && isNotEmpty(removedFormatting.getRemoved())) {
                    text = removedFormatting.getRemoved() + NEWLINE + result;
                } else {
                    text = removedFormatting.getRemoved() + result;
                }

                if (containsAny(removed, " ")) {
                    // if it is text inside of a link, make sure that not preceded by the [ wiki, as can not append a space after that
                    if (linkAndImageHandler.isInsideLinkWithText()) {
                        if (accum.charAt(accum.length() - 1) != WIKI_LINK_START_CHAR) {
                            doAppending(' ');
                        }
                    } else {
                        doAppending(' ');
                    }
                }

                if (whitespaceNewlineHandling == WhitespaceNewlineHandling.ONE) {
                    if (contains(removed, NEWLINE_CHAR) && !tableHandler.isStartOfTableData(node)) {
                        // just in case. when inside a table, it needs some special newline handling, as a newline in
                        // wrong place breaks the whole table :(
                        if (!tableHandler.isInTable() || tableHandler.isEndOfRow() || tableHandler.isFirstTableRowData()) {
                            doAppending(NEWLINE);
                        }
                    }
                }
            }

            if (startsWith(text, NEWLINE)) {
                // more special table treatment
                final char last = this.accum.charAt(this.accum.length() - 1);
                if (tableHandler.isStartOfTableData(node) && (isWhitespace(last) || last == WIKI_TABLE_CHAR)) {
                    text = strip(text, NEWLINE);
                }
            }

            if (accum.length() > 0 && text.length() > 0 && tableHandler.isFirstTableRowData() && !linkAndImageHandler.isInsideLinkWithText() && contains(trimToEmpty(text), WIKI_TABLE_CHAR)) {
                final char last = this.accum.charAt(this.accum.length() - 1);
                if (last != NEWLINE_CHAR) {
                    doAppending(NEWLINE);
                }
            }

            doAppending(text);

            linkAndImageHandler.reset();
        }

        private void doAppending(char character) {
            doAppending(String.valueOf(character));
        }

        private void doAppending(String string) {
            if (colorAccum.length() > 0) {
                accum.append(colorAccum);
                colorAccum.delete(0, colorAccum.length());
            }

            fontStyleHandler.wrapStylesAround(accum, string);
        }

        @Override
        public String toString() {
            return accum.toString();
        }
    }

}
