package com.atlassian.mail.queue;

import com.atlassian.mail.MailException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.List;
import java.util.Queue;
import java.util.concurrent.PriorityBlockingQueue;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReadWriteLock;
import java.util.concurrent.locks.ReentrantReadWriteLock;

/**
 * <p>This is a volatile queue of the outgoing emails from JIRA.</p>
 *
 * <p>Note - this class may lose emails if the server shuts down.</p>
 */
public class MailQueueImpl implements MailQueue {
    private static final Logger log = LoggerFactory.getLogger(MailQueueImpl.class);
    private static final int MAX_SEND_ATTEMPTS = 10;

    private final Queue<MailQueueItem> items;
    private final Queue<MailQueueItem> errorItems;
    private volatile boolean sending;
    private volatile MailQueueItem itemBeingSent;
    private volatile Timestamp sendingStarted;
    private volatile Timestamp lastSendingAttempt;
    private final Lock sharedLock;
    private final Lock exclusiveLock;


    public MailQueueImpl() {
        items = new PriorityBlockingQueue<>();
        errorItems = new PriorityBlockingQueue<>();

        final ReadWriteLock lock = new ReentrantReadWriteLock();
        sharedLock = lock.readLock();
        exclusiveLock = lock.writeLock();
    }

    public void sendBuffer() {
        sharedLock.lock();
        try {
            sendBufferUnderLock();
        } finally {
            sharedLock.unlock();
        }
    }

    public void sendBufferBlocking() {
        exclusiveLock.lock();
        try {
            sendBufferUnderLock();
        } finally {
            exclusiveLock.unlock();
        }
    }

    private void sendBufferUnderLock() {
        if (sending) {
            log.warn("Already sending {} mails:", items.size());
            for (final MailQueueItem item : items) {
                log.warn("Queued to send: {}, {}", item, item.getClass());
            }
            return;
        }

        sendingStarted();
        final List<MailQueueItem> failed = new ArrayList<>();

        try {
            while (!items.isEmpty()) {
                String origThreadName = Thread.currentThread().getName();
                MailQueueItem item = items.poll();
                this.itemBeingSent = item;
                this.lastSendingAttempt = new Timestamp(System.currentTimeMillis());
                log.debug("Sending: {}", item);
                try {
                    Thread.currentThread().setName("Sending mailitem " + item);
                    item.send();
                } catch (MailException e) {
                    if (item.getSendCount() > MAX_SEND_ATTEMPTS)
                        errorItems.add(item);
                    else
                        failed.add(item);

                    log.error("Error occurred in sending e-mail: " + item, e);
                } finally {
                    Thread.currentThread().setName(origThreadName);
                }
            }

            items.addAll(failed);
        } finally {
            // Set sending to false no matter what happens
            sendingStopped();
        }
    }

    public int size() {
        return items.size();
    }

    public int errorSize() {
        return errorItems.size();
    }

    public void addItem(MailQueueItem item) {
        log.debug("Queued: {}", item);
        items.add(item);
    }

    public void addErrorItem(MailQueueItem item) {
        log.debug("Queued error: {}", item);
        errorItems.add(item);
    }

    public Queue<MailQueueItem> getQueue() {
        return items;
    }

    public Queue<MailQueueItem> getErrorQueue() {
        return errorItems;
    }

    public boolean isSending() {
        return sending;
    }

    public Timestamp getSendingStarted() {
        return sendingStarted;
    }

    public Timestamp getLastSendingAttempt() {
        return lastSendingAttempt;
    }

    public MailQueueItem getItemBeingSent() {
        return itemBeingSent;
    }

    public void unstickQueue() {
        log.error("Mail on queue was considered stuck: {}", itemBeingSent);
        sendingStopped();
    }

    public void emptyErrorQueue() {
        errorItems.clear();
    }

    public void resendErrorQueue() {
        items.addAll(errorItems);
        emptyErrorQueue();
    }

    public void sendingStarted() {
        sending = true;
        sendingStarted = new Timestamp(System.currentTimeMillis());
    }

    public void sendingStopped() {
        sending = false;
        sendingStarted = null;
        lastSendingAttempt = null;
    }
}
