package com.atlassian.mail;

import com.atlassian.mail.server.MailServer;
import org.apache.commons.lang3.StringUtils;

import javax.mail.Multipart;
import java.io.Serializable;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;

public class Email implements Serializable {

    private static final long serialVersionUID = 1763820874219520737L;

    // mandatory fields
    private String to;
    private String subject;

    // optional fields
    private String from;
    private String fromName;
    private String cc;
    private String bcc;
    private String replyTo;
    private String inReplyTo;
    private String body;
    private String mimeType;
    private String encoding;
    private Multipart multipart;
    private String messageId;
    private Map<String, String> headers;
    private boolean excludeSubjectPrefix;

    private void init(String to) {
        this.to = to;

        // set defaults
        this.subject = "";
        this.body = ""; // needs to be instantiated to empty string else send() will throw a NullPointer
        this.mimeType = "text/plain";
        this.encoding = "UTF-8";
        this.headers = new HashMap<>();
        this.excludeSubjectPrefix = false;
        loadDefaultHeaders();
    }

    /**
     * <b>Note</b>: By default the message has the "Precedence" header set to "bulk". Use {@link #removeHeader(java.lang.String)}
     * to remove
     *
     * @param to email address(es) to send the email to
     */
    public Email(String to) {
        if (StringUtils.isBlank(to))
            throw new IllegalArgumentException("'To' is a required field");

        init(to);
    }

    /**
     * <b>Note</b>: By default the message has the "Precedence" header set to "bulk". Use {@link #removeHeader(java.lang.String)}
     * to remove
     *
     * @param to email address(es) to send the email to
     * @param cc email address(es) to cc email to
     * @param bcc email address(es) to bcc email to
     */
    public Email(String to, String cc, String bcc) {
        if (StringUtils.isBlank(to) && StringUtils.isBlank(cc) && StringUtils.isBlank(bcc))
            throw new IllegalArgumentException("One of 'To', 'CC' or 'BCC' is required");

        init(to);
        this.cc = cc;
        this.bcc = bcc;
    }

    protected void loadDefaultHeaders() {
        // Set the "Precedence" header to "bulk". All the mails coming from atlassian products likely need this header.
        // This header should stop mail clients generating automatic messages to "answer" mail from atlassian products.
        // For example, "away on holiday" messages. JRA-2622
        headers.put("Precedence", "bulk");
        headers.put("Auto-Submitted", "auto-generated"); // see JRA-15325
    }

    public Email setFrom(String from) {
        this.from = from;
        return this;
    }

    public Email setFromName(String fromName) {
        this.fromName = fromName;
        return this;
    }

    public Email setTo(String to) {
        this.to = to;
        return this;
    }

    public Email setSubject(String subject) {
        this.subject = subject;
        return this;
    }

    public Email setCc(String cc) {
        this.cc = cc;
        return this;
    }

    public Email setBcc(String bcc) {
        this.bcc = bcc;
        return this;
    }

    public Email setReplyTo(String replyTo) {
        this.replyTo = replyTo;
        return this;
    }

    public Email setInReplyTo(String inReplyTo) {
        this.inReplyTo = inReplyTo;
        return this;
    }

    public Email setBody(String body) {
        this.body = body;
        return this;
    }

    public Email setMimeType(String mimeType) {
        this.mimeType = mimeType;
        return this;
    }

    public Email setEncoding(String encoding) {
        this.encoding = encoding;
        return this;
    }

    public Email setMultipart(Multipart multipart) {
        this.multipart = multipart;
        return this;
    }

    public Email setExcludeSubjectPrefix(boolean excludeSubjectPrefix) {
        this.excludeSubjectPrefix = excludeSubjectPrefix;
        return this;
    }

    public String getFrom() {
        return from;
    }

    public String getFromName() {
        return fromName;
    }

    public String getTo() {
        return to;
    }

    public String getSubject() {
        return subject;
    }

    public String getCc() {
        return cc;
    }

    public String getBcc() {
        return bcc;
    }

    public String getReplyTo() {
        return replyTo;
    }

    public String getInReplyTo() {
        return inReplyTo;
    }

    public String getBody() {
        return body;
    }

    public String getMimeType() {
        return mimeType;
    }

    public String getEncoding() {
        return encoding;
    }

    public Multipart getMultipart() {
        return multipart;
    }

    public String getMessageId() {
        return messageId;
    }

    public void setMessageId(String messageId) {
        this.messageId = messageId;
    }

    /**
     * @return true if the email should not have its subject prefixed when sent by a {@link MailServer}
     */
    public boolean isExcludeSubjectPrefix() {
        return excludeSubjectPrefix;
    }

    /**
     * Body is NOT included in comparing two Email objects
     */
    public boolean equals(Object o) {
        if (this == o) return true;
        if (!(o instanceof Email)) return false;

        final Email email = (Email) o;

        if (!Objects.equals(bcc, email.bcc)) return false;
        if (!Objects.equals(cc, email.cc)) return false;
        if (!Objects.equals(encoding, email.encoding)) return false;
        if (!Objects.equals(from, email.from)) return false;
        if (!Objects.equals(fromName, email.fromName)) return false;
        if (!Objects.equals(inReplyTo, email.inReplyTo)) return false;
        if (!Objects.equals(messageId, email.messageId)) return false;
        if (!Objects.equals(mimeType, email.mimeType)) return false;
        if (!Objects.equals(multipart, email.multipart)) return false;
        if (!Objects.equals(replyTo, email.replyTo)) return false;
        if (!Objects.equals(subject, email.subject)) return false;
        if (!to.equals(email.to)) return false;
        if (excludeSubjectPrefix != email.excludeSubjectPrefix) return false;

        return true;
    }

    /**
     * Body is NOT included in calculating the hashCode for the object.
     */
    public int hashCode() {
        int result;
        result = to.hashCode();
        result = 29 * result + (subject != null ? subject.hashCode() : 0);
        result = 29 * result + (from != null ? from.hashCode() : 0);
        result = 29 * result + (fromName != null ? fromName.hashCode() : 0);
        result = 29 * result + (cc != null ? cc.hashCode() : 0);
        result = 29 * result + (bcc != null ? bcc.hashCode() : 0);
        result = 29 * result + (replyTo != null ? replyTo.hashCode() : 0);
        result = 29 * result + (inReplyTo != null ? inReplyTo.hashCode() : 0);
        result = 29 * result + (mimeType != null ? mimeType.hashCode() : 0);
        result = 29 * result + (encoding != null ? encoding.hashCode() : 0);
        result = 29 * result + (multipart != null ? multipart.hashCode() : 0);
        result = 29 * result + (messageId != null ? messageId.hashCode() : 0);
        result = 29 * result + (Boolean.hashCode(excludeSubjectPrefix));
        return result;
    }

    public String toString() {
        return "To='" + to + "' Subject='" + subject + "' From='" + from + "' FromName='" + fromName + "' Cc='" + cc +
                "' Bcc='" + bcc + "' ReplyTo='" + replyTo + "' InReplyTo='" + inReplyTo + "' MimeType='" + mimeType +
                "' Encoding='" + encoding + "' Multipart='" + multipart + "' MessageId='" + messageId +
                "' ExcludeSubjectPrefix=" + Boolean.toString(excludeSubjectPrefix) + "'";
    }

    public void addHeader(String headerName, String headerValue) {
        headers.put(headerName, headerValue);
    }

    /**
     * @param headerName name of the email header to remove
     * @return the value of the removed header
     */
    public String removeHeader(String headerName) {
        if (headers.containsKey(headerName)) {
            return (String) headers.remove(headerName);
        } else {
            return null;
        }
    }

    public Map getHeaders() {
        return headers;
    }
}
