package com.atlassian.mail.converters.wiki;

import com.atlassian.mail.MailUtils;
import com.google.common.base.Predicate;
import com.google.common.collect.ImmutableList;
import org.jsoup.nodes.Element;
import org.jsoup.nodes.Node;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;
import java.net.URL;
import java.util.List;
import java.util.regex.Pattern;

import static com.google.common.collect.Iterables.any;
import static org.apache.commons.lang3.StringUtils.EMPTY;
import static org.apache.commons.lang3.StringUtils.equalsIgnoreCase;
import static org.apache.commons.lang3.StringUtils.isNotBlank;
import static org.apache.commons.lang3.StringUtils.removeStart;
import static org.apache.commons.lang3.StringUtils.split;
import static org.apache.commons.lang3.StringUtils.startsWithIgnoreCase;
import static org.apache.commons.lang3.StringUtils.trimToEmpty;

@ParametersAreNonnullByDefault
final class LinkAndImageHandler {

    private static final Logger log = LoggerFactory.getLogger(LinkAndImageHandler.class);

    public static final String HTML_LINK = "a";
    public static final String HTML_IMG = "img";
    public static final String HTML_HREF = "href";

    private static final String CONTENT_ID_PREFIX = "cid:";

    private static final String WIKI_LINK_OPEN_BRACKET = "[";
    private static final String WIKI_LINK_CLOSE_BRACKET = "]";
    private static final String WIKI_LINK_SEPARATOR = "|";

    private static final String WIKI_MACRO_ATTACHMENT_FORMAT = "!%s!";

    private static String WIKI_MACRO_ATTACHMENT(final String value) {
        return String.format(WIKI_MACRO_ATTACHMENT_FORMAT, value);
    }

    private static final String WIKI_MACRO_THUMBNAIL_FORMAT = "!%s|thumbnail!";
    // not exhaustive list
    private static final Pattern THUMBNAIL_PATTERN = Pattern.compile("(?i)^image/(jpeg|jpg|png|pjpeg)");

    private static String WIKI_IMAGE_MACRO(final String value, @Nullable final String contentType, final boolean thumbnailsAllowed) {
        if (thumbnailsAllowed && THUMBNAIL_PATTERN.matcher(trimToEmpty(contentType)).find()) {
            return String.format(WIKI_MACRO_THUMBNAIL_FORMAT, value);
        }

        return WIKI_MACRO_ATTACHMENT(value);
    }

    private static final String NON_WIKI_TEXT_FORMAT = " <%s>";

    private static String NON_WIKI_TEXT(final String value) {
        return String.format(NON_WIKI_TEXT_FORMAT, value);
    }

    static final List<String> NON_WIKI_LINK_URIS = ImmutableList.of("tel:", "sms:", "callto:", "fax:", "modem:", "wtai:", "rhomailto:");

    private final BlockStyleHandler blockStyleHandler;
    private final ColorHandler colorHandler;
    private final List<MailUtils.Attachment> attachments;
    private final boolean thumbnailsAllowed;

    private boolean linkNonWikiHref;
    private boolean inLink;
    private boolean linkHasText;
    private boolean imageInsideLink;
    private boolean urlInLinkText;
    private boolean toReset;

    public LinkAndImageHandler(final BlockStyleHandler blockStyleHandler, final ColorHandler colorHandler, final List<MailUtils.Attachment> attachments, final boolean thumbnailsAllowed) {
        this.blockStyleHandler = blockStyleHandler;
        this.colorHandler = colorHandler;
        this.attachments = ImmutableList.copyOf(attachments);
        this.thumbnailsAllowed = thumbnailsAllowed;
    }

    public String enter(final Node node, final String name) {
        if (!this.blockStyleHandler.isFormattingPossible()) {
            return EMPTY;
        }

        if (HTML_LINK.equals(name)) {
            inLink = false;
            linkHasText = false;
            imageInsideLink = false;
            urlInLinkText = false;
            toReset = false;

            final String href = node.attr(HTML_HREF);
            linkNonWikiHref = any(NON_WIKI_LINK_URIS, new Predicate<String>() {
                @Override
                public boolean apply(@Nullable String input) {
                    return startsWithIgnoreCase(href, input);
                }
            });
            if (!linkNonWikiHref) {
                if (node instanceof Element) {
                    final Element element = (Element) node;

                    // if there is also an image inside the link, don't try and markup any text, will get mixed in with image
                    if (element.select(HTML_IMG).isEmpty()) {
                        inLink = true;
                        linkHasText = element.hasText();

                        if (linkHasText) {
                            urlInLinkText = containsUrlInLinkText(element);
                        }

                        return WIKI_LINK_OPEN_BRACKET;
                    } else {
                        imageInsideLink = true;
                    }
                }
            }
        } else if (HTML_IMG.equals(name)) {
            return replaceWikiMacroForImage(node);
        }

        return EMPTY;
    }

    public String exit(final StringBuilder accum, final Node node, final String name) {
        if (!this.blockStyleHandler.isFormattingPossible()) {
            return EMPTY;
        }

        String val = EMPTY;
        if (HTML_LINK.equals(name)) {
            final String href = node.attr(HTML_HREF);
            if (linkNonWikiHref) {
                val += NON_WIKI_TEXT(href);
            } else if (imageInsideLink) {
                val = NON_WIKI_TEXT(WIKI_LINK_OPEN_BRACKET + href + WIKI_LINK_CLOSE_BRACKET);
            } else {
                if (linkHasText) {
                    val += WIKI_LINK_SEPARATOR;
                }
                val += href;
                val += WIKI_LINK_CLOSE_BRACKET;

                val = colorHandler.handleAroundNonSupportedFormatting(accum, val, EMPTY, false, false, false, false);
            }
            toReset = true;
        }

        return val;
    }

    public boolean isInsideLinkWithText() {
        return isInsideAnyLink() && linkHasText;
    }

    public boolean isInsideAnyLink() {
        return inLink;
    }

    public boolean isUrlInLinkText() {
        return isInsideLinkWithText() && urlInLinkText;
    }

    public void reset() {
        if (toReset) {
            inLink = false;
            linkHasText = false;
            imageInsideLink = false;
            linkNonWikiHref = false;
            urlInLinkText = false;
        }
        toReset = false;
    }

    private String replaceWikiMacroForImage(final Node node) {
        final String imageSource = node.attr("src");

        if (startsWithIgnoreCase(imageSource, CONTENT_ID_PREFIX)) {
            String contentId = removeStart(imageSource, CONTENT_ID_PREFIX);

            MailUtils.Attachment image = getAttachmentById(contentId);
            if (image != null) {
                return WIKI_IMAGE_MACRO(image.getFilename(), image.getContentType(), thumbnailsAllowed);
            }

            String alternativeText = node.attr("alt");
            String titleText = node.attr("title");

            log.warn("Could not find attachment: '" + alternativeText + "' (" + titleText + ") for content id: " + contentId);

            final String text;
            if (isNotBlank(alternativeText)) {
                text = alternativeText;
            } else if (isNotBlank(titleText)) {
                text = titleText;
            } else {
                text = contentId;
            }

            return text;
        }

        return WIKI_MACRO_ATTACHMENT(imageSource);
    }

    private MailUtils.Attachment getAttachmentById(final String contentId) {
        for (MailUtils.Attachment attachment : attachments) {
            if (equalsIgnoreCase(attachment.getContentId(), contentId)) {
                return attachment;
            }
        }
        return null;
    }

    private static boolean containsUrlInLinkText(final Element element) {
        // if apply colour to a block that contains something that looks like a valid URL it breaks it, as even though
        // it is link text it gets converted to a link in a weird way with colour block also there, so just don't use colour
        // if detect this
        for (String text : split(element.text())) {
            try {
                new URL(trimToEmpty(text));
                return true;
            } catch (Exception e) {
                // nope
            }
        }

        return false;
    }
}
