package com.atlassian.mail.converters.wiki;

import com.google.common.base.Function;
import com.google.common.collect.ImmutableMap;
import org.jsoup.nodes.Element;
import org.jsoup.nodes.Node;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.Map;

import static com.google.common.collect.Lists.newArrayList;
import static com.google.common.collect.Lists.transform;
import static com.google.common.collect.Maps.newHashMap;
import static org.apache.commons.lang.StringUtils.EMPTY;
import static org.apache.commons.lang.StringUtils.endsWith;
import static org.apache.commons.lang.StringUtils.isBlank;
import static org.apache.commons.lang.StringUtils.isNotBlank;
import static org.apache.commons.lang.StringUtils.join;
import static org.apache.commons.lang.StringUtils.substring;

@ParametersAreNonnullByDefault
final class FontStyleHandler {

    public static final String HTML_B = "b";
    public static final String HTML_STRONG = "strong";
    public static final String HTML_I = "i";
    public static final String HTML_EM = "em";
    public static final String HTML_U = "u";
    public static final String HTML_INS = "ins";
    public static final String HTML_STRIKE = "strike";
    public static final String HTML_DEL = "del";
    public static final String HTML_S = "s";
    public static final String HTML_Q = "q";
    public static final String HTML_CITE = "cite";

    private static final String NEWLINE = "\n";

    private static final Map<String, String> styleMap = ImmutableMap.<String, String>builder()
            .put(HTML_B, "*")
            .put(HTML_STRONG, "*")
            .put(HTML_I, "_")
            .put(HTML_EM, "_")
            .put(HTML_U, "+")
            .put(HTML_INS, "+")
            .put(HTML_STRIKE, "-")
            .put(HTML_DEL, "-")
            .put(HTML_S, "-")
            .put(HTML_Q, "\"")
            .put(HTML_CITE, "??")
            .build();

    private static final Comparator<Map.Entry<String, Integer>> COMPARATOR = new Comparator<Map.Entry<String, Integer>>() {
        @Override
        public int compare(Map.Entry<String, Integer> o1, Map.Entry<String, Integer> o2) {
            final int x = o1 == null ? 0 : o1.getValue();
            final int y = o2 == null ? 0 : o2.getValue();

            return (x < y) ? -1 : ((x == y) ? 0 : 1);
        }
    };

    private final Map<String, Integer> depthMap = newHashMap();
    private final Map<String, Boolean> reportedStartMap = newHashMap();
    private final BlockStyleHandler blockStyleHandler;

    public FontStyleHandler(final BlockStyleHandler blockStyleHandler) {
        this.blockStyleHandler = blockStyleHandler;
    }

    public String enter(final Node node, final String name, final int depth) {
        if (!this.blockStyleHandler.isFormattingPossible()) {
            return EMPTY;
        }

        final String wiki = styleMap.get(name);

        if (isBlank(wiki)) {
            return EMPTY;
        }

        final String val;
        if (depthMap.containsKey(wiki)) {
            val = EMPTY;
        } else {
            boolean hasText = false;
            if (node instanceof Element) {
                final Element element = (Element) node;
                hasText = element.hasText();
            }
            if (hasText) {
                val = wiki;
                depthMap.put(wiki, depth);
                reportedStartMap.put(wiki, false);
            } else {
                val = EMPTY;
            }
        }

        return val;
    }

    public String exit(final String name, final int depth) {
        if (!this.blockStyleHandler.isFormattingPossible()) {
            return EMPTY;
        }

        final String wiki = styleMap.get(name);

        if (isBlank(wiki)) {
            return EMPTY;
        }

        final Integer startDepth = depthMap.get(wiki);
        if (startDepth == null) {
            return EMPTY;
        }

        final String val;
        if (startDepth != depth) {
            val = EMPTY;
        } else {
            val = wiki;
            depthMap.remove(wiki);
            reportedStartMap.remove(wiki);
        }

        return val;
    }

    public boolean isAnyAtStart(final int depth) {
        Boolean val = false;
        for (Map.Entry<String, Integer> element : depthMap.entrySet()) {
            final String key = element.getKey();
            final Integer value = element.getValue();
            if (value != null && depth >= value) {
                final Boolean temp = reportedStartMap.get(key);
                if (temp != null && !temp) {
                    reportedStartMap.put(key, true);
                    val = true;
                }
            }
        }
        return val;
    }

    public void resetAllCurrentStartValues() {
        for (Map.Entry<String, Boolean> entry : reportedStartMap.entrySet()) {
            entry.setValue(false);
        }
    }

    public boolean isPrecededByStyle(final String html) {
        if (isBlank(html)) {
            return false;
        }

        for (String wiki : styleMap.values()) {
            if (endsWith(html, wiki)) {
                return true;
            }
        }

        return false;
    }

    public void wrapStylesAround(final StringBuilder accum, final String text) {
        if (depthMap.isEmpty() || !endsWith(text, NEWLINE)) {
            accum.append(text);
            return;
        }

        stopStylings(accum);
        accum.append(text);
        startStylings(accum);
    }

    public void stopStylings(final StringBuilder accum) {
        // this is handling the global accumulator, as unfortunately not possible to have all information in one place currently
        String currentStyles = getCurrentStyles(false);

        if (isNotBlank(currentStyles)) {
            String removedWhitespace = DocumentUtilities.removeTrailingWhitespace(accum);

            while (accum.length() > 0 && currentStyles.length() > 0 && accum.charAt(accum.length() - 1) == currentStyles.charAt(0)) {
                accum.deleteCharAt(accum.length() - 1);
                currentStyles = substring(currentStyles, 1);

                removedWhitespace = DocumentUtilities.removeTrailingWhitespace(accum) + removedWhitespace;
            }

            accum.append(currentStyles);
            accum.append(removedWhitespace);
        }
    }

    public void startStylings(final StringBuilder sb) {
        // this is handling the global accumulator, as unfortunately not possible to have all information in one place currently
        final String currentStyles = getCurrentStyles(true);

        if (isNotBlank(currentStyles)) {
            sb.append(" ");
            sb.append(currentStyles);
            resetAllCurrentStartValues();
        }
    }

    private String getCurrentStyles(boolean start) {
        final List<Map.Entry<String, Integer>> entries = newArrayList(depthMap.entrySet());

        Collections.sort(entries, COMPARATOR);

        if (!start) {
            Collections.reverse(entries);
        }

        return join(transform(entries, new Function<Map.Entry<String, Integer>, String>() {
            @Nullable
            @Override
            public String apply(@Nullable Map.Entry<String, Integer> input) {
                if (input != null) {
                    return input.getKey();
                }

                return EMPTY;
            }
        }), EMPTY);
    }

}
