package com.atlassian.mail.converters.wiki;

import com.google.common.base.Function;
import com.google.common.base.Optional;
import com.google.common.base.Predicate;
import com.google.common.collect.Iterables;
import org.jsoup.nodes.Node;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;
import java.util.Arrays;
import java.util.Deque;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import static com.google.common.collect.Lists.newLinkedList;
import static java.lang.String.format;
import static org.apache.commons.lang.StringUtils.EMPTY;
import static org.apache.commons.lang.StringUtils.defaultString;
import static org.apache.commons.lang.StringUtils.equalsIgnoreCase;
import static org.apache.commons.lang.StringUtils.isBlank;
import static org.apache.commons.lang.StringUtils.isNotBlank;
import static org.apache.commons.lang.StringUtils.isNotEmpty;
import static org.apache.commons.lang.StringUtils.remove;
import static org.apache.commons.lang.StringUtils.replaceOnce;
import static org.apache.commons.lang.StringUtils.split;
import static org.apache.commons.lang.StringUtils.substringAfter;
import static org.apache.commons.lang.StringUtils.substringBefore;
import static org.apache.commons.lang.StringUtils.trimToEmpty;
import static org.apache.commons.lang.math.NumberUtils.toInt;

@ParametersAreNonnullByDefault
final class ColorHandler {

    public static final String HTML_FONT = "font";

    private static final String WIKI_COLOR_END = "{color}";
    private static final String WIKI_COLOR_START_FORMAT = "{color:%s}";
    private static final String HTML_ATTRIBUTE_STYLE = "style";
    private static final String HTML_ATTRIBUTE_COLOR = "color";

    private static final Pattern RGB_PATTERN = Pattern.compile("^rgb\\((\\d+),(\\d+),(\\d+)\\)$");
    private static final Pattern COLOR_PATTERN = Pattern.compile("^\\s*(\\{color}|\\{color\\s*:\\s*\\p{Graph}+\\s*})(.*|\\n*)*");

    private static String WIKI_COLOR_START(String value) {
        value = defaultString(value);
        final String noWhitespace = remove(trimToEmpty(value), ' ');
        final Matcher matcher = RGB_PATTERN.matcher(noWhitespace);

        if (matcher.matches()) {
            final int red = toInt(matcher.group(1));
            final int green = toInt(matcher.group(2));
            final int blue = toInt(matcher.group(3));

            String hex = String.format("#%02x%02x%02x", red, green, blue);
            return format(WIKI_COLOR_START_FORMAT, hex);
        }

        return format(WIKI_COLOR_START_FORMAT, value);
    }

    private final Deque<ColorTagProperties> colorsInUse = newLinkedList();
    private final FontStyleHandler fontStyleHandler;
    private final BlockStyleHandler blockStyleHandler;

    public ColorHandler(FontStyleHandler fontStyleHandler, BlockStyleHandler blockStyleHandler) {
        this.fontStyleHandler = fontStyleHandler;
        this.blockStyleHandler = blockStyleHandler;
    }

    public String enter(final StringBuilder accum, final Node node, final String name, final int depth, final boolean isInsideAnyLink, final boolean isInsideLinkWithText, final boolean isUrlInLinkText, final boolean isInsideTable) {
        if (!this.blockStyleHandler.isFormattingPossible()) {
            return EMPTY;
        }

        final String color;
        if (HTML_FONT.equals(name)) {
            color = node.attr(HTML_ATTRIBUTE_COLOR);
        } else {
            final String style = node.attr(HTML_ATTRIBUTE_STYLE);
            if (isNotBlank(style)) {
                final List<String> split = Arrays.asList(split(style, ";"));

                final Optional<String> colorAttr = Iterables.tryFind(split, new Predicate<String>() {
                    @Override
                    public boolean apply(@Nullable String input) {
                        if (isBlank(input)) {
                            return false;
                        }

                        final String trimmed = trimToEmpty(substringBefore(trimToEmpty(input), ":"));
                        return equalsIgnoreCase(trimmed, "color");
                    }
                });

                color = colorAttr.transform(new Function<String, String>() {
                    @Nullable
                    @Override
                    public String apply(@Nullable String input) {
                        final String value = substringAfter(trimToEmpty(input), ":");
                        return trimToEmpty(value);
                    }
                }).or(EMPTY);
            } else {
                color = EMPTY;
            }
        }

        if (isBlank(color)) {
            // no color .... append nothing
            return EMPTY;
        }

        final StringBuilder sb = new StringBuilder();

        // have a color. need to check if there is already a color style in use, as will need to end that, to use this new color
        final ColorTagProperties first = colorsInUse.peekFirst();
        if (first != null) {
            // end the current color
            if (first.started) {
                fontStyleHandler.stopStylings(accum);
                sb.append(WIKI_COLOR_END);
            }
        }

        // start the new color
        final boolean started;
        if (!isInsideAnyLink || isInsideLinkWithText) {
            if (!isInsideAnyLink || !isUrlInLinkText) {
                sb.append(WIKI_COLOR_START(color));
                fontStyleHandler.startStylings(sb);
                started = true;
            } else {
                started = false;
            }
        } else {
            started = false;
        }

        // add this new color to head of queue
        colorsInUse.offerFirst(new ColorTagProperties(color, name, depth, started));

        return sb.toString();
    }

    public String exit(final StringBuilder accum, final String name, final int depth, final String precedingWhitespace, final boolean isInsideAnyLink, final boolean isUrlInLinkText) {
        if (!this.blockStyleHandler.isFormattingPossible()) {
            return EMPTY;
        }

        final ColorTagProperties first = colorsInUse.pollFirst();

        if (first == null) {
            return EMPTY;
        } else if (!first.tag.equals(name) || first.depth != depth) {
            // put it back at the start of the queue and leave as does not match expected
            colorsInUse.offerFirst(first);
            return EMPTY;
        }

        final StringBuilder sb = new StringBuilder();

        // end this color tag
        handlePrecedingWhiteSpace(fontStyleHandler, accum, sb, precedingWhitespace, first);

        // is there another? check but do not remove it
        final ColorTagProperties nextColor = colorsInUse.pollFirst();

        final boolean started;
        if (nextColor != null && (!isInsideAnyLink || !isUrlInLinkText)) {
            // this was already being used, but we ended, to use the color we just finished. Set this one back now
            sb.append(WIKI_COLOR_START(nextColor.color));
            fontStyleHandler.startStylings(sb);
            started = true;
        } else {
            started = false;
        }

        if (nextColor != null) {
            colorsInUse.offerFirst(new ColorTagProperties(nextColor.color, nextColor.tag, nextColor.depth, started));
        }

        return sb.toString();
    }

    public String handleAroundNonSupportedFormatting(final StringBuilder accum, String string, final String precedingWhitespace, final boolean isInsideAnyLink, final boolean isInsideLinkWithText, final boolean isUrlInLinkText, final boolean isStartOrEndOfTable) {
        string = defaultString(string);

        if (!this.blockStyleHandler.isFormattingPossible()) {
            return precedingWhitespace + string;
        }

        if (colorsInUse.isEmpty() || (isStartOrEndOfTable || (isInsideAnyLink && !isInsideLinkWithText))) {
            final StringBuilder builder = new StringBuilder();

            final ColorTagProperties colorTagProperties = colorsInUse.pollFirst();
            if (colorTagProperties != null) {
                if (colorTagProperties.started) {
                    fontStyleHandler.stopStylings(accum);
                    builder.append(WIKI_COLOR_END);
                }

                colorsInUse.offerFirst(new ColorTagProperties(colorTagProperties.color, colorTagProperties.tag, colorTagProperties.depth, false));
            }

            builder.append(precedingWhitespace);
            builder.append(string);
            return builder.toString();
        }

        final StringBuilder sb = new StringBuilder();

        final ColorTagProperties first = colorsInUse.pollFirst();

        // end this color tag
        handlePrecedingWhiteSpace(fontStyleHandler, accum, sb, precedingWhitespace, first);

        sb.append(string);

        final boolean started;
        if (!isInsideAnyLink || !isUrlInLinkText) {
            sb.append(WIKI_COLOR_START(first.color));
            fontStyleHandler.startStylings(sb);
            started = true;
        } else {
            started = false;
        }

        colorsInUse.offerFirst(new ColorTagProperties(first.color, first.tag, first.depth, started));

        return sb.toString();
    }

    @Nonnull
    public StripResult removeFormatting(String text) {
        Matcher matcher = COLOR_PATTERN.matcher(text);
        String result = text;
        String removed = EMPTY;

        while (matcher.matches()) {
            String group = matcher.group(1);
            removed += group;
            result = replaceOnce(result, group, EMPTY);
            matcher = COLOR_PATTERN.matcher(result);
        }

        return new StripResult(result, removed);
    }

    private static void handlePrecedingWhiteSpace(final FontStyleHandler fontStyleHandler, final StringBuilder accum, final StringBuilder sb, final String precedingWhitespace, @Nullable final ColorTagProperties colorTagProperties) {
        if (colorTagProperties == null || !colorTagProperties.started) {
            sb.append(precedingWhitespace);
            return;
        }

        fontStyleHandler.stopStylings(accum);

        // end this color tag
        if (isNotEmpty(precedingWhitespace)) {
            sb.append(WIKI_COLOR_END);
            sb.append(precedingWhitespace);
        } else {
            sb.append(precedingWhitespace);
            sb.append(WIKI_COLOR_END);
        }
    }

    public static class StripResult {
        private final String result;
        private final String removed;

        public StripResult(String result, String removed) {
            this.result = result;
            this.removed = removed;
        }

        public String getResult() {
            return result;
        }

        public String getRemoved() {
            return removed;
        }
    }

    private static class ColorTagProperties {
        final String color;
        final String tag;
        final int depth;
        final boolean started;

        public ColorTagProperties(String color, String tag, int depth, boolean started) {
            this.color = defaultString(color);
            this.tag = defaultString(tag);
            this.depth = depth;
            this.started = started;
        }
    }

}
