package com.atlassian.mail.converters.wiki;

import com.google.common.collect.HashMultiset;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.Multiset;
import org.jsoup.helper.StringUtil;
import org.jsoup.nodes.Element;
import org.jsoup.nodes.Node;

import javax.annotation.ParametersAreNonnullByDefault;
import java.util.Map;

import static com.google.common.collect.Maps.newHashMap;
import static org.apache.commons.lang.StringUtils.EMPTY;
import static org.apache.commons.lang.StringUtils.isBlank;

@ParametersAreNonnullByDefault
final class BlockStyleHandler {

    private static final String NEWLINE = "\n";

    public static final String HTML_BLOCKQUOTE = "blockquote";
    public static final String HTML_PRE = "pre";
    public static final String HTML_CODE = "code";

    private static final Map<String, String> styleMap = ImmutableMap.<String, String>builder()
            .put(HTML_BLOCKQUOTE, "{quote}")
            .put(HTML_PRE, "{noformat}")
            .put(HTML_CODE, "{code}")
            .build();

    private final Map<String, Integer> depthMap = newHashMap();
    private final Multiset<String> depthCounter = HashMultiset.create();

    public String enter(final Node node, final String name, final int depth) {
        String val = styleMap.get(name);

        if (isBlank(val)) {
            return EMPTY;
        }

        final int currentDepth = getCurrentDepth(val);

        if (HTML_PRE.equals(name)) {
            if (node instanceof Element) {
                final Element element = (Element) node;
                final Element firstChild = element.children().first();
                if (firstChild != null && HTML_CODE.equals(firstChild.nodeName())) {
                    // this should be displayed as a code block if have <pre><code>....</code></pre>
                    return EMPTY;
                }
            }
        }

        if (StringUtil.in(name, HTML_PRE, HTML_CODE, HTML_BLOCKQUOTE)) {
            if (currentDepth >= 0) {
                if (currentDepth == depth) {
                    // add a second value to the multimap to help us with when the exit actually happened
                    depthCounter.add(val);
                }
                val = NEWLINE;
            } else {
                depthMap.put(val, depth);
                depthCounter.add(val);
                val = NEWLINE + val;
            }
        }

        return val;
    }

    public String exit(final String name, final int depth) {
        String val = styleMap.get(name);

        if (isBlank(val)) {
            return EMPTY;
        }

        final int currentDepth = getCurrentDepth(val);

        if (StringUtil.in(name, HTML_PRE, HTML_CODE, HTML_BLOCKQUOTE)) {
            if (currentDepth == depth) {
                depthCounter.remove(val);
                if (depthCounter.count(val) == 0) {
                    depthMap.remove(val);
                    val += NEWLINE;
                } else {
                    val = NEWLINE;
                }
            } else {
                val = NEWLINE;
            }
        }

        return val;
    }

    public boolean isFormattingPossible() {
        final int currentPreDepth = getCurrentDepth(styleMap.get(HTML_PRE));
        final int currentCodeDepth = getCurrentDepth(styleMap.get(HTML_CODE));

        return currentPreDepth == -1 && currentCodeDepth == -1;
    }

    private int getCurrentDepth(final String wikiVal) {
        if (isBlank(wikiVal)) {
            return -1;
        }

        return depthMap.containsKey(wikiVal) ? depthMap.get(wikiVal) : -1;
    }

}
