package com.atlassian.mail.converters.wiki;

import com.atlassian.mail.MailUtils.Attachment;
import com.atlassian.mail.converters.HtmlConverter;
import com.google.common.collect.ImmutableList;
import org.jsoup.nodes.Element;
import org.jsoup.nodes.Node;
import org.jsoup.nodes.TextNode;
import org.jsoup.select.NodeTraversor;
import org.jsoup.select.NodeVisitor;

import javax.annotation.Nonnull;
import java.io.IOException;
import java.util.List;
import java.util.regex.Pattern;

import static com.atlassian.mail.converters.wiki.BlockStyleHandler.HTML_BLOCKQUOTE;
import static com.atlassian.mail.converters.wiki.BlockStyleHandler.HTML_CODE;
import static com.atlassian.mail.converters.wiki.BlockStyleHandler.HTML_PRE;
import static com.atlassian.mail.converters.wiki.FontStyleHandler.HTML_B;
import static com.atlassian.mail.converters.wiki.FontStyleHandler.HTML_CITE;
import static com.atlassian.mail.converters.wiki.FontStyleHandler.HTML_DEL;
import static com.atlassian.mail.converters.wiki.FontStyleHandler.HTML_EM;
import static com.atlassian.mail.converters.wiki.FontStyleHandler.HTML_I;
import static com.atlassian.mail.converters.wiki.FontStyleHandler.HTML_INS;
import static com.atlassian.mail.converters.wiki.FontStyleHandler.HTML_Q;
import static com.atlassian.mail.converters.wiki.FontStyleHandler.HTML_S;
import static com.atlassian.mail.converters.wiki.FontStyleHandler.HTML_STRIKE;
import static com.atlassian.mail.converters.wiki.FontStyleHandler.HTML_STRONG;
import static com.atlassian.mail.converters.wiki.FontStyleHandler.HTML_U;
import static com.atlassian.mail.converters.wiki.LinkAndImageHandler.HTML_IMG;
import static com.atlassian.mail.converters.wiki.LinkAndImageHandler.HTML_LINK;
import static com.atlassian.mail.converters.wiki.ListHandler.HTML_DD;
import static com.atlassian.mail.converters.wiki.ListHandler.HTML_DL;
import static com.atlassian.mail.converters.wiki.ListHandler.HTML_DT;
import static com.atlassian.mail.converters.wiki.ListHandler.HTML_LI;
import static com.atlassian.mail.converters.wiki.ListHandler.HTML_OL;
import static com.atlassian.mail.converters.wiki.ListHandler.HTML_UL;
import static com.atlassian.mail.converters.wiki.TableHandler.HTML_TABLE;
import static com.atlassian.mail.converters.wiki.TableHandler.HTML_TD;
import static com.atlassian.mail.converters.wiki.TableHandler.HTML_TH;
import static com.atlassian.mail.converters.wiki.TableHandler.HTML_TR;
import static org.apache.commons.lang.StringUtils.EMPTY;
import static org.apache.commons.lang.StringUtils.contains;
import static org.apache.commons.lang.StringUtils.containsAny;
import static org.apache.commons.lang.StringUtils.isNotBlank;
import static org.apache.commons.lang.StringUtils.isNotEmpty;
import static org.apache.commons.lang.StringUtils.replace;
import static org.apache.commons.lang.StringUtils.startsWith;
import static org.apache.commons.lang.StringUtils.strip;
import static org.apache.commons.lang.StringUtils.stripStart;
import static org.apache.commons.lang.StringUtils.substring;
import static org.jsoup.helper.StringUtil.in;
import static org.jsoup.helper.StringUtil.isWhitespace;

/**
 * Helper class to convert basic HTML to Wiki macro.
 * >
 * For example <code>img</code> to <code>!image.png|thumbnail!</code>.
 *
 * @link https://extranet.atlassian.com/display/VPORT/CQ+-+Update+Html+To+Wiki+Text+converter+in+Atlassian+Mail for some more details
 * @since v2.6.0 was moved from {@link com.atlassian.mail.HtmlToWikiTextConverter}
 */
public class HtmlToWikiTextConverter implements HtmlConverter {

    private static final char WIKI_LINK_START_CHAR = '[';
    private static final char WIKI_TABLE_CHAR = '|';
    private static final char NEWLINE_CHAR = '\n';

    private static final String NEWLINE = Character.toString(NEWLINE_CHAR);
    private static final String WIKI_TABLE = Character.toString(WIKI_TABLE_CHAR);
    private static final String NON_WIKI_TEXT_REPLACE = ":";
    private static final String WIKI_LINK_START = Character.toString(WIKI_LINK_START_CHAR);
    private static final String WIKI_LINK_END = "]";

    private final List<Attachment> attachments;

    public HtmlToWikiTextConverter(@Nonnull final List<Attachment> attachments) {
        this.attachments = ImmutableList.copyOf(attachments);
    }

    private enum WhitespaceNewlineHandling {
        NONE,
        ONE,
        UNCHANGED
    }

    @Override
    public String convert(@Nonnull final String html) throws IOException {
        final DocumentUtilities.DocumentElement document = DocumentUtilities.parseHtml(html);

        final DocumentUtilities.BodyElement body = DocumentUtilities.getBody(document);

        FormattingVisitor formatter = new FormattingVisitor();
        NodeTraversor traversor = new NodeTraversor(formatter);
        traversor.traverse(body.getBody()); // walk the DOM, and call .head() and .tail() for each node

        return formatter.toString();
    }

    // the formatting rules, implemented in a breadth-first DOM traverse
    private class FormattingVisitor implements NodeVisitor {
        private final StringBuilder accum = new StringBuilder(); // holds the accumulated text
        private final BlockStyleHandler blockStyleHandler = new BlockStyleHandler();
        private final FontStyleHandler fontStyleHandler = new FontStyleHandler(blockStyleHandler);
        private final ListHandler listHandler = new ListHandler(blockStyleHandler);
        private final TableHandler tableHandler = new TableHandler(blockStyleHandler);
        private final LinkAndImageHandler linkAndImageHandler = new LinkAndImageHandler(blockStyleHandler, attachments);

        // hit when the node is first seen
        public void head(Node node, int depth) {
            final String name = node.nodeName();

            // TextNodes carry all user-readable text in the DOM.
            if (node instanceof TextNode) {
                String text = ((TextNode) node).text();
                if (fontStyleHandler.isAnyAtStart(depth)) {
                    text = stripStart(text, null);
                }
                if (tableHandler.isInTable()) {
                    // if we are inside of a table, then to prevent weirdness in the wiki output, replace | with someething else (in our case :)
                    text = replace(text, WIKI_TABLE, NON_WIKI_TEXT_REPLACE);
                }
                if (linkAndImageHandler.isInsideLinkWithText()) {
                    text = replace(text, WIKI_LINK_START, NON_WIKI_TEXT_REPLACE);
                    text = replace(text, WIKI_LINK_END, NON_WIKI_TEXT_REPLACE);
                }

                // now append
                if (tableHandler.isInTable()) {
                    // if we are inside of a table, then to prevent weirdness in the wiki output, replace | with someething else (in our case :)
                    append(node, text, WhitespaceNewlineHandling.ONE);
                } else if (linkAndImageHandler.isInsideLinkWithText()) {
                    append(node, text, WhitespaceNewlineHandling.NONE);
                } else {
                    append(node, text);
                }
            } else if (in(name, HTML_B, HTML_STRONG, HTML_I, HTML_EM, HTML_U, HTML_INS, HTML_STRIKE, HTML_DEL, HTML_S, HTML_Q, HTML_CITE)) {
                final String enterText = fontStyleHandler.enter(node, name, depth);
                if (isNotEmpty(enterText) && node instanceof Element) {
                    final Element element = (Element) node;
                    if (element.hasText()) {
                        String removedWhitespace = EMPTY;

                        final Node firstChildNode = element.childNodeSize() > 0 ? element.childNode(0) : null;

                        String text;
                        if (firstChildNode instanceof TextNode) {
                            text = ((TextNode) firstChildNode).getWholeText();
                        } else {
                            text = element.text();
                        }

                        while (text.length() > 0 && isWhitespace(text.charAt(0))) {
                            removedWhitespace += text.charAt(0);
                            text = substring(text, 1);
                        }

                        if (!fontStyleHandler.isPrecededByStyle(this.accum.toString())) {
                            append(node, removedWhitespace);
                        }
                    }
                }
                append(node, enterText);
            } else if (in(name, HTML_BLOCKQUOTE, HTML_PRE, HTML_CODE)) {
                append(node, blockStyleHandler.enter(node, name, depth));
            } else if (in(name, "h1", "h2", "h3", "h4", "h5", "h6")) {
                append(node, NEWLINE + name + ". ");
            } else if (in(name, HTML_OL, HTML_UL, HTML_LI, HTML_DL, HTML_DD, HTML_DT)) {
                append(node, listHandler.enter(name), WhitespaceNewlineHandling.ONE);
            } else if (in(name, HTML_TABLE, HTML_TR, HTML_TH, HTML_TD)) {
                WhitespaceNewlineHandling whitespaceNewlineHandling = WhitespaceNewlineHandling.NONE;
                if (in(name, HTML_TABLE, HTML_TR) || tableHandler.isFirstTableRowData()) {
                    whitespaceNewlineHandling = WhitespaceNewlineHandling.ONE;
                }
                append(node, tableHandler.enter(name), whitespaceNewlineHandling);
            } else if (in(name, HTML_LINK, HTML_IMG)) {
                append(node, linkAndImageHandler.enter(node, name));
            } else if (in(name, "p", "div")) {
                if (name.equals("div")) {
                    if (node.childNodeSize() > 0) {
                        final Node child = node.childNode(0);
                        final boolean treatNewline;
                        if (child instanceof TextNode) {
                            final String text = ((TextNode) child).text();
                            treatNewline = isNotBlank(text);
                        } else if (child instanceof Element) {
                            treatNewline = !in(((Element) child).tagName(), "br", "p");
                        } else {
                            treatNewline = true;
                        }

                        if (treatNewline && !fontStyleHandler.isInsideStyling()) {
                            append(node, NEWLINE);
                        }
                    }
                } else {
                    if (!fontStyleHandler.isInsideStyling()) {
                        append(node, NEWLINE + NEWLINE);
                    }
                }
            }
        }

        // hit when all of the node's children (if any) have been visited
        public void tail(Node node, int depth) {
            String name = node.nodeName();
            if (name.equals("br")) {
                if (!fontStyleHandler.isInsideStyling()) {
                    append(node, NEWLINE);
                }
            } else if (name.equals("hr")) {
                append(node, NEWLINE + "----" + NEWLINE, WhitespaceNewlineHandling.ONE);
            } else if (in(name, HTML_B, HTML_STRONG, HTML_I, HTML_EM, HTML_U, HTML_INS, HTML_STRIKE, HTML_DEL, HTML_S, HTML_Q, HTML_CITE)) {
                final String fontExit = fontStyleHandler.exit(name, depth);
                String removedWhitespace = EMPTY;
                if (isNotBlank(fontExit) && accum.length() > 0) {
                    // need to wrap tightly
                    while (isWhitespace(accum.charAt(accum.length() - 1))) {
                        removedWhitespace += accum.substring(accum.length() - 1);
                        accum.deleteCharAt(accum.length() - 1);
                    }
                }
                append(node, fontExit);

                if (!fontStyleHandler.isInsideStyling()) {
                    if (node instanceof Element) {
                        final Element element = (Element) node;
                        final Node sibling = element.nextSibling();
                        if (sibling != null && sibling instanceof TextNode) {
                            final String text = ((TextNode) sibling).text();
                            if (isNotEmpty(text) && !isWhitespace(text.charAt(0)) && !Pattern.matches("\\p{Punct}", substring(text, 0, 1))) {
                                append(node, " ");
                            }
                        }
                    }
                }

                append(node, removedWhitespace);
            } else if (in(name, HTML_BLOCKQUOTE, HTML_PRE, HTML_CODE)) {
                append(node, blockStyleHandler.exit(name, depth));
            } else if (in(name, "h1", "h2", "h3", "h4", "h5", "h6")) {
                append(node, NEWLINE);
            } else if (in(name, HTML_OL, HTML_UL, HTML_LI, HTML_DL, HTML_DD, HTML_DT)) {
                append(node, listHandler.exit(name), WhitespaceNewlineHandling.ONE);
            } else if (in(name, HTML_TABLE, HTML_TR, HTML_TH, HTML_TD)) {
                append(node, tableHandler.exit(name), WhitespaceNewlineHandling.NONE);
                if (in(name, HTML_TABLE, HTML_TR)) {
                    append(node, NEWLINE, WhitespaceNewlineHandling.ONE);
                }
            } else if (HTML_LINK.equals(name)) {
                append(node, linkAndImageHandler.exit(node, name));
            }
        }

        private void append(Node node, String text) {
            append(node, text, WhitespaceNewlineHandling.UNCHANGED);
        }

        private void append(Node node, String text, WhitespaceNewlineHandling whitespaceNewlineHandling) {

            text = replace(text, "\r" + NEWLINE, NEWLINE);
            text = replace(text, "\f", NEWLINE);
            text = replace(text, "\r", NEWLINE);
            text = replace(text, "\t", " ");

            if (accum.length() == 0) {
                text = stripStart(text, null);
            } else if (whitespaceNewlineHandling == WhitespaceNewlineHandling.NONE || whitespaceNewlineHandling == WhitespaceNewlineHandling.ONE) {
                String removed = EMPTY;
                while (isWhitespace(accum.charAt(accum.length() - 1))) {
                    removed += accum.charAt(accum.length() - 1);
                    accum.deleteCharAt(accum.length() - 1);
                }
                while (isNotEmpty(text) && isWhitespace(text.charAt(0))) {
                    removed += text.charAt(0);
                    text = substring(text, 1);
                }

                if (containsAny(removed, " ")) {
                    // if it is text inside of a link, make sure that not preceded by the [ wiki, as can not append a space after that
                    if (linkAndImageHandler.isInsideLinkWithText()) {
                        if (accum.charAt(accum.length() - 1) != WIKI_LINK_START_CHAR) {
                            accum.append(' ');
                        }
                    } else {
                        accum.append(' ');
                    }
                }

                if (whitespaceNewlineHandling == WhitespaceNewlineHandling.ONE) {
                    if (contains(removed, NEWLINE_CHAR) && !tableHandler.isStartOfTableData(node)) {
                        // just in case. when inside a table, it needs some special newline handling, as a newline in
                        // wrong place breaks the whole table :(
                        if (!tableHandler.isInTable() || tableHandler.isEndOfRow() || tableHandler.isFirstTableRowData()) {
                            accum.append(NEWLINE);
                        }
                    }
                }
            }

            if (startsWith(text, NEWLINE)) {
                // more special table treatment
                final char last = this.accum.charAt(this.accum.length() - 1);
                if (tableHandler.isStartOfTableData(node) && (isWhitespace(last) || last == WIKI_TABLE_CHAR)) {
                    text = strip(text, NEWLINE);
                }
            }

            accum.append(text);
        }

        @Override
        public String toString() {
            return accum.toString();
        }
    }

}
