package com.atlassian.mail.converters.wiki;

import com.google.common.collect.ImmutableMap;
import org.jsoup.nodes.Element;
import org.jsoup.nodes.Node;

import javax.annotation.ParametersAreNonnullByDefault;
import java.util.Map;

import static com.google.common.collect.Maps.newHashMap;
import static org.apache.commons.lang.StringUtils.EMPTY;
import static org.apache.commons.lang.StringUtils.endsWith;
import static org.apache.commons.lang.StringUtils.isBlank;

@ParametersAreNonnullByDefault
final class FontStyleHandler {

    public static final String HTML_B = "b";
    public static final String HTML_STRONG = "strong";
    public static final String HTML_I = "i";
    public static final String HTML_EM = "em";
    public static final String HTML_U = "u";
    public static final String HTML_INS = "ins";
    public static final String HTML_STRIKE = "strike";
    public static final String HTML_DEL = "del";
    public static final String HTML_S = "s";
    public static final String HTML_Q = "q";
    public static final String HTML_CITE = "cite";

    private static final Map<String, String> styleMap = ImmutableMap.<String, String>builder()
            .put(HTML_B, "*")
            .put(HTML_STRONG, "*")
            .put(HTML_I, "_")
            .put(HTML_EM, "_")
            .put(HTML_U, "+")
            .put(HTML_INS, "+")
            .put(HTML_STRIKE, "-")
            .put(HTML_DEL, "-")
            .put(HTML_S, "-")
            .put(HTML_Q, "\"")
            .put(HTML_CITE, "??")
            .build();

    private final Map<String, Integer> depthMap = newHashMap();
    private final Map<String, Boolean> reportedStartMap = newHashMap();
    private final BlockStyleHandler blockStyleHandler;

    public FontStyleHandler(final BlockStyleHandler blockStyleHandler) {
        this.blockStyleHandler = blockStyleHandler;
    }

    public String enter(final Node node, final String name, final int depth) {
        if (!this.blockStyleHandler.isFormattingPossible()) {
            return EMPTY;
        }

        final String wiki = styleMap.get(name);

        if (isBlank(wiki)) {
            return EMPTY;
        }

        final String val;
        if (depthMap.containsKey(wiki)) {
            val = EMPTY;
        } else {
            boolean hasText = false;
            if (node instanceof Element) {
                final Element element = (Element) node;
                hasText = element.hasText();
            }
            if (hasText) {
                val = wiki;
                depthMap.put(wiki, depth);
                reportedStartMap.put(wiki, false);
            } else {
                val = EMPTY;
            }
        }

        return val;
    }

    public String exit(final String name, final int depth) {
        if (!this.blockStyleHandler.isFormattingPossible()) {
            return EMPTY;
        }

        final String wiki = styleMap.get(name);

        if (isBlank(wiki)) {
            return EMPTY;
        }

        final Integer startDepth = depthMap.get(wiki);
        if (startDepth == null) {
            return EMPTY;
        }

        final String val;
        if (startDepth != depth) {
            val = EMPTY;
        } else {
            val = wiki;
            depthMap.remove(wiki);
            reportedStartMap.remove(wiki);
        }

        return val;
    }

    public boolean isAnyAtStart(final int depth) {
        Boolean val = false;
        for (Map.Entry<String, Integer> element : depthMap.entrySet()) {
            final String key = element.getKey();
            final Integer value = element.getValue();
            if (value != null && depth >= value) {
                final Boolean temp = reportedStartMap.get(key);
                if (temp != null && !temp) {
                    reportedStartMap.put(key, true);
                    val = true;
                }
            }
        }
        return val;
    }

    public boolean isPrecededByStyle(final String html) {
        if (isBlank(html)) {
            return false;
        }

        for (String wiki : styleMap.values()) {
            if (endsWith(html, wiki)) {
                return true;
            }
        }

        return false;
    }

    public boolean isInsideStyling() {
        return !depthMap.isEmpty();
    }

}
