package com.atlassian.mail;

import com.atlassian.mail.MailUtils.Attachment;
import org.apache.commons.lang.StringUtils;
import org.apache.log4j.Category;
import org.jsoup.Jsoup;
import org.jsoup.nodes.Document;
import org.jsoup.nodes.Element;
import org.jsoup.nodes.TextNode;
import org.jsoup.select.Elements;

import java.io.IOException;
import java.util.List;

/**
 * Helper class to convert HTML <code>img</code> to Wiki macro, for example <code>!image.png|thumbnail!</code>.
 */
public class HtmlToWikiTextConverter implements HtmlConverter
{
    private static final String HTML_IMG = "img";
    private static final String HTML_ATL = "alt";
    private static final String HTML_SRC = "src";
    private static final String WIKI_MACRO_ATTACHMENT = "!%s|thumbnail!";
    private static final String CONTENT_ID_PREFIX = "cid:";
    private static final Category log = Category.getInstance(HtmlToTextConverter.class);

    private final HtmlToTextConverter htmlToPlainTextConverter = new HtmlToTextConverter();
    private final List<Attachment> attachments;

    public HtmlToWikiTextConverter(final List<MailUtils.Attachment> attachments)
    {
        super();
        this.attachments = attachments;
    }

    public String convert(final String html) throws IOException
    {
        Document document = Jsoup.parse(html);
        Elements imageElements = document.body().getElementsByTag(HTML_IMG);
        for (Element imageElement : imageElements)
        {
            imageElement.replaceWith(wikiMacroForImage(imageElement));
        }
        return htmlToPlainTextConverter.convert(document.outerHtml());
    }

    /**
     *  @return Wiki macro for image, if the image has a src attribute and its value corresponds to one of the attachments.
     *  Otherwise returns the alternative text attribute of the image.
     */
    private TextNode wikiMacroForImage(Element imageElement)
    {
        String imageSource = imageElement.attr(HTML_SRC);
        String contentId = StringUtils.removeStart(imageSource, CONTENT_ID_PREFIX);
        Attachment image = getAttachmentById(contentId);
        if (image != null)
        {
            return textNode(String.format(WIKI_MACRO_ATTACHMENT, image.getFilename()));
        }
        String alternativeText = imageElement.attr(HTML_ATL);
        log.warn("Could not find attachment: " + alternativeText + " for content id: " + contentId);
        return textNode(alternativeText);
    }

    private TextNode textNode(String text)
    {
        return new TextNode(text, StringUtils.EMPTY);
    }

    private Attachment getAttachmentById(String contentId)
    {
        for (Attachment attachment : attachments)
        {
            if (StringUtils.equalsIgnoreCase(attachment.getContentId(), contentId))
            {
                return attachment;
            }
        }
        return null;
    }

}
